%% PREPROCESSING
%----------------

clear; %clear all data from memory
accuracy = 2; %number of decimal digits

% INPUT
%-------
inputfile  = '.\testinput.txt';  %file to read
outputfile = '.\testoutput.txt'; %file to write
delim = '\t';  %delimiter in inputfile (' '  = space, 
               %                        '\t' = tab, 
               %                        ';'  = semicolon,
               %                        ','  = comma, ...)
headerlines = 0; %number of headerlines to skip from inputfile

% VARIABLES
%-----------
% Col Var          Desc
% --- ---          ----
% 1.  Ta         = luchttemperatuur (C)
% 2.  R          = inkomende kortgolvige straling (W/m)
% 3.  hr         = relatieve vochtigheid (%)
% 4.  u          = windsnelheid (m/s)
% 5.  w          = breedte blad (m)
% 6.  gvsad      = stomatale dampgeleiding aan de adaxiale bladzijde (mol/ms)
% 7.  gvsab      = stomatale dampgeleiding aan de abaxiale bladzijde (mol/ms)
% 8.  cloudcover = fractie van de hemel bedekt met wolken
% 9.  St         = Type of greenhouse material
%                  1: no greenhouse
%                  2: glass
%                  3: PVC coated polyester 0.6 mm
%                  4: LDPE 0.15 mm

data = importdata(inputfile, delim, headerlines);

% CONSTANTS
%-----------
g = 9.81;            %gravitatie constante (m/s)
sigma = 5.67e-8;     %Stefan-Boltzman constante (W/mK^4)
cp = 29.3;           %specifieke warmte van lucht (J/molC)
Esoil = 0.945;       %langgolvige emissiviteit van de bodem
Eleaf = 0.97;        %langgolvige emissiviteit van een blad
Al = Eleaf;          %langgolvige absorptiviteit van een blad
As = 0.5;            %kortgolvige absorptiviteit van een blad
Rsoil = 0.1;         %kortgolvige reflectiviteit van de bodem
Pa = 102000;         %luchtdruk (Pa)
Ts = 15;             %bodemtemperatuur (C)

% GREENHOUSE MATERIAL CONSTANTS
%---------------------------
% Type of greenhouse material
% 1: no greenhouse
% 2: glass
% 3: PVC coated polyester 0.6 mm
% 4: LDPE 0.15 mm
% 5: Glass, no shortwave effect

% Skeleton
Ss = 0.10; % % of radiation stopped by greenhouse skeleton
% Shortwave
Rts = [ 1+Ss 0.8  0.9  0.89 1+Ss ] - Ss; % Transmissivity
% Longwave
Rtl = [ 1+Ss 0.0  0.11 0.7  0.0  ] - Ss; % Transmissivity
Rel = [ 0    0.84 0.15 0.2  0.84 ]     ; % Emissivity
Rrl = [ 0    0.16 0.74 0.1  0.16 ]     ; % Reflectivity


%% PROCESSING
%-------------

results = zeros(size(data,1),6); %preallocate results matrix
results(:,:) = NaN; %fill matrix with NaNs

for k = 1:size(data,1) %step by step through data series
    
    % ASSIGN DATA SERIE TO VARIABLES
    %--------------------------------
    foo = num2cell([data(k,1) data(k,2) data(k,3) data(k,4) data(k,5) data(k,6) data(k,7) data(k,8) data(k,9)]);
    [Ta R hr u w gvsad gvsab cloudcover St] = foo{:};
    
    Tai = Ta; %Inner air temperature = Air temperature
    Tao = Ta; %Outer air temperature = Air temperature
    
    % DECLARE LOCAL VARIABLES
    %-------------------------
    Tl2 = Tai;
    z = 10;
    i = 0;
    budget2 = 10;
    budget1 = budget2;
    
    
    % CALCULATION_TL_INDEPENDENT_VARIABLES
    %--------------------------------------
    d = 0.72 * w;                     %karakteristieke dimensie (m)
    lambda = -42.575 * Tai + 44994;     %latente warmte van verdamping (J/mol)
    esTa = esT(Tai);                    %gesatureerde dampdruk bij luchttemperatuur (Pa)
    ea = hr * esTa;                   %omgevings dampdruk (Pa)
    xi = 46.5 * ( (ea/100) / ( Tao + 273.15 ) );
    Eclearsky = 1 - (1 + xi ) * exp( -( 1.2 + 3 * xi )^( 1/2 ) ); %emissiviteit van open hemel (Prata)
    % Eclearsky = 1.72 * ( (ea/1000) / (Tao+273.15) )^(1/7) %emissiviteit van open hemel (Campbell)
    Esky = (1 - 0.84 * cloudcover) * Eclearsky + 0.84 * cloudcover; %emissiviteit van de lucht
    
    nu = 0.0000000908 * Tai + 0.000013267; %kinematic viscosity (m/s)
    
    Re = u * d / nu;                      %getal van Reynolds
    
    Rsin = R * Rts(St) * As * 0.5 + R * Rts(St) * As * Rsoil * 0.5;  %kortgolvige inkomende straling (W/m)
    % END_CALCULATION_TL_INDEPENDENT_VARIABLES
    
    while (z >= 10^-(accuracy+1))              %10-tallen, eenheden, tienden, ...
        while (z >= 10^-(accuracy+1))   %omliggende getallen bepalen
            Tl1 = Tl2;
            budget1 = budget2;
            if (i > 0)
                Tl2 = Tl1 + z*sign(budget1) ;
            end
            
            % CALCULATION_TL_DEPENDENT_VARIABLES
            %------------------------------------
            Gr = (g * d^3 * (Tl1 - Tai) / ( ( (Tl1 + Tai) / 2 ) + 273.15 ) * nu^2); %getal van Grashof
            
            if (Re > 5E5)
                %Turbulent: 1.4*Forced
                gHaad = 1.4*0.135*sqrt(u/d);  %warmtegeleiding (mol/ms)
                gvaad = 1.4*0.147*sqrt(u/d);  %grenslaag dampgeleiding (mol/ms)
                gHaab = gHaad;
                gvaab = gvaad;
            else
                if (Gr/Re^2 > 1)
                    %Free convection
                    if (Tl1 >= Tai)
                        gHaad = 0.050*((Tl1-Tai)/d)^(1/4);  %warmtegeleiding (mol/ms)
                        gvaad = 0.055*((Tl1-Tai)/d)^(1/4);  %grenslaag dampgeleiding (mol/ms)
                        gHaab = 0.5 * gHaad;
                        gvaab = 0.5 * gvaad;
                    else
                        gHaab = 0.050*((Tl1-Tai)/d)^(1/4);  %warmtegeleiding (mol/ms)
                        gvaab = 0.055*((Tl1-Tai)/d)^(1/4);  %grenslaag dampgeleiding (mol/ms)
                        gHaad = 0.5 * gHaad;
                        gvaad = 0.5 * gvaad;
                    end
                else
                    %Forced convection
                    gHaad = 0.135*sqrt(u/d);  %warmtegeleiding (mol/ms)
                    gvaad = 0.147*sqrt(u/d);  %grenslaag dampgeleiding (mol/ms)
                    gHaab = gHaad;
                    gvaab = gvaad;
                end
            end
            
            %gv = ifelse((gvsad==0) & (gvsab==0) , 0 , ifelse (gvsad==0 , 0.5/(1/gvsab+1/gvaab) , ifelse (gvsab==0 , 0.5/(1/gvsad+1/gvaad) , 0.5/(1/gvsad+1/gvaad)+0.5/(1/gvsab+1/gvaab) ) ) ); %totale dampgeleiding (mol/ms)
            if (gvsad==0) && (gvsab==0)
                gv = 0;
            elseif gvsad==0
                gv = 0.5/(1/gvsab+1/gvaab);
            elseif gvsab==0
                gv = 0.5/(1/gvsad+1/gvaad);
            else
                gv = 0.5/(1/gvsad+1/gvaad)+0.5/(1/gvsab+1/gvaab);
            end
            
            Rlout = Eleaf * sigma * (Tl1+273.15)^4;  %langgolvige uitgaande straling (W/m)
            Rlin = Al * (Esoil   * sigma * (Ts  + 273.15)^4) * 0.5           + ... %langgolvige inkomende straling (bodem) (W/m)
                   Al * (Esky    * sigma * (Tao + 273.15)^4) * 0.5 * Rtl(St) + ... %langgolvige inkomende straling (lucht) (W/m)
                   Al * (Rel(St) * sigma * (Tao + 273.15)^4) * 0.5           + ... %langgolvige inkomende straling (serre) (W/m)
                   Al * Rlout                                * 0.5 * Rrl(St);      %langgolvige inkomende straling (serre reflectie) (W/m)
    
            H = (gHaad/2 + gHaab/2) * cp * (Tl1 - Tai); %convectie (W/m)
            E = gv * (esT(Tl1) - ea) / Pa;  %verdamping (mol/ms)
            budget2 = Rsin + Rlin - Rlout - H - lambda * E; %energie budget berekenen (W/m)
            % END_CALCULATION_TL_DEPENDENT_VARIABLES
            
            % CHECK ENERGY BUDGET CHANGE
            %----------------------------
            if (i == 0)
                budget1 = budget2;
            end
            i = i + 1;
            if ( (sign(budget1) ~= sign(budget2)) && (i > 1) )
                break;
            end
        end
        z = z/10;
    end
    
    results(k,:) = [mean([Tl1,Tl2]) Rsin Rlin Rlout H (lambda * E)];
end

frmt = ['Tl = %4.' num2str(accuracy) 'f\n'];
fprintf(frmt, results(:,1))

% EXPORT TO FILE
%----------------
% Col Desc
% --- ----
% 1.  Leaftemperature
% 2.  Shortwave incident radiation
% 3.  Longwave incident radiation
% 4.  Longwave outward radiation
% 5.  Sensible heatflux
% 6.  Latent heatflux

export(dataset(results),'file',outputfile,'delimiter',delim,'WriteVarNames',false);