MODULE trcsms_mang
   !!======================================================================
   !!                         ***  MODULE trcmang  ***
   !! MANG :   Compute sources minus sinks of manganese
   !!======================================================================
   !! History     0.0  ! 2015-04  (M. van Hulten)  implementation of
   !!                    dissolved Mn(II) and sinking manganese oxide
   !!             0.1  ! 2015-08  (M. van Hulten)  multiple improvements
   !!             0.2  ! 2016-10  (M. van Hulten)  latest NEMO 3.6_STABLE
   !!----------------------------------------------------------------------
#if defined key_top && key_mang
   !!----------------------------------------------------------------------
   !!   'key_top'       and                                       TOP model
   !!   'key_pisces'    and            (FIXME)                 PISCES model
   !!   'key_mang'                                          Manganese model
   !!----------------------------------------------------------------------
   !!   trc_sms_mang    : Compute SMS of manganese
   !!   mang_init       : Initialisation of parameters for remineralisation
   !!----------------------------------------------------------------------
   USE oce_trc       ! Shared variables between ocean and passive tracers
   USE par_trc       ! TOP parameters
   USE trc           ! Passive tracers common variables 
   USE iom           ! I/O manager
   USE p4zsbc        ! Input fields
   USE sms_pisces,   ONLY : neln, etot, heup
   USE trcsink       ! For calling trc_sink2()

   IMPLICIT NONE
   PRIVATE

   INTEGER, PUBLIC   :: numnammang = -1   ! logical unit for the MANG namelist
   REAL(wp), PUBLIC  :: kox               ! oxidation rate
   REAL(wp), PUBLIC  :: kredPZ, kredAZ    ! reduction rates (Photic and Aphotic)
   REAL(wp), PUBLIC  :: niter3max         ! upper value for iteration oMn sinking
   REAL(wp), PUBLIC  :: rfact             ! timestep (Euler or leap-frog)
   REAL(wp), PUBLIC  :: zwsmax
   LOGICAL, PUBLIC   :: ln_mndepo         ! boolean for Mn input from dust
   LOGICAL, PUBLIC   :: ln_rivermn        ! boolean for Mn input from rivers
   LOGICAL, PUBLIC   :: ln_mangsed        ! boolean for Mn input from sediments
   LOGICAL, PUBLIC   :: ln_hydromn        ! boolean for Mn input from hydrothermal vents
   REAL(wp), PUBLIC  :: dustsolubMn       ! solubility of Mn in the dust
   REAL(wp), PUBLIC  :: riverMnFe         ! relative Mn from rivers compared to Fe
   REAL(wp), PUBLIC  :: sedimMnFe         ! relative Mn from sediments compared to Fe
   REAL(wp), PUBLIC  :: hydroMnHe         ! relative Mn from hydrothermal vents compared to Fe

   REAL(wp), PUBLIC, ALLOCATABLE, SAVE, DIMENSION(:,:,:) :: hydromn   !: Hydrothermal vent supply of manganese
   REAL(wp), PUBLIC, ALLOCATABLE, SAVE, DIMENSION(:,:,:) :: ws_ox     !: Mn oxide sinking speed
   REAL(wp), PUBLIC, ALLOCATABLE, SAVE, DIMENSION(:,:,:) :: sinkmang  !: Manganese sinking fluxes

   PUBLIC   trc_sms_mang       ! called in trcsms.F90
   PUBLIC   mang_init          ! called in this file

   !!* Substitution
#  include "top_substitute.h90"
   !!----------------------------------------------------------------------
   !! NEMO/TOP 3.6 , NEMO Consortium (2015)
   !! Software governed by the CeCILL licence  (NEMOGCM/License_CeCILL.txt)
   !!----------------------------------------------------------------------
CONTAINS

   SUBROUTINE trc_sms_mang( kt )
      !!---------------------------------------------------------------------
      !!                     ***  ROUTINE trc_sms_mang  ***
      !!
      !! ** Purpose :   Compute sources and scavenging terms of manganese
      !!
      !! ** Method  :   
      !! 
      !!---------------------------------------------------------------------
      !
      INTEGER, INTENT(in) ::   kt   ! ocean time step
      !
      INTEGER  ::   ji, jj, jk, jit, ikt, jn
      INTEGER  ::   iiter3                ! Not used right now (set to 1)
      REAL(wp) ::   zfact
      REAL(wp) ::   red_Mn, oxi_Mn        ! Deltas for redox
      !!---------------------------------------------------------------------
      !
      IF( nn_timing == 1 )  CALL timing_start('trc_sms_mang')

      IF( kt == nittrc000 )   CALL mang_init

      ! Choose timestepping (euler or leapfrog)
      ! ---------------------------------------
      ! 
      IF( ( neuler == 0 .AND. kt == nittrc000 ) .OR. ln_top_euler ) THEN
         rfact = rdttrc(1)
      ELSEIF( kt <= nittrc000 + nn_dttrc ) THEN
         rfact = 2. * rdttrc(1)
      ENDIF

      IF( ( neuler == 0 .AND. kt == nittrc000 ) .OR. ln_top_euler ) THEN
         DO jn = jp_mang0, jp_mang1
            trb(:,:,:,jn) = trn(:,:,:,jn)
         END DO
      ENDIF

      ! Sinking of manganese oxides
      ! -----------------------------
      ! 
      ws_ox(:,:,:) = 1.0
      DO jk = 1, jpkm1
         DO jj = 1, jpj
            DO ji = 1,jpi
              !zmax  = MAX( heup(ji,jj), hmld(ji,jj) )
               zfact = MAX( 0., fsdepw(ji,jj,jk+1) - hmld(ji,jj) ) / 2500._wp
              !zfact = MIN( 1., MAX( 0., fsdepw(ji,jj,jk+1) - hmld(ji,jj) ) / 2500._wp )
               IF ( trb(ji,jj,jk,jpomn) > 0.025e-9 ) THEN
                  ws_ox(ji,jj,jk) = 1. + 9. * zfact ! 1, up to 10 m/d if high particle concentration
               ENDIF
            END DO
         END DO
      END DO

      iiter3 = 1  ! iteration time splitting not used; see iter/ directory if needed!
      sinkmang(:,:,:) = 0.e0
      CALL trc_sink2( ws_ox , sinkmang, jpomn, iiter3, rfact )

      ! Burial in sediment
      ! ------------------
      ! 
      DO jj = 1, jpj
         DO ji = 1, jpi
            ikt = mbkt(ji,jj)                      ! Bottom k-index of T-gridbox
            zfact = 1. / rday / fse3t(ji,jj,ikt)   ! Reciprocal bottom gridbox thickness
            tra(ji,jj,ikt,jpomn) = tra(ji,jj,ikt,jpomn) - trb(ji,jj,ikt,jpomn)  &
               &                                        * ws_ox(ji,jj,ikt) * zfact
         END DO
      END DO

      ! External sources of manganese
      ! -----------------------------
      ! 
      IF( ln_mndepo  ) THEN
         tra(:,:,1,jpdmn) = tra(:,:,1,jpdmn) + dustsolubMn * ( dust(:,:) * mfrac * (0.0880/3.5) &
                                             / 54.938 ) / fse3t(:,:,1) !+ 3.e-10 * r1_ryyss
      ENDIF
      IF( ln_rivermn ) THEN
         tra(:,:,1,jpdmn) = tra(:,:,1,jpdmn) + riverMnFe * rivdic(:,:) * 3.e-5 !5.e-5
                                                               ! increased from 3.e-5 to 5.e-5
                                                               ! following jpfer in p4zsed.F90
      ENDIF
      IF( ln_mangsed ) THEN
         tra(:,:,:,jpdmn) = tra(:,:,:,jpdmn) + sedimMnFe * ironsed(:,:,:)
      ENDIF
      IF( ln_hydromn ) THEN
         tra(:,:,:,jpdmn) = tra(:,:,:,jpdmn) + hydromn(:,:,:)  ! calculated in mang_init()
      ENDIF

      ! Redox of manganese
      ! ------------------
      ! 
      DO jj = 1, jpj
         DO ji = 1, jpi
            DO jk = 1, neln(ji,jj)
               red_Mn = kredPZ*trb(ji,jj,jk,jpomn)
               oxi_Mn = kox   *trb(ji,jj,jk,jpdmn)
               tra(ji,jj,jk,jpdmn) = tra(ji,jj,jk,jpdmn) + red_Mn - oxi_Mn
               tra(ji,jj,jk,jpomn) = tra(ji,jj,jk,jpomn) - red_Mn + oxi_Mn
            END DO
            DO jk = neln(ji,jj)+1, jpkm1
               red_Mn = kredAZ*trb(ji,jj,jk,jpomn)
               oxi_Mn = kox   *trb(ji,jj,jk,jpdmn)
               tra(ji,jj,jk,jpdmn) = tra(ji,jj,jk,jpdmn) + red_Mn - oxi_Mn
               tra(ji,jj,jk,jpomn) = tra(ji,jj,jk,jpomn) - red_Mn + oxi_Mn
            END DO
         END DO
      END DO

      ! Add trends to new concentration
      ! -------------------------------
      ! 
      DO jn = jp_mang0, jp_mang1
         trb(:,:,:,jn) = trb(:,:,:,jn) + tra(:,:,:,jn) * rfact
         tra(:,:,:,jn) = 0._wp
      END DO
      !
      IF( ln_top_euler ) THEN
         DO jn = jp_mang0, jp_mang1
            trn(:,:,:,jn) = trb(:,:,:,jn)
         END DO
      ENDIF

   END SUBROUTINE trc_sms_mang

   SUBROUTINE mang_init
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE mang_init  ***
      !!
      !! ** Purpose :   Sets constants for the manganese model
      !!
      !! ** Method  :   Read the namelist and print the parameters
      !!                This routine is called at the first timestep
      !!
      !! ** Input   :   Namelist sections nammang_decay, nammang_src and nammang_scav
      !!
      !!----------------------------------------------------------------------
      NAMELIST/nammang_src/ ln_mndepo,   ln_rivermn, ln_mangsed, ln_hydromn,  &
         &                  dustsolubMn, riverMnFe,  sedimMnFe,  hydroMnHe
      NAMELIST/nammang_scav/ kox, kredPZ, kredAZ, niter3max
      INTEGER           :: ios                  ! Local output status for namelist read
      INTEGER           :: mang_aloc            ! Local output status for allocation of arrays
      INTEGER           :: ryyss                ! Number of seconds per year
      INTEGER           :: numhydro

      ryyss = nyear_len(1) * rday

      CALL ctl_opn( numnammang, 'namelist_mang', 'UNKNOWN', 'FORMATTED', 'SEQUENTIAL', -1, &
      &             numout, .FALSE. )
      REWIND( numnammang )
      READ  ( numnammang, nammang_src,  IOSTAT = ios, ERR = 921 )
      READ  ( numnammang, nammang_scav,  IOSTAT = ios, ERR = 922 )
921   IF ( ios /= 0 ) CALL ctl_nam( ios , 'nammang_src   in namelist_mang', lwp )
922   IF ( ios /= 0 ) CALL ctl_nam( ios , 'nammang_scav  in namelist_mang', lwp )

      IF(lwp) THEN                         ! control print
         WRITE(numout,*) ' '
         WRITE(numout,*) ' Namelist parameters for scavenging, nammang_src'
         WRITE(numout,*) ' ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
         WRITE(numout,*) '    dust manganese solubility                dustsolubMn=', dustsolubMn
         WRITE(numout,*) '    river input dissolved Mn/Fe fraction     riverMnFe  =', riverMnFe
         WRITE(numout,*) '    sediment input Mn/Fe fraction            sedimMnFe  =', sedimMnFe
         WRITE(numout,*) '    hydrothermal input Mn/He fraction        hydroMnHe  =', hydroMnHe
         WRITE(numout,*) ' '
         WRITE(numout,*) ' Namelist parameters for scavenging, nammang_scav'
         WRITE(numout,*) ' ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
         WRITE(numout,*) '    oxidation rate                    (/s)   kox        =', kox
         WRITE(numout,*) '    reduction rate (photic zone)      (/s)   kredPZ     =', kredPZ
         WRITE(numout,*) '    reduction rate (aphotic zone)     (/s)   kredAZ     =', kredAZ
        !WRITE(numout,*) '    iteration for timesplit oMn sink  (-)    niter3max  =', niter3max
      ENDIF

      IF( ln_hydromn ) THEN
         ALLOCATE( hydromn(jpi,jpj,jpk), STAT=mang_aloc )
         IF( mang_aloc /= 0 ) CALL ctl_warn('mang_init : failed to allocate array hydromn().')
         !
         CALL iom_open ( 'hydrofe.orca.nc', numhydro )   ! hydrofe.orca.nc == epsd.orca.nc
         CALL iom_get  ( numhydro, jpdom_data, 'epsdb', hydromn(:,:,:), 1 )
         CALL iom_close( numhydro )
         !
         ! This is analogous to hydrofe code (that is in p4zsbc.F90)
         hydromn(:,:,:) = ( hydromn(:,:,:) * hydroMnHe ) / ( cvol(:,:,:) * ryyss + rtrn ) / 1000._wp
      ENDIF

      ALLOCATE( ws_ox(jpi,jpj,jpk), sinkmang(jpi,jpj,jpk), STAT=mang_aloc )
      IF( mang_aloc /= 0 ) CALL ctl_warn('mang_init : failed to allocate arrays.')

   END SUBROUTINE mang_init

#else
   !!======================================================================
   !!  Dummy module :                                   No manganese
   !!======================================================================
CONTAINS
   SUBROUTINE trc_sms_mang( kt )                    ! Empty routine
      INTEGER, INTENT(in) ::   kt

      WRITE(*,*) 'trc_sms_mang: This routine should not have been called! Time step: ', kt
   END SUBROUTINE trc_sms_mang
#endif 

   !!======================================================================
END MODULE trcsms_mang
