MODULE trcsms_protac
   !!======================================================================
   !!                         ***  MODULE trcprotac  ***
   !! PROTAC :   Compute sources minus sinks of protactinium and thorium
   !!======================================================================
   !! History     0.0  ! 2015-04  (M. van Hulten)  improved implementation of
   !!                    Dutay et al. (2009): dissolved and two adorbed phases
   !!----------------------------------------------------------------------
#if defined key_top && key_protac
   !! At the moment: protac works only with key_pisces (particle fields)
   !! TODO: Make it work without key_pisces (off-line)
   !! TODO: Logging of messages / saving of extra output fields...
   !!----------------------------------------------------------------------
   !!   'key_top'       and                                       TOP model
   !!   'key_pisces'    and            (FIXME)                 PISCES model
   !!   'key_protac'                                            Pa/Th model
   !!----------------------------------------------------------------------
   !!   trc_sms_protac  : Compute SMS of Th and Pa
   !!   protac_init     : Initialisation of parameters for remineralisation
   !!----------------------------------------------------------------------
   USE oce_trc       ! Shared variables between ocean and passive tracers
   USE par_trc       ! TOP parameters
   USE trc           ! Passive tracers common variables 
   USE iom           ! I/O manager
   USE p4zsink, ONLY : wsbio3, wsbio4

   IMPLICIT NONE
   PRIVATE

   INTEGER, PUBLIC   :: numnamprotac = -1    ! logical unit for the PROTAC namelist
   REAL(wp), PUBLIC  :: beta_pa231, beta_th230, lambda_pa231, lambda_th230
   REAL(wp), PUBLIC  :: k_pa_poc, k_th_poc, k_pa_goc, k_th_goc, k_pa_opal, &
   &                          k_th_opal, k_pa_caco3, k_th_caco3

   PUBLIC   trc_sms_protac       ! called in trcsms.F90
   PUBLIC   protac_init          ! called in this file

   !!* Substitution
#  include "top_substitute.h90"
   !!----------------------------------------------------------------------
   !! NEMO/TOP 3.6 , NEMO Consortium (2015)
   !! Software governed by the CeCILL licence  (NEMOGCM/License_CeCILL.txt)
   !!----------------------------------------------------------------------
CONTAINS

   SUBROUTINE trc_sms_protac( kt )
      !!---------------------------------------------------------------------
      !!                     ***  ROUTINE trc_sms_protac  ***
      !!
      !! ** Purpose :   Compute production, decay and scavenging terms
      !!                of Pa and Th isotopes.
      !!
      !! ** Method  :   Direct partioning between dissolved and particulate
      !!                fractions
      !! 
      !! ** TODO    :   Seperate routine for scavenging of all trace elements
      !!                Also, delayed adsorption (with rate constants k1, k2)
      !!                Adsorbed tracers also decay (but decay is small)
      !!---------------------------------------------------------------------
      !
      INTEGER, INTENT(in) ::   kt   ! ocean time step
      !
      INTEGER  ::   ji, jj, jk, ikt, jn
      REAL(wp) ::   zfact, zmax
      REAL(wp) ::   prod230th, decay230th, scav230th,   &   ! Th-230 SMS
      &             prod231pa, decay231pa, scav231pa        ! Pa-231 SMS
      !!---------------------------------------------------------------------
      !
      IF( nn_timing == 1 )  CALL timing_start('trc_sms_protac')

      IF( kt == nittrc000 )   CALL protac_init

      ! Th-230 and Pa-231 scavenging (Logboek p. 78)
      ! --------------------------------------------
      ! 
      trn(:,:,:,jpth230d) = ( trn(:,:,:,jpth230d) + trn(:,:,:,jpth230s) + trn(:,:,:,jpth230f) ) &
      &                   / ( 1. + k_th_poc *trn(:,:,:,jppoc) + k_th_goc  *trn(:,:,:,jpgoc)     &
      &                          + k_th_opal*trn(:,:,:,jpgsi) + k_th_caco3*trn(:,:,:,jpcal) )
      trn(:,:,:,jpth230s) =   k_th_poc  *trn(:,:,:,jppoc) * trn(:,:,:,jpth230d)
      trn(:,:,:,jpth230f) = ( k_th_goc  *trn(:,:,:,jpgoc) + k_th_opal*trn(:,:,:,jpgsi)          &
      &                     + k_th_caco3*trn(:,:,:,jpcal) ) * trn(:,:,:,jpth230d)
      !
      trn(:,:,:,jppa231d) = ( trn(:,:,:,jppa231d) + trn(:,:,:,jppa231s) + trn(:,:,:,jppa231f) ) &
      &                   / ( 1. + k_pa_poc *trn(:,:,:,jppoc) + k_pa_goc  *trn(:,:,:,jpgoc)     &
      &                          + k_pa_opal*trn(:,:,:,jpgsi) + k_pa_caco3*trn(:,:,:,jpcal) )
      trn(:,:,:,jppa231s) =   k_pa_poc  *trn(:,:,:,jppoc) * trn(:,:,:,jppa231d)
      trn(:,:,:,jppa231f) = ( k_pa_goc  *trn(:,:,:,jpgoc) + k_pa_opal*trn(:,:,:,jpgsi)          &
      &                     + k_pa_caco3*trn(:,:,:,jpcal) ) * trn(:,:,:,jppa231d)

      ! The trend, tra() in Bq/m3/s, is set to zero every timestep in trc_stp()

      DO jk = 1, jpkm1
         DO jj = 1, jpj
            DO ji = 1, jpi
               ! Th-230 and Pa-231 production
               ! ----------------------------
               prod231pa = beta_pa231
               prod230th = beta_th230
               !
               ! Th-230 and Pa-231 decay
               ! ----------------------------
               decay231pa = lambda_pa231 * trn(ji,jj,jk,jppa231d)
               decay230th = lambda_th230 * trn(ji,jj,jk,jpth230d)
               !
               ! Update of the tracers trends
               ! ----------------------------
               tra(ji,jj,jk,jppa231d) = tra(ji,jj,jk,jppa231d) + prod231pa - decay231pa
               tra(ji,jj,jk,jpth230d) = tra(ji,jj,jk,jpth230d) + prod230th - decay230th
            END DO
         END DO
      END DO

      ! Burial in sediment
      ! ------------------
      ! 
      DO jj = 1, jpj
         DO ji = 1, jpi
            ikt = mbkt(ji,jj)                      ! Bottom k-index of T-gridbox
            zfact = 1. / rday / fse3t(ji,jj,ikt)   ! Reciprocal bottom gridbox thickness
            tra(ji,jj,ikt,jpth230s) = tra(ji,jj,ikt,jpth230s) - trn(ji,jj,ikt,jpth230s)  &
            &                                                 * wsbio3(ji,jj,ikt) * zfact
            tra(ji,jj,ikt,jpth230f) = tra(ji,jj,ikt,jpth230f) - trn(ji,jj,ikt,jpth230f)  &
            &                                                 * wsbio4(ji,jj,ikt) * zfact
            tra(ji,jj,ikt,jppa231s) = tra(ji,jj,ikt,jppa231s) - trn(ji,jj,ikt,jppa231s)  &
            &                                                 * wsbio3(ji,jj,ikt) * zfact
            tra(ji,jj,ikt,jppa231f) = tra(ji,jj,ikt,jppa231f) - trn(ji,jj,ikt,jppa231f)  &
            &                                                 * wsbio4(ji,jj,ikt) * zfact
         END DO
      END DO

      ! Add trends to new concentration
      ! -------------------------------
      trn(:,:,:,jpth230d) = trn(:,:,:,jpth230d) + tra(:,:,:,jpth230d) * rdt
      trn(:,:,:,jppa231d) = trn(:,:,:,jppa231d) + tra(:,:,:,jppa231d) * rdt
      trn(:,:,:,jpth230s) = trn(:,:,:,jpth230s) + tra(:,:,:,jpth230s) * rdt
      trn(:,:,:,jppa231s) = trn(:,:,:,jppa231s) + tra(:,:,:,jppa231s) * rdt
      trn(:,:,:,jpth230f) = trn(:,:,:,jpth230f) + tra(:,:,:,jpth230f) * rdt
      trn(:,:,:,jppa231f) = trn(:,:,:,jppa231f) + tra(:,:,:,jppa231f) * rdt
      DO jn = jp_protac0, jp_protac1
         tra(:,:,:,jn) = 0._wp         ! Don't add these trends again in later routine
         trb(:,:,:,jn) = trn(:,:,:,jn) ! Update `before' timestep for NEMO (leapfrog)
      END DO

   END SUBROUTINE trc_sms_protac

   SUBROUTINE protac_init
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE protac_init  ***
      !!
      !! ** Purpose :   Sets constants for the thorium and protactinium model
      !!
      !! ** Method  :   Read the namelist and print the parameters
      !!                This routine is called at the first timestep
      !!
      !! ** Input   :   Namelist sections namprotac_decay and namprotac_scav
      !!
      !!----------------------------------------------------------------------
      NAMELIST/namprotac_decay/ beta_pa231, beta_th230, lambda_pa231, lambda_th230
      NAMELIST/namprotac_scav/ k_pa_poc, k_th_poc, k_pa_goc, k_th_goc, k_pa_opal, &
      &                        k_th_opal, k_pa_caco3, k_th_caco3
      INTEGER           :: ios                  ! Local output status for namelist read
      INTEGER           :: ryyss                ! Number of seconds per year

      ryyss = nyear_len(1) * rday

      CALL ctl_opn( numnamprotac, 'namelist_protac', 'UNKNOWN', 'FORMATTED', 'SEQUENTIAL', -1, &
      &             numout, .FALSE. )
      REWIND( numnamprotac )
      READ  ( numnamprotac, namprotac_decay, IOSTAT = ios, ERR = 911 )
      READ  ( numnamprotac, namprotac_scav,  IOSTAT = ios, ERR = 912 )
911   IF ( ios /= 0 ) CALL ctl_nam( ios , 'namprotac_decay in namelist_protac', lwp )
912   IF ( ios /= 0 ) CALL ctl_nam( ios , 'namprotac_scav  in namelist_protac', lwp )

      ! Convert radioactivities to SI units
      ! ------------------------------------
      beta_pa231  = beta_pa231 / 60. / ryyss
      beta_th230  = beta_th230 / 60. / ryyss
      lambda_pa231= lambda_pa231 / ryyss
      lambda_th230= lambda_th230 / ryyss

      ! Convert K values from gSW/g to L/mol
      ! ------------------------------------
      k_pa_poc    = (k_pa_poc  / 1035.) * 12.
      k_th_poc    = (k_th_poc  / 1035.) * 12.
      k_pa_goc    = (k_pa_goc  / 1035.) * 12.
      k_th_goc    = (k_th_goc  / 1035.) * 12.
      k_pa_opal   = (k_pa_opal / 1035.) * 67.3
      k_th_opal   = (k_th_opal / 1035.) * 67.3
      k_pa_caco3  = (k_pa_caco3/ 1035.) * 100.08
      k_th_caco3  = (k_th_caco3/ 1035.) * 100.08
      ! L/mol        gSW/g       gSW/L    g/mol 

      IF(lwp) THEN                         ! control print
         WRITE(numout,*) ' '
         WRITE(numout,*) ' Namelist parameters for decay, namprotac_decay'
         WRITE(numout,*) ' ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
         WRITE(numout,*) '    production rate of Pa-231 (Bq/m3/s)       beta_pa231 =', beta_pa231
         WRITE(numout,*) '    production rate of Th-230 (Bq/m3/s)       beta_th230 =', beta_th230
         WRITE(numout,*) '    decay rate of Pa-231           (/s)     lambda_pa231 =', lambda_pa231
         WRITE(numout,*) '    decay rate of Th-230           (/s)     lambda_th230 =', lambda_th230
         !
         WRITE(numout,*) ' Namelist parameters for scavenging, namprotac_scav'
         WRITE(numout,*) ' ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~'
         WRITE(numout,*) '    partition coefficient Pa on POC   (L/mol)   k_pa_poc =', k_pa_poc
         WRITE(numout,*) '    partition coefficient Th on POC   (L/mol)   k_th_poc =', k_th_poc
         WRITE(numout,*) '    partition coefficient Pa on GOC   (L/mol)   k_pa_goc =', k_pa_goc
         WRITE(numout,*) '    partition coefficient Th on GOC   (L/mol)   k_th_goc =', k_th_goc
         WRITE(numout,*) '    partition coefficient Pa on opal  (L/mol)  k_pa_opal =', k_pa_opal
         WRITE(numout,*) '    partition coefficient Th on opal  (L/mol)  k_th_opal =', k_th_opal
         WRITE(numout,*) '    partition coefficient Pa on CaCO3 (L/mol) k_pa_caco3 =', k_pa_caco3
         WRITE(numout,*) '    partition coefficient Th on CaCO3 (L/mol) k_th_caco3 =', k_th_caco3
      ENDIF
   END SUBROUTINE protac_init

#else
   !!======================================================================
   !!  Dummy module :                                   No Pa/Th
   !!======================================================================
CONTAINS
   SUBROUTINE trc_sms_protac( kt )                    ! Empty routine
      INTEGER, INTENT(in) ::   kt

      WRITE(*,*) 'trc_sms_protac: This routine should not have been called! Time step: ', kt
   END SUBROUTINE trc_sms_protac
#endif 

   !!======================================================================
END MODULE trcsms_protac
