!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This is part of the program for demostrating the bubble formation and growth  !     
! ebullition module (EBG) used in:                                              !
!                                                                               !
! Peltola, O., Raivonen, M., Li, X., and Vesala, T.: Technical Note: Comparison !
! of methane ebullition modelling approaches used in terrestrial wetland models,!
! Biogeosiences Discuss., XXX, 2017.                                            !
!                                                                               !
! Please see the pdf-document distributed along with these codes                !
! and the article Peltola et al., Biogeosciences Discussions xxx, 2017, for     !
! more information.                                                             ! 
!                                                                               !
! If you use these codes in your research, please cite the above-mentioned      !
! article.                                                                      !
!                                                                               !
! Contact:                                                                      !
! Olli Peltola                                                                  !
! olli.peltola@helsinki.fi                                                      !
!                                                                               !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


module bubble_geometry

  use bubble_parameters, only: dp, min_dz

  implicit none

  private
  public geometry, new_geom

  type geometry
    integer :: s1, s2, a1, a2, w1, w2
    logical :: surfaceQ, airQ, waterQ
    real(dp), allocatable :: levelsP(:), dzP(:)

    ! Terminology: Levels are z-coordinates, and layers are between levels. The
    ! array fixed_levelsP (in the main code) defines the default levels we want
    ! to use for the numerics. Levels can be of different thickness. Then we
    ! have one extra level to track the location of the WTD. In this way,
    ! whatever the WTD, we can always arrange a level at the WTD, so all the
    ! layers (boxes) will always be totally air (above WTD) or water (below
    ! WTD), which simplifies the main transport code.
    !
    ! levelsP: z-coordinates of the leves, after the extra, moving, level for
    ! the WTD has been added.
    !
    ! dzP: thicknesses of the layers between the levels.
    !
    ! surfaceQ : if surface water layer is present?
    ! airQ     : if air-peat layer is present?
    ! waterQ   : if water-peat layer is present?
    !
    ! s1: Index of the first box in levelsP, where surface water is. If this is
    !     0, then there is no surface water and s2 has no meaning. So this is
    !     either 0 or 1.
    ! s2: Last box of surface water.
    ! a1: First box of peat-air. If 0, there is no peat-air, a2 has no meaning.
    ! a2: Last box of peat-air.
    ! w1: First box of peat-water. If 0, no peat-water, w2 has no meaning.
    ! w2: Last box of peat-water.
    !
    ! So, compared to fixed_levelsP, levelsP here usually has one extra layer.
    ! This extra layer is for following the WTD. If WTD is negative, so above
    ! peat surface, then this extra layer is the water above peat surface. (If
    ! WTD were to have large negative values, for numerical reasons it might be
    ! needed to add several surface water layers, but currently this is not
    ! implemented, because then we would need to add boxes to all the depth
    ! profile arrays.)
    !
    ! But, if WTD is too close to one of the fixed levels in fixed_levelsP
    ! then, for reasons related to numerical solution of the diffusion
    ! equation, we would have one very small dz, leading to the need for a very
    ! small dt. For this reason, if WTD is closer than min_dz (currently 0.01 m
    ! i.e. 1 cm) to a fixed level, we "force" it to exactly to the level
    ! instead. This then causes the extra layer to be of 0 thickness.
    !
    ! So, if s1 /= 0, then a1 = 0 and w1 = 2 and w2 = nz.
    !
    ! If WTD is nearer than min_dz to the peat surface (assumed to be at
    ! fixed_levelsP(1)), then s1 = 0, a1 = 0, w1 = 2.
    !
    ! If WTD is nearer than min_dz to some other level (except the last), then
    ! s1 = 0, a1 = 1, w1 = a2+2, and the dz=0 layer is the box a2+1.
    !
    ! If WTD is larger than fixed_levelsP(nz) (the last level), then s1 = 0,
    ! a1 = 1, a2 = nz-1, w1 = 0, and the dz=0 layer is the box nz.
    !
    ! If WTD is nicely between two levels, then s1 = 0, a1 = 1, w1 = a2+1.
  end type geometry

contains


  !pure
  subroutine new_geom(geom, wtd_eff, nz, fixed_levelsP, wtd_in)
    ! Construct new geometry (levels and coordinates), given new wtd ('wtd_in')
    ! and the constant 'fixed_levelsP'. If 'wtd_in' is closer than 1 cm (i.e.
    ! 'min_dz') to a level in 'fixed_levelsP', then we just force it to be
    ! exactly that closest level in 'fixed_levelsP'. This way there will never
    ! be a layer thinner than 1 cm.
    !
    ! Return both the geometry 'geom' and the 'wtf_eff' that may be different
    ! than what 'wtd_in' was. I don't think this "cheating" by at most 1 cm
    ! matters for the actual results.
    !
    ! (Maybe a more "stylish" solution would be to round everyhing closer than
    ! 0.5 cm to the level, and from 0.5...1 cm to 1 cm away. But here I go for
    ! the simpler code logic and don't add that extra step, and extra lines of
    ! code. It can be added later if somebody feels the need.)

    type(geometry), intent(inout) :: geom
    ! I think this needs to be (inout), because 'geom' contains allocated
    ! arrays. If this was (out), I think I would need to allocate them here.
    ! From pure program logic point of view, (out) would suffice.
    real(dp), intent(out)         :: wtd_eff
    integer, intent(in)           :: nz
    ! geom%levelsP has nz+1 levels, nz layers
    ! fixed_levelsP has nz levels, nz-1 layers
    real(dp), intent(in)          :: fixed_levelsP(nz)
    real(dp), intent(in)          :: wtd_in
    integer                       :: i_wtd
    real(dp)                      :: z1, z2

    z1 = fixed_levelsP(1) - min_dz
    z2 = fixed_levelsP(1) + min_dz

    ! I believe all cases are handled, and handled properly. But this
    ! subroutine could benefit from a bit clearer structure. If I just would be
    ! able to think of something better.

    if (wtd_in <= z1) then
      ! WTD is above surface, the extra box is this box, being water. Also,
      ! usually fixed_levelsP(1) is 0, but let's try to be general.
      geom % s1 = 1; geom % a1 = 0; geom % w1 = 2
      geom % s2 = 1; geom % a2 = 0; geom % w2 = nz
      geom % surfaceQ = .true.; geom % airQ = .false.; geom % waterQ = .true.
      geom % levelsP(1) = wtd_in
      geom % levelsP(2:nz + 1) = fixed_levelsP(1:nz)
      wtd_eff = wtd_in ! no need to change
    else if ((z1 < wtd_in) .and. (wtd_in < z2)) then
      ! WTD is at peat surface level
      geom % s1 = 0; geom % a1 = 0; geom % w1 = 2
      geom % s2 = 0; geom % a2 = 0; geom % w2 = nz
      geom % surfaceQ = .false.; geom % airQ = .false.; geom % waterQ = .true.
      geom % levelsP(1) = fixed_levelsP(1)
      geom % levelsP(2:nz + 1) = fixed_levelsP(1:nz)
      wtd_eff = fixed_levelsP(1)
    else
      ! WTD is below peat surface
      i_wtd = find_wtd_index(fixed_levelsP, wtd_in)
      z2 = fixed_levelsP(nz) - min_dz
      if (i_wtd == 0) then
      write(*, *) 'new_geom: something is very wrong.'
    else if (z2 < wtd_in) then
      ! WTD is near, at, or below lowest level
      geom % s1 = 0; geom % a1 = 1; geom % w1 = 0
      geom % s2 = 0; geom % a2 = nz - 1; geom % w2 = 0
      geom % surfaceQ = .false.; geom % airQ = .true.; geom % waterQ = .false.
      geom % levelsP(1:nz) = fixed_levelsP(1:nz)
      geom % levelsP(nz + 1) = fixed_levelsP(nz)
      wtd_eff = fixed_levelsP(nz)
    else
      z1 = fixed_levelsP(i_wtd) + min_dz
      z2 = fixed_levelsP(i_wtd + 1) - min_dz
      if (wtd_in < z1) then
      ! WTD is very close to fixed_levelsP(i_wtd)
      geom % s1 = 0; geom % a1 = 1; geom % w1 = i_wtd + 1
      geom % s2 = 0; geom % a2 = i_wtd - 1; geom % w2 = nz
      geom % surfaceQ = .false.; geom % airQ = .true.; geom % waterQ = .true.
      geom % levelsP(1:i_wtd) = fixed_levelsP(1:i_wtd)
      geom % levelsP(i_wtd + 1) = fixed_levelsP(i_wtd)
      geom % levelsP(i_wtd + 2:nz + 1) = fixed_levelsP(i_wtd + 1:nz)
      wtd_eff = fixed_levelsP(i_wtd)
    else if (z2 < wtd_in) then
      ! WTD is very close to fixed_levelsP(i_wtd+1)
      geom % s1 = 0; geom % a1 = 1; geom % w1 = i_wtd + 2
      geom % s2 = 0; geom % a2 = i_wtd; geom % w2 = nz
      geom % surfaceQ = .false.; geom % airQ = .true.; geom % waterQ = .true.
      geom % levelsP(1:i_wtd + 1) = fixed_levelsP(1:i_wtd + 1)
      geom % levelsP(i_wtd + 2) = fixed_levelsP(i_wtd + 1)
      geom % levelsP(i_wtd + 3:nz + 1) = fixed_levelsP(i_wtd + 2:nz)
      wtd_eff = fixed_levelsP(i_wtd + 1)
    else
      ! WTD is nicely between fixed_levelsP(i_wtd) and fixed_levelsP(i_wtd+1)
      geom % s1 = 0; geom % a1 = 1; geom % w1 = i_wtd + 1
      geom % s2 = 0; geom % a2 = i_wtd; geom % w2 = nz
      geom % surfaceQ = .false.; geom % airQ = .true.; geom % waterQ = .true.
      geom % levelsP(1:i_wtd) = fixed_levelsP(1:i_wtd)
      geom % levelsP(i_wtd + 1) = wtd_in
      geom % levelsP(i_wtd + 2:nz + 1) = fixed_levelsP(i_wtd + 1:nz)
      wtd_eff = wtd_in ! no need to change
      end if
      end if
    end if

    geom % dzP = geom % levelsP(2:nz + 1) - geom % levelsP(1:nz)
  end subroutine new_geom

  pure function find_wtd_index(levelsP, wtd) result(index)
    ! Search for the index of the layer in which wtd is in. The bottom level
    ! (boundary) is part of the layer, the top level is not. If wtd is smaller
    ! than, or equal to, levelsP(1), return 0. If wtd is larger than
    ! levelsP(nz+1), return nz+1. Will not return the index of dz=0 layer, but
    ! the one physically above.
    real(dp), intent(in)  :: levelsP(:), wtd
    integer               :: nz, i, index

    nz = size(levelsP) - 1; index = nz + 1
    do i = nz, 0, -1
      if (wtd <= levelsP(i + 1)) index = i
    end do
  end function find_wtd_index

end module bubble_geometry
