!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This is the main program for demostrating the bubble formation and growth     !     
! ebullition module (EBG) used in:                                              !
!                                                                               !
! Peltola, O., Raivonen, M., Li, X., and Vesala, T.: Technical Note: Comparison !
! of methane ebullition modelling approaches used in terrestrial wetland models,!
! Biogeosiences Discuss., XXX, 2017.                                            !
!                                                                               !
! Please see the pdf-document and the license file distributed along with these !
! codes and the article Peltola et al., Biogeosciences Discussions xxx, 2017,   !
! for more information.                                                         ! 
!                                                                               !
! If you use these codes or their derivatives in your research, please cite the !
! above-mentioned article.                                                      !
!                                                                               !
! Contact:                                                                      !
! Olli Peltola                                                                  !
! olli.peltola@helsinki.fi                                                      !
!                                                                               !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

program bubble_main

  use bubble_parameters, only: dp, &
  Rgas, por, nlayers, layer_thickness, f, mebuair, mebutot, n_flux_parts
  use bubble_geometry, only: geometry, new_geom
  use bubble_output, only: output_data

  implicit none

  ! time loop control:
  integer :: time_i, n_steps, day_first, day_last
  real(dp) :: time, dt

  ! miscellaneous
  integer :: i
  

  ! input file
  CHARACTER(LEN = 42) :: filein = 'C:\documents\bubble_module\input\meteo.txt'
  ! Tprofile input file
  CHARACTER(LEN = 42) :: filein2 = 'C:\documents\bubble_module\input\Tprof.txt'
  ! predefined CH4 concentration input file
  CHARACTER(LEN = 44) :: filein3 = 'C:\documents\bubble_module\input\CH4prof.txt'
  ! output folder  
  CHARACTER(LEN = 27) :: foldout = 'C:\documents\bubble_module\'



  ! geometry

  ! In principle, we set the model grid to have 'nlayers' background layers,
  ! but we need one extra layer to deal with the moving water table.
  integer :: nz ! will always be nlayers + 1

  ! naming convention: arrays of size = nz end with P (P = "profile"), also
  ! levelsP although its size is nz+1

  real(dp) :: wtd, wtd_eff, wtd_new !,acro_depth
  real(dp), allocatable :: fixed_levelsP(:)
  type(geometry) :: geom

  ! fixed_levelsP: coordinates in [meter] of levels that separate the layers in
  ! the background grid. The moving level will then be inside one of these
  ! layers, at water table level. Or perhaps above fixed_levels(1), if water is
  ! above the peat surface.
  ! geom: see type contents in module methane_geometry. Contains:
  !       levelsP, dzP (arrays)
  !       surfaceQ, airQ, waterQ (logical)
  !       s1, s2, a1, a2, w1, w2 (integer)
  ! wtd: water table depth (WTD), positive is below surface, negative
  !      value means the water surface is above peat surface
  ! wtd_eff: effective wtd, returned by subroutine new_geom. If the input wtd
  !          is too close (parameter min_dz, usually 1 cm) to a fixed level,
  !          we don't accept the value, but round to the fixed level, thus
  !          avoiding to ever have a layer less than 1 cm thick.
  ! acro_depth: acrotelm depth, hopefully never below wtd
  ! por: peat porosity, volume fraction of air or water in peat, we use 80%

  ! temperature
  real(dp), allocatable :: tempP(:)
  ! pressure
  real(dp), allocatable :: presP(:)
  ! xP: amount of x in each layer [mol], really [mol] not [mol m-3]
  real(dp), allocatable :: methaneP(:) ! [mol]
  real(dp), allocatable :: Vp(:) ! bubble volume in different layers [m3]
  real(dp), allocatable :: methanebP(:) ! bubble CH4 in different layers [mol]
  real(dp), allocatable :: z_mid(:) ! depth of each layer


  ! temporary
  real(dp), allocatable :: methane_newP(:)
  real(dp), allocatable :: methane_newbP(:)
  real(dp), allocatable :: alphaP(:)
  real(dp), allocatable :: methanebP_new(:) ! bubble CH4 in different layers
  real(dp) :: methanebub1, methanebub2


  ! input
  integer, parameter :: n_data = 3650, cols_data = 3, cols_data2 = 11
  ! when input data changes, need to adjust this by hand
  real(dp), dimension(cols_data, n_data) :: input_data
  real(dp), dimension(cols_data2, n_data) :: Tprof_data
  real(dp), dimension(cols_data2, n_data) :: CH4prof_data
  integer, dimension(n_data) :: day_data
  real(dp), dimension(n_data) :: temp_data, wtd_data, pres_data

  ! output
  real(dp), dimension(n_flux_parts) :: gases_out, gases_out_rates
  real(dp), dimension(nlayers + 1) :: mebu_rateP



  !!! main program !!!


  nz = nlayers + 1 ! number of layers including the extra moving layer

  ! For now, it would have been simpler to have defined these as constant
  ! length arrays, but I aim for full generality for later.
  allocate(fixed_levelsP(nz), geom % levelsP(nz + 1), geom % dzP(nz))
  allocate(methaneP(nz), methane_newP(nz), methane_newbP(nz))
  allocate(alphaP(nz))
  allocate(tempP(nz))
  allocate(presP(nz))
  allocate(Vp(nz))
  allocate(methanebP(nz))
  allocate(methanebP_new(nz))
  allocate(z_mid(nz))

  ! initialize height levels
  do i = 1, nz
    fixed_levelsP(i) = (i - 1) * layer_thickness
  end do

  ! read input data
  open(11, file = filein, action = 'READ')
  read(11, *) input_data
  close(11)

  day_data = floor(input_data(1,:)) ! day number
  temp_data = input_data(2,:) + 273.15_dp ! temperature [K]
  wtd_data = 0.0_dp ! wtd depth below surface [m]
  pres_data = input_data(3,:) * 100! air pressure [Pa]

  ! read Tprofile input data [K]
  open(11, file = filein2, action = 'READ')
  read(11, *) Tprof_data
  close(11)

  ! read CH4 profile input data [mol m-3]
  open(11, file = filein3, action = 'READ')
  read(11, *) CH4prof_data
  close(11)



  day_first = day_data(1)
  day_last = day_data(size(day_data)) ! takes the last element



  ! initialize geometry
  call new_geom(geom, wtd_eff, nz, fixed_levelsP, wtd_data(1))
  wtd = wtd_eff
  z_mid = (geom % levelsP(1:nz) + geom % levelsP(2:nz + 1))/2 ! midpoint depths



  write(*, *) 'geom%dzP', geom % dzP
  write(*, *) 'CH4prof_data(:,1)', CH4prof_data(:, 1)
  write(*, *) 'press enter'
  read(*, *)
  ! from mol m-3 to mol
  methaneP = CH4prof_data(:, 1) * geom % dzP

  ! time loop preparation
  time = day_first * 24 * 3600
  dt = 24 * 3600 ! 1 day in seconds

  gases_out = 0
  mebu_rateP = 0


  gases_out_rates = gases_out / dt

  presP = estimate_pressure_profile(pres_data(1), geom, nz)
  tempP = Tprof_data(:, 1)

  ! initializing bubble variables
  Vp = 0
  Vp(4:size(Vp)) = 0.01
  ! amount of CH4 moles in bubbles
  methanebP = f * presP * Vp/(Rgas * tempP);
  methanebP_new = 0



  call output_data(time, nz, geom, por, methaneP, gases_out_rates, Vp, foldout, &
  mebu_rateP)

  n_steps = day_last - day_first + 1
  do time_i = 1, n_steps ! main time loop ********************************
    write(*, *) 'day: ', time_i ! debug


    time = time + dt
    wtd_new = wtd_data(time_i)

    ! new predefined CH4 profile. Converting from mol m-3 to mol
    ! normally this would be solved by the model
    methaneP = CH4prof_data(:, time_i) * geom % dzP

    ! new temperature profile
    tempP = Tprof_data(:, time_i)

    ! new pressure profile. There may be changes due to 1) changes in airP
    ! or 2) changes in WTD
    presP = estimate_pressure_profile(pres_data(time_i), geom, nz)


    ! bubble volume (m3)
    Vp = methanebP * Rgas * tempP/(presP * f)
    
    ! amount of CH4 stored in bubbles
    methanebub1 = methane_in_bubbles(Vp, tempP, presP, nz) 


    call bigstep(Vp, presP, methaneP, gases_out, &
    nz, geom, por, dt, tempP, mebu_rateP, methanebP)



    gases_out_rates = gases_out / dt

    ! amount of CH4 stored in bubbles
    methanebub2 = methane_in_bubbles(Vp, tempP, presP, nz) 

    ! outputting
    call output_data(time, nz, geom, por, methaneP, gases_out_rates, Vp, foldout, &
    mebu_rateP)


  end do ! end of main time loop *********************************************


  
  
  
  
  
  
contains

  ! Subroutine for solving the concentration time derivative
  subroutine bigstep(Vp, presP, methaneP, gases_out_bigstep, &
    nz, geom, por, big_dt, tempP, mebu_rate_out, methanebP)
    ! INPUT:
    !       Vp = bubble volume vector, unit=m3, size=nz
    !       presP = pressure vector, unit=Pa, size=nz
    !       methaneP = CH4 pore water concentration vector, unit=mol, size=nz
    !       nz = amount of model layers, size=1
    !       geom = model geometry
    !       por = peat porosity, unitless, size=1
    !       big_dt = lenght of model time step, unit=s, size=1
    !       tempP = pore water temperature vector, unit=K, size=nz
    !       methanebP = CH4 bubble concentration vector, unit=mol, size=nz
    ! OUTPUT:
    !       gases_out_bigstep = vector containing CH4 flux caused by ebullition, unit = mol s-1, size=n_flux_parts
    !       mebu_rate_out = pore water CH4 concentration change rate due to interaction with bubbles, unit = mol s-1, size=nz

    integer, intent(in)         :: nz
    real(dp), intent(inout)     :: Vp(nz), methaneP(nz)
    real(dp), intent(inout)     :: methanebP(nz)
    real(dp), intent(out)       :: gases_out_bigstep(n_flux_parts), mebu_rate_out(nz)
    type(geometry), intent(in)  :: geom
    real(dp), intent(in)        :: presP(nz), por, big_dt, tempP(nz)
    real(dp)                    :: met_d1(nz), met_d2(nz), met_d3(nz), met_d4(nz), &
                                    met_d(nz)
    real(dp)                    :: met_alphaP(nz)
    real(dp)                    :: max_dt, dt
    real(dp)                    :: mebu_out, met_db(nz)
    real(dp)                    :: mebu_rateP1(nz), mebu_rateP2(nz), mebu_rateP3(nz), mebu_rateP4(nz), mebu_rateP(nz)
    integer                     :: n_small_steps, i, j

    met_alphaP = methane_solubility(tempP)
    max_dt = find_max_ebullition_dt()
    max_dt = find_max_methane_to_bubble_dt(presP, tempP, methaneP, methanebP, met_alphaP, &
    nz, geom, por, Vp)


    n_small_steps = ceiling(big_dt / max_dt)
    dt = big_dt / n_small_steps

    gases_out_bigstep = 0
    mebu_rate_out = 0



    do i = 1, n_small_steps

      ! Runge-Kutta-4
      call derivative(methaneP, met_d1, &
      nz, geom, por, tempP, met_alphaP, presP, mebu_rateP1, Vp)

      call derivative(methaneP + 0.5 * dt * met_d1, met_d2, &
      nz, geom, por, tempP, met_alphaP, presP, mebu_rateP2, Vp)

      call derivative(methaneP + 0.5 * dt * met_d2, met_d3, &
      nz, geom, por, tempP, met_alphaP, presP, mebu_rateP3, Vp)

      call derivative(methaneP + dt * met_d3, met_d4, &
      nz, geom, por, tempP, met_alphaP, presP, mebu_rateP4, Vp)


      mebu_rateP = (mebu_rateP1 + 2 * mebu_rateP2 + 2 * mebu_rateP3 + mebu_rateP4)/6
      met_d = (met_d1 + 2 * met_d2 + 2 * met_d3 + met_d4)/6

      ! removing/adding the same amount of CH4 to the bubbles that was added/removed
      ! to the pore water due to interaction with the bubbles
      methanebP = methanebP - dt * mebu_rateP

      ! making sure that the concentrations do not go negative
      do j = 1, nz
        if (methanebP(j) < 0) then
          methaneP(j) = methaneP(j) + methanebP(j)
          methanebP(j) = 0
        end if
      end do

      ! updating bubble volumes
      Vp = methanebP * Rgas * tempP/(presP * f)

      methaneP = methaneP + dt * met_d
      mebu_rate_out = mebu_rate_out + dt * mebu_rateP

    end do

    ! change rate of pore water CH4 due to exchange with bubble
    mebu_rate_out = mebu_rate_out/big_dt

    ! releasing part of the gas volume if it exceeds certain size
    call release_excess_bubbles(Vp, presP, tempP, met_db, mebu_out, geom, por, big_dt, nz, methanebP)

    ! bubbles are released to the lowest air layer if wtd is below surface
    methaneP = methaneP + big_dt * met_db

    ! for output
    ! amount of CH4 released to the atmosphere
    gases_out_bigstep(mebuair) = gases_out_bigstep(mebuair) + big_dt * mebu_out
    ! amount of CH4 released to the lowest air layer if WTD is below the surface
    gases_out_bigstep(mebutot) = gases_out_bigstep(mebutot) + big_dt * sum(met_db)

  end subroutine bigstep


  ! Time derivative of all state variables (arrays), also outputs fluxes
  subroutine derivative(methaneP, methane_d, nz, geom, por, tempP, &
    met_alphaP, presP, mebu_rateP, Vp)
    ! INPUT:
    !       methaneP = CH4 pore water concentration vector, unit=mol, size=nz
    !       nz = amount of model layers, size=1
    !       geom = model geometry
    !       por = peat porosity, unitless, size=1
    !       tempP = pore water temperature vector, unit=K, size=nz
    !       met_alphaP = CH4 solubility (dimensionless Henry solubility) vector, unit=-, size=nz
    !       presP = pressure vector, unit=Pa, size=nz
    !       Vp = bubble volume vector, unit=m3, size=nz
    ! OUTPUT:
    !       methane_d = pore water CH4 concentration change rate, unit = mol s-1, size=nz
    !       mebu_rateP = pore water CH4 concentration change rate due to interaction with bubbles, unit = mol s-1, size=nz


    real(dp), intent(out)       :: methane_d(nz)
    real(dp), intent(out)       :: mebu_rateP(nz)
    integer, intent(in)         :: nz
    type(geometry), intent(in)  :: geom
    real(dp), intent(in)        :: methaneP(nz), por, tempP(nz), met_alphaP(nz)
    real(dp), intent(in)        :: presP(nz), Vp(nz)


    call methane_to_bubble(presP, tempP, mebu_rateP, methaneP, met_alphaP, nz, geom, por, Vp)

    methane_d = mebu_rateP
  end subroutine derivative

  
  ! Finding max dt that is safe for numerics
  pure function find_max_ebullition_dt() result(max_dt)
    use bubble_parameters, only: ebu_hl, time_scale
    real(dp) :: max_dt
    max_dt = time_scale * ebu_hl
  end function find_max_ebullition_dt

  

  ! Finding max dt that is safe for numerics
  function find_max_methane_to_bubble_dt(presP, tempP, methaneP, methanebP, met_alphaP, &
    nz, geom, por, Vp) result(max_dt)
    ! INPUT:
    !       presP = pressure vector, unit=Pa, size=nz
    !       tempP = pore water temperature vector, unit=K, size=nz
    !       methaneP = CH4 pore water concentration vector, unit=mol, size=nz
    !       methanebP = CH4 bubble concentration vector, unit=mol, size=nz
    !       met_alphaP = CH4 solubility (dimensionless Henry solubility) vector, unit=-, size=nz
    !       nz = amount of model layers, size=1
    !       geom = model geometry
    !       por = peat porosity, unitless, size=1
    !       Vp = bubble volume vector, unit=m3, size=nz
    ! OUTPUT:
    !       max_dt = max dt that is safe for numerics, unit = s, size=1

    use bubble_parameters, only: Rgas, Nbub, ebu_hl, f, pi, time_scale
    ! finding a safe time step for numerics


    real(dp)                    :: max_dt
    real(dp), intent(in)        :: methanebP(nz), methaneP(nz), met_alphaP(nz), &
                                  por, presP(nz), tempP(nz), Vp(nz)
    integer, intent(in)         :: nz
    type(geometry), intent(in)  :: geom
    real(dp)                    :: mCwP, pP
    integer                     :: i
    real(dp)                    :: Dw
    real(dp)                    :: dc, r
    real(dp)                    :: mrateP(nz)
    real(dp)                    :: derivative, turnover_rate, turnover_rateb, new_dt



    ! if levelsP(1) < wtd,
    ! then bubbles will transport gas from gas-saturated water layers
    ! to the deepest air layer.
    !
    ! if wtd <= levelsP(1),
    ! then the transport is directly to atmosphere.
    !
    ! mrateP:   The rate at which gas transfer with bubbles modifies gas at each
    !          layer, [mol s-1].

    mrateP = 0
    max_dt = time_scale * ebu_hl;
    max_dt = time_scale * 24 * 3600;

    if (geom % waterQ) then
      ! layers with water peat mixture exist
      do i = geom % w1, geom % w2



        ! Pressure (Pa) = air pressure + hydrostatic pressure
        pP = presP(i)
        ! CH4 molar density in the water (mol m-3)
        mCwP = methaneP(i) / (geom % dzP(i) * por)

        ! concentration difference between pore water and bubble surface
        dc = (mCwP - met_alphaP(i) * f * pP/(Rgas * tempP(i)))



        if (Vp(i) > 0) then
          ! growing the bubble only if it already exists

          ! radius of one bubble (m)
          r = (3.0_dp/4.0_dp * Vp(i)/Nbub/pi)**(1.0_dp/3.0_dp)

          ! CH4 diffusion coefficient in water & peat mixture
          Dw = methane_D_water(tempP(i))


          ! change in pore water CH4 due to mass transfer with a bubble
          mrateP(i) = -4.0_dp * pi * r * Dw * dc


          ! Nbub bubbles
          mrateP(i) = mrateP(i) * Nbub


        end if

      end do

    end if


    do i = 1, nz
      if (mrateP(i) /= 0 .AND. methaneP(i) /= 0 .AND. methanebP(i) /= 0) then
        derivative = abs(mrateP(i))
        turnover_rate = derivative / methaneP(i)
        turnover_rateb = derivative / methanebP(i)
        !        write(*,*)'dt',time_scale * 1 / turnover_rate
        !        write(*,*)'dtb',time_scale * 1 / turnover_rateb
        !        write(*,*)'press enter'
        !        read(*,*)
        turnover_rate = max(turnover_rate, turnover_rateb)

        new_dt = time_scale * 1 / turnover_rate
        if (new_dt < max_dt) then
          max_dt = new_dt
        end if
      end if
    end do

  end function find_max_methane_to_bubble_dt


  
  ! Transferring CH4 between bubbles and the surrounding pore water
  subroutine methane_to_bubble(presP, tempP, mrateP, methaneP, met_alphaP, &
    nz, geom, por, Vp)
    ! INPUT:
    !       presP = pressure vector, unit=Pa, size=nz
    !       tempP = pore water temperature vector, unit=K, size=nz
    !       methaneP = CH4 pore water concentration vector, unit=mol, size=nz
    !       met_alphaP = CH4 solubility (dimensionless Henry solubility) vector, unit=-, size=nz
    !       nz = amount of model layers, size=1
    !       geom = model geometry
    !       por = peat porosity, unitless, size=1
    !       Vp = bubble volume vector, unit=m3, size=nz
    ! OUTPUT:
    !       mrateP = pore water CH4 concentration change rate due to interaction with bubbles, unit = mol s-1, size=nz

    use bubble_parameters, only: Rgas, Nbub, ebu_hl, f, pi
    ! Basically this subroutine creates, grows (and shrinks) bubbles in each layer based on Henry's law,
    ! pressure and temperature changes
    ! Currently considers only CH4


    real(dp), intent(out)       :: mrateP(nz)
    real(dp), intent(in)        :: methaneP(nz), met_alphaP(nz), &
                                  por, presP(nz), tempP(nz)
    integer, intent(in)         :: nz
    type(geometry), intent(in)  :: geom
    real(dp)                    :: mCwP, pP
    integer                     :: i
    real(dp)                    :: Vp(nz)
    real(dp)                    :: Dw
    real(dp)                    :: dc, r

    real(dp), parameter         :: k = log(2.0_dp)/ebu_hl



    ! if levelsP(1) < wtd,
    ! then bubbles will transport gas from gas-saturated water layers
    ! to the deepest air layer.
    !
    ! if wtd <= levelsP(1),
    ! then the transport is directly to atmosphere.
    !
    ! mrateP:   The rate at which gas transfer with bubbles modifies gas at each
    !          layer, [mol s-1].

    mrateP = 0

    if (geom % waterQ) then
      ! layers with water peat mixture exist
      ! forming bubbles which will be released if their volume exceeds certain treshold
      do i = geom % w1, geom % w2



        ! Pressure (Pa) = air pressure + hydrostatic pressure
        pP = presP(i)
        ! CH4 molar density in the water (mol m-3)
        mCwP = methaneP(i) / (geom % dzP(i) * por)

        ! concentration difference between pore water and bubble surface
        dc = (mCwP - met_alphaP(i) * f * pP/(Rgas * tempP(i)))


        if (Vp(i) == 0) then
          ! creating a bubble from the excess CH4
          if (dc > 0) then
            ! making sure that CH4 exchange FROM bubble TO water does not 
            ! exist when there is no bubble
            mrateP(i) = -k * dc * (geom % dzP(i) * por)
          end if

        else
          ! growing the bubble only if it already exists

          ! radius of one bubble (m)
          r = (3.0_dp/4.0_dp * Vp(i)/Nbub/pi)**(1.0_dp/3.0_dp)

          ! CH4 diffusion coefficient in water & peat mixture
          Dw = methane_D_water(tempP(i))


          ! change in pore water CH4 due to mass transfer with one bubble
          mrateP(i) = -4.0_dp * pi * r * Dw * dc

          ! Nbub bubbles
          mrateP(i) = mrateP(i) * Nbub


        end if

      end do

    end if

  end subroutine methane_to_bubble

  
  
  ! Checking if bubble volume at certain depth exceeds the predefined treshold.
  ! If it does, then releasing the extra volume
  subroutine release_excess_bubbles(Vp, presP, tempP, mrateP, mebu_out, geom, por, dt, nz, methanebP)
    ! INPUT:
    !       Vp = bubble volume vector, unit=m3, size=nz
    !       presP = pressure vector, unit=Pa, size=nz
    !       tempP = pore water temperature vector, unit=K, size=nz
    !       geom = model geometry
    !       por = peat porosity, unitless, size=1
    !       dt = model time step, unit=s,size=1
    !       nz = amount of model layers, size=1
    !       methanebP = CH4 bubble concentration vector, unit=mol, size=nz
    ! OUTPUT:
    !       mrateP = pore water CH4 concentration change rate due to interaction with bubbles, unit = mol s-1, size=nz
    !       mebu_out = CH4 released in bubbles to deepest air layer, unit = mol s-1, size=1

    use bubble_parameters, only: Rgas, bubprob, Vmaxfraction, f

    real(dp), intent(out)       :: mrateP(nz), mebu_out
    real(dp), intent(inout)     :: Vp(nz)
    real(dp), intent(in)        :: presP(nz), tempP(nz), dt, por
    real(dp), intent(inout)     :: methanebP(nz)
    integer, intent(in)         :: nz
    type(geometry), intent(in)  :: geom
    real(dp)                    :: mnV
    real(dp)                    :: Vout, nout, Vtmp(nz)
    real(dp)                    :: mebu_out2(nz) ! ebullition from the layers are temporarily saved here       
    real(dp)                    :: randbub
    integer                     :: i, ind


    ! If bubble in a certain layer grows larger than Vmax then it is transported to the deepest air layer
    ! Currently considers only CH4
    
    mrateP = 0; mebu_out = 0
    mebu_out2 = 0;

    Vtmp = Vp

    if (geom % waterQ) then
      ! layers with water peat mixture exist

      ! looping from bottom to top
      do i = geom % w2, geom % w1, -1
        ! CH4 molar density in the bubble (mol m-3)
        mnV = f * presP(i)/(Rgas * tempP(i))

        ! releasing the bubble if it exceeds certain size
        if ((Vtmp(i) - Vmaxfraction * geom % dzP(i) * por) > 1e-11_dp) then

          ! bubble was released


          Vout = Vtmp(i) - Vmaxfraction * geom % dzP(i) * por
          Vtmp(i) = Vmaxfraction * geom % dzP(i) * por

          ! the size of the bubble increases as it ascends, but the amount of moles
          ! in the bubble stay the same
          nout = mnV * Vout
          methanebP(i) = methanebP(i) - nout

          ind = i
          do while (Vout > 0 .AND. ind >= geom % w1 + 1)
            ind = ind - 1

            call RANDOM_NUMBER(randbub)

            if (randbub <= bubprob) then
              ! bubble got stuck
              methanebP(ind) = methanebP(ind) + nout
              Vtmp(ind) = methanebP(ind) * Rgas * tempP(ind)/(f * presP(ind))

              Vout = 0
              nout = 0
            end if

          end do

          ! bubble did not get stuck => it is released to the lowest air layer
          if (Vout > 0) then
            mebu_out2(i) = nout/dt

          end if


        end if
      end do

      ! updating the bubble volume profile
      Vp = Vtmp

      if (geom % airQ) then
        mrateP(geom % a2) = sum(mebu_out2) ! bubbles released to deepest air layer
        ! ebu_out is already 0, no need to set again
      else
        mebu_out = -sum(mebu_out2);
      end if

    end if
  end subroutine release_excess_bubbles

  
  
  ! methane diffusion coefficient for medium consisting of xx% peat,
  ! yy% water, depending on temperature 
  pure elemental function methane_D_water(temperature) result(Dw)
    use bubble_parameters, only: peat_coeff_w

    real(dp), intent(in)  :: temperature ! [K]
    real(dp)              :: Dw
    real(dp), parameter   :: Dw_298 = 1.5e-9_dp ! [m2 s-1]

    ! Arah and Stephen (1998), Eq. 11
    Dw = peat_coeff_w * Dw_298 * (temperature/298._dp)
    ! if (temperature < 273.15) Dw = 0
  end function methane_D_water

  
  
  pure elemental function methane_solubility(temperature) result(alpha)
    real(dp), intent(in)  :: temperature ! [K]
    real(dp)              :: alpha ! [mol(CH4,water) m(water)-3 mol(CH4,air)-1 m(air)3]
    real(dp), parameter   :: a = 1.3e-3_dp, b = 1700._dp, c = 298.0_dp
    real(dp), parameter   :: Rgas = 8.3144621_dp ! [J mol-1 K-1]
    real(dp)              :: Hcp ! [mol(CH4) m(H2O)-3 Pa-1]
    ! Tang et al. 2010, eq. A4
    ! see also http://www.henrys-law.org/henry-3.0.pdf
    Hcp = a * exp(b * (1._dp/temperature - 1._dp/c)) * 9.86923266716e-3_dp
    alpha = Rgas * temperature * Hcp
  end function methane_solubility


  
  ! Moles of CH4 stored in bubbles in the whole model domain
  function methane_in_bubbles(Vp, tempP, presP, nz) &
    result(bubble_methane_tot)
    ! INPUT:
    !       Vp = bubble volume vector, unit=m3, size=nz
    !       tempP = pore water temperature vector, unit=K, size=nz
    !       presP = pressure vector, unit=Pa, size=nz
    !       nz = amount of model layers, size=1
    ! OUTPUT:
    !       bubble_methane_tot = moles of CH4 stored in bubbles in the whole model domain, unit = mol, size=1

    use bubble_parameters, only: Rgas, f
    ! sum of CH4 stored in bubbles in total (mol) in the model domain
    ! Used in estimating the total CH4 balance during calculation

    real(dp), intent(in)  :: Vp(nz), tempP(nz), presP(nz)
    integer, intent(in)   :: nz
    real(dp)              :: bubble_methane_tot


    bubble_methane_tot = sum(f * presP/(Rgas * tempP) * Vp)
  end function methane_in_bubbles


  
  ! Estimating the pressure profile from water table depth and atmospheric pressure
  function estimate_pressure_profile(pres, geom, nz) &
    result(presP)
    ! INPUT:
    !       pres = atmospheric pressure, unit=Pa, size=1
    !       geom = model geometry
    !       nz = amount of model layers, size=1
    ! OUTPUT:
    !       presP = pressure vector, unit = Pa, size=nz

    use bubble_parameters, only: atm

    real(dp), intent(in)        :: pres
    integer, intent(in)         :: nz
    type(geometry), intent(in)  :: geom
    real(dp)                    :: presP(nz)
    real(dp)                    :: z, z_mid(nz), wtd

    real(dp), parameter :: dist_atm = 10.3_dp! 1 atm water height [m], so 1m water height generate 1/dist_atm (atm) pressure

    z_mid = (geom % levelsP(1:nz) + geom % levelsP(2:nz + 1))/2 ! midpoint depths

    presP = pres

    if (geom % surfaceQ) then
      ! WTD is above surface
      wtd = geom % levelsP(1)
      do i = geom % s1, geom % s2
        z = z_mid(i) - wtd

        ! Pressure (Pa) = air pressure + hydrostatic pressure
        presP(i) = pres + z/dist_atm * atm
      end do
    end if

    if (geom % waterQ) then
      ! layers with water peat mixture exist
      wtd = geom % levelsP(geom % w1)
      do i = geom % w1, geom % w2
        z = z_mid(i) - wtd

        ! Pressure (Pa) = air pressure + hydrostatic pressure
        presP(i) = pres + z/dist_atm * atm
      end do


    end if
  end function estimate_pressure_profile

  
end program bubble_main