function g=LognormalDecay(pars,t)
% This function calculates the laplace transform, g, of the lognormal
% distribution, p(k), with laplace transform parameter t.  Calculations are
% performed over log k space: k-->ln k, p(k)-->rho(ln k).  In the case of
% the organic matter decay problem, it calculates the mass fraction, g, 
% remaining as a function of time t: 
% g(t) = int_{-infinity}^infty rho(logk;mu, sigma) exp(-kt) dlogk, 
% where rho(log(k)) is the normal distribution.
%
% g = LognormalDecay(pars,t) returns the laplace transform at times t
% of the lognormal distribution with parameters specified by the 1-by-2
% vector pars.  pars(1)=mu, pars(2)=sigma.  t is a time vector of arbitrary
% length. The output g is a vector having the same length as t.  0 < g < 1
%
% This function discretizes log k over an interval centered at mu,
% [mu - 8*sigma < log k < mu + 8*sigma].  This range is appropriate for
% being within the double precision error of typical combinations of sigma
% and t. In case extreme values of time are specified, the algorithm
% theoretically identifies the maximum of the remaining distribution,
% rho(logk; mu, sigma)*e^(kt), and searches over its width.
%
% by David Forney, forneyd@alum.mit.edu.  2/14/12

%% parameters
mu=pars(1);
sigma=abs(pars(2));

nk=1000;
kf=8;
kp=6;
kmin=exp(mu-kf*sigma);
kmax=exp(mu+kf*sigma);
k=logspace(log10(kmin),log10(kmax),nk);
lk1=log(k);
g=zeros(length(t),1);

% integrate    
    for i=1:length(t)
        T=t(i);
        
         plk=1/(sqrt(2*pi)*sigma)*exp(-(log(k)-mu).^2/(2*sigma^2)).*exp(-k*T);
         lk=lk1;
        if plk(1)/max(plk)>.01,
            sign(min(plk))
            % remaining distribuiton contains significant fraction k < kmin
            warning('Warning:IntegrationDomain','LognormalDecay.m distribution outside of integration limit kmin, using analyitical limits instead')
            % identify maximum of distribution at time =T
            peak = mu - lambertw(sigma^2*T*exp(mu));
            % set new limits about maximum
            kmin2=exp(peak-kp*sigma);
            kmax2=exp(peak+kp*sigma);
            k2=logspace(log10(kmin2),log10(kmax2),nk);
            lk2=log(k2);            
            plk=1/(sqrt(2*pi)*sigma)*exp(-(log(k2)-mu).^2/(2*sigma^2)).*exp(-k2*T);
            lk=lk2;
            
        end
        % integrate
        g(i) = trapz(lk,plk);
        
    end
    
    
