function [rho0,lk,gp,A] = ILaplaceNN(t,g,kmin,kmax,n)
% iLaplaceNN:  Inverse Laplace transform of data g(t) with non-negative
% constraints on the solution rho0.  A nonnegative linear least squares
% algorithm is used.  
%
% [rho0,lk] = iLaplaceNN(t,g) returns the inverse laplace transform with
% non-negative constraints in the vector rho0.  lk is the discretized rates
% log(k) over which rho0 is distributed, rho0=rho0(lk).  Decay data g(t) is
% specified by the m by 1 column vector t, representing time, the m by 1
% column vector g, representing the mass fraction remaining at times t.
% 0 <= g <= 1.  g and t must include the time = 0 data
% point, t=0, g=1. Other optional inputs are set to their default values:
% kmax=10^4, kmin=10^-6, n = 16m (appropriate for LIDET litter decay data
% with time t in years.)
%
% [rho0,lk] = iLaplaceNN(t,g,kmin,kmax) the minimum and maximum rates in the
% search domain are set by the inputs kmin and kmax.
%
% [rho0,lk] = iLaplaceNN(t,g,kmin,kmax,n) also specifies n, the number of
% discretization intervals in rho0, n is the length of the output rho0.
%
% [rho0,lk,gp,A] = iLaplaceNN(...) also outputs the predicted mass loss
% gp, the discretized Laplace transform operator A, the log of the
% discretized rates lk.
%
% This code proceeds by first converting the laplace transform to ln k
% coordinates, 
% g(t) = int_{-infinity}^{infinity} rho0(ln k) exp(-kt) d(ln k). 
% We consider the distribution to be nonzero over the domain
% kmin<k<kmax:  
% g(t) = int_{ln kmin}^{ln kmax} rho0(ln k) exp(-kt) d(lnk) 
% and then discretize that integral as g=A*rho0.  
%
% by David Forney, forneyd@alum.mit.edu.  2/14/12


% set default values if not specified in the input
if nargin<3
    kmax=10^4;
    kmin=10^-6;
    n=16*length(t);
elseif nargin<5
    n=16*length(t);
end


% discretize the Laplace transform over the time range t.  Calculate
% discretized transform A, and the range of log(k), lk.
[A,lk]=discretize(t,kmin,kmax,n);

% set g(0)=1 constraint: weigh the first data point g(0)=1 to make sure rho0
% integrates to 1 and is a probability distribution function.
cf=10;
Acon=A;  Acon(1,:)=A(1,:)*cf;
bcon=g;  bcon(1)=1*cf;

% solve the inverse problem using the nonlinear least squares solver
options=optimset('TolX',100*eps*norm(Acon,1)*length(Acon));
rho0 = lsqnonneg(Acon,bcon,options);
gp=A*rho0;

end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% discretize function
function [A,l]=discretize(t,kn,kx,n)

% this function will discretize the laplace transform integral in the form
% of linear equation g = A rho
%
% INPUT
% t = range of the known output
% kx = max(k), the maximum relevant value of k
% kn = min(k), the minimum relevant value of k
% nk = length of k
%
% OUTPUT
% l = vector log(k) equally spaced between kx and
% kn, logarithmically spaced
%
% A = m x n matrix where n is the length of t.  Aij is
% exp(-k_i*t_j) * w_i.   w_i is the weight dk, the logarithmic spacing.

m = length(t);

A=zeros(m,n);

lx = log(kx);
ln = log(kn);

l = ((1:n)-1/2)*(lx-ln)/n+ln;
w = ones(1,n)*(lx-ln)/n;

for i = 1:m
    A(i,:) = exp(-exp(l)*t(i)).*w;
end

end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
