function [a,k,c,r,type,constpool,initdpool,npoolactive,npooltot]=MultipoolAnalysis(l,rho,t1)
% this function analyzes and extracts pools from the unregularized discrete
% distribution rho(log(k))
%
% [a,k,c,r,type]=MultipoolAnalysis(l,rho,t1) returns the mass fraction of
% active pools as the vector a, the rates associated with active pools as
% the vector k, the mass fraction of the constant inert pool as c, the mass
% fraction of the rapidly leached pool as r, and the model type as type.
% Inputs to this function are rho, the discrete distribution which is the
% output from ILaplaceNN.m, the vector l which is the discretization of 
% ln{k}, also an output from ILaplaceNN.m, and t1 which is time of the
% first measured datapoint after t=0.  
% Model types are as follows:
% 1: 1 active pool (exponential decay)
% 2: 1 active pool + const
% 3: 1 active pool + init decay
% 4: 1 active pool + const + init decay
% 5: 2 active pools
% 6: 2 active pools + const
% 7: 2 active pools + init decay
% 8: 2 active pools + const + init decay
% 9: 3 active pools
% 10: all other combinations of active, constant, and rapid pools
% 
% [a,k,c,r,type,constpool,initdpool,npoolactive,npooltot]=MultipoolAnalyis(l,rho,t1);
% also outputs the flag constpool = 1 if a constant pool is present,
% constpool=0 if not present; initdpool=1 if an initial decay(rapid) pool
% is present, initdpool=0 otherwise; npoolactive = number of active pools
% which decay exponentially; npooltotal = total number of active, constant
% and initially decaying pools.
%
% by David Forney, forneyd@alum.mit.edu.  2/14/12

%% parameters

n=1; % pool counter
m=0; % pool width counter
R0=0; % previous value of rho_i


%% extract pools
for j=2:length(l);
    R=rho(j);
    L=l(j);
    
    % flag beginning of pool
    if R~=0
        m=m+1;
        group(n).rho(m)=R;
        group(n).l(m)=L;
    end
    
    % flag end of pool
    if (R==0 && R0~=0) || (j==length(l) && R~=0)
        m=0;
        n=n+1;
    end
    
    R0=R;
end

% evaluate mean rate and total mass of each pool
a=zeros(n-1,1); k=a;
for j=1:n-1
    rs=group(j).rho/sum(group(j).rho);
    meanl=sum(rs.*group(j).l);
    tot=sum(group(j).rho);
    a(j)=tot;
    k(j)=exp(meanl);
end

%% process the pools

% constant pool always in first element
c=rho(1);

% normalize all pools by total mass
tot=sum(a)+c;
c=c/tot;
a=a/tot;


% const pool flag
constpool=(c>0);

% identify whether right pool is an istantaneous decay pool
r=0;
dp=10^(-14); % give a 3-digit buffer for double precision cutoff
iid=find(k>(-log(dp)/t1));
initdpool=length(iid);
if initdpool
    r=a(iid);
end
% remove instantaneous pool from active pools
a(iid)=[];
k(iid)=[];


% number of active pools
npoolactive=length(a); %-initdpool;

% count total number of pools
npooltot=length(a)+(c>0)+(r>0);


% go through all multi-pool types
% 1: exp decay
% 2: exp decay + const
% 3: exp decay + init decay
% 4: exp decay + const + init decay
% 5: 2G
% 6: 2G + const
% 7: 2G + init decay
% 8: 2G + const + init decay
% 9: 3G
% 10: all others

if npoolactive==1 && constpool==0 && initdpool==0
    type=1;
    
elseif npoolactive==1 && constpool==1 && initdpool==0
    type=2;
    
elseif npoolactive==1 && constpool==0 && initdpool==1
    type=3;
    
elseif npoolactive==1 && constpool==1 && initdpool==1
    type=4;
    
elseif npoolactive==2 && constpool==0 && initdpool==0
    type=5;
    
elseif npoolactive==2 && constpool==1 && initdpool==0
    type=6;
    
elseif npoolactive==2 && constpool==0 && initdpool==1
    type=7;
    
elseif npoolactive==2 && constpool==1 && initdpool==1
    type=8;
    
elseif npoolactive==3 && constpool==0 && initdpool==0
    type=9;
    
else
    
    type=10;
end
end