function [rho_c,lk,A,L,lambda_c,err_c,rgh_c,lambda,err,rgh,flag]=ReguILT(t,g,kmin,kmax,n)
% Regularized Inverse Laplace Transform for decay studies
% David Forney ---  2/14/12
% 
% This program inverts a decay time series to calculate a distribution of
% rates rho(log(k)) associated with the decay.  It uses Tikhonov
% regularization to solve the problem rho = inv(A+lambda*L)*g for various
% values of the regularization parameter lambda as described in Hansen PC,
% REGULARIZATION TOOLS: A Matlab package for analysis and solution of
% discrete ill-posed problems, Numerical Algorithms, 1994, 1-35.  More
% information can be found at http://www2.imm.dtu.dk/~pch/Regutools/
%
% [rho_c,lk] = ReguILT(t,g) returns the regularized inverse laplace
% transform with non-negative constraints in the vector rho_c.  rho_c is
% discretized over the values of log(k) specified in the vector lk.  The
% decay data g(t) is specified by the time vector t and decay (mass
% fraction remaining) vector g.  The vectors t and g must be m by 1 column
% vectors.  g represents mass fraction remaining, with values 0 <= g <= 1.
% g and t must include the time = 0 data point, t=0, g=1.  Optional inputs
% not specified are set to their default values: kmax=10^4, kmin=10^-6, n =
% 16m (appropriate for LIDET litter decay data with time units in years.) n
% is the length of the output vectors lk and rho_c.  If decay data has
% different units or has a duration drastically different than the 2 - 10
% yr duration of the LIDET study, then kmax and kmin need to be specified.
%
% [rho_c,lk] = ReguILT(t,g,kmin,kmax) the minimum and maximum rates in the
% search domain are set by the inputs kmin and kmax.
%
% [rho_c,lk] = ReguILT(t,g,kmin,kmax,n) also specifies n, the number of
% discretization intervals in lk and rho_c, n is the length of the outputs 
% rho_c and lk.
%
% [rho_c,lk,A,L,lambda_c,err_c,rgh_c,lambda,err,rgh,flag] = ReguILT(...)
% also outputs the discretized Laplace transform operator A; the roughness
% operator L, the value lambda_c of the regularization parameter associated
% with the corner and solution rho_c; the RMSE (error norm) associated with
% the corner, err_c; the rescaled norm of roughness associated with the
% corner, rgh_c; the vector lambda containing all values of the
% regularization parameter used to generate the L-curve; the vector err
% containing the RMSE of the entire L-curve; and rgh containing the
% roughness norm of the entire L-curve. Flag takes three values: 1) the
% simplest solution is an exponential decay associated with lambda =0, no
% need for regularization.  2) The method fails because all portions of the
% L-curve contain solutions having a significant trend in the residual
% error.  3) warning that the solution rho does not appear to be finite
% over the range of kmin and kmax chosen, user may widen the search range
% kmax, kmin if desired.
%
% USAGE 1) function ReguILT, will plot the L-curve and attempt to
% automatically identify the corner.  The L-curve is plotted on figure 1
% and the solution estimate rho is plotted on figure 2, and the data and
% prediction is plotted on figure 3.
% 
% 2) The user is prompted on the command line to verify that the automatic
% corner is appropriate, based on looking at the fit of the data in Figure
% 3 and the statistics of the current corner are displayed on the matlab
% command line. These statistics include pearson and kendall correlation
% coefficients and p-values of the resdiual error and an estimate of the
% solution bias fraction = (1/m)*sum(g_i - \hat{g}_i)/RMSE, where g_i is
% predicted and \hat{g} is measured, and m is the length of g.  Solutions
% with p-values < .1 or bias fraction > .1 generally are not recommended.
% The user can look at both the log-scale and linear-scale L-curves.  The
% user must enter 'y' to verify the corner is correct (default), or 'n' to
% choose a different corner.
%
% 3) After typing 'n' the user must click the mouse to select the desired
% corner on either the log or linear scale plots.  The statistics of the
% chosen corner now appear on the command line and Figures 2 and 3
% containing the distribution rho and the fit to the data are updated.
% Repeat the process until the smoothest solution which fits the data
% without residual trends or bias is identified.
%
% requires: MultipoolAnalysis.m
%
% by David Forney, forneyd@alum.mit.edu.  2/14/12

%% INVERSION PARAMETERS
% search for distribution p(k) over the range of rates specified here

% set default values of search range if not specified in the input
if nargin<3
    kmax=10^4;
    kmin=10^-6;
    n=16*length(t);
elseif nargin<5
    n=16*length(t);
end


cf=10;  % constraint factor to force the g(0)=1 constraint.
figL=1; % figure on which L-curve is drawn.
figrho=2;  % figure on which rho(log(k)) is drawn.
figtime=3; % figure on which g(t) is drawn.

pcutoff=.1; % kendall rank correlation p-value cutoff, do not consider 
% solutions with lower p-value than pcutoff.

LCurveOpt='log'; %[linear|log] %scaling chosen to automatically pick the 
% corner of the L-curve

%% MODEL SETUP

% discretize the Laplace transform to the linear form g = A * rho
[A,lk]=discretize(t,kmin,kmax,n);

% incorperate the g(0)=1 constraint into the A matrix.  This is done via a
% weighting on the first data point by the factor cf
Acon=A;  Acon(1,:)=A(1,:)*cf;
bcon=g;  bcon(1)=1*cf;

% roughness matrix L.  Here, the first derivative is used as the
% roughness operator.  
L=L1pdf(n);  

%% CHECK UNREGULARIZED INVERSION IN CASE OF ONE RATE
options=optimset('TolX',100*eps*norm(Acon,1)*length(Acon));
[x_u,~] = lsqnonneg(Acon,bcon,options);

t1=t(2); % first measured datapoint
[x_mod,npool]=checkexponential(x_u,lk,t1);

% if exponential (only 1 pool), then thats the simplest solution, return
% the distribution rho containing only one peak
if npool==1
    rho_c=x_mod;
    lambda_c=0;
    rgh_c=norm(L*x_mod)/sqrt(length(x_mod));
    err_c=norm(A*x_mod-g)/sqrt(length(g));    
    lambda=[];
    err=[];
    rgh=[];
    flag=1;   
return
end

%% REGULARIZED INVERSION

% Calculate L-curve, 
[rho,lambda,err,rgh] = Lcurve(bcon,Acon,L);

% find corner in the portion of the L-curve not having significant trend
% in the residual error by considering only the region having a kendall 
% rank correlation pvalue < pcutoff
plambda=zeros(1,length(lambda));

% find p-vals of residual error
for i=1:length(lambda)
    res=A*rho(:,i)-g;
    %res_u(i)=sqrt(sum((A*rho(:,i)-A*x_u).^2)/length(g));
    [~,pv]=corr(t(2:end),res(2:end),'type','kendall');
    plambda(i)=pv;    
end
[idx]=find(plambda>pcutoff);

% if the residual error of all solutions contains a trend, then the data 
% doesn't appear to be described by a superposition of exponenetial decays.
if(isempty(idx)) 
    fprintf('superposition of exponentials does not fit data')
    rho_c=[];
    lambda_c=[];
    err_c=[];
    rgh_c=[];
    lambda=[];
    err=[];
    rgh=[];
    flag=2;
return 
end

% input only the p>.1 portion of L-curve to corner finding algorithm
rgh_r=rgh(idx);
err_r=err(idx);
lambda_r=lambda(idx);
[~,~,lambda_c] = findcorner(lambda_r,rgh_r,err_r,LCurveOpt);

% calculate rho(log(k)) associated with automatic corner
[x_c,err_c,rgh_c]=tikhnn(Acon,L,bcon,lambda_c);



%% DRAW L-CURVE, CHOOSE CORNER
% Draw L-curve, rho(log(k)), and g(t), and show automatically chosen corner

            
% inspect automatic corner, verify it is indeed a "corner", and check
% the properties of the trend in residual error
q=0;
while q<1
    
    % evaluate properties of corner in order to decide whether there is a trend
    % or bias in residual error.
    respick=A*x_c-g;
    rsse=sqrt(sum(respick.^2));
    display(rsse);
    bias=sum(respick);
    display(bias);
    biasfraction=bias/rsse;
    display(biasfraction);
    [rKendall,pKendall]=corr(t(2:end),respick(2:end),'type','kendall');
    [rPearson,pPearson]=corr(t(2:end),respick(2:end),'type','pearson');
    display(rKendall); display(pKendall);
    display(rPearson); display(pPearson);


    plot_L(figL,figrho,figtime,err,rgh,err_c,rgh_c,lk,x_c,A,t,g,idx);
    
    % Ask user if the corner chosen is satisfactory.  If not,
    % repeat selection again.
    condi=input('Is corner ok? ([y]/n) \n','s');    

    if isempty(condi) || condi=='y'
        q=1;
    else
        q=0;
        fprintf('choose corner manually \n')
        [lambda_c,err_c,rgh_c,id]=manualcorner(lambda,err, ...
        rgh,figL);
        x_c=rho(:,id);
    end
end

rho_c=x_c;

% set flag for finding probability distribution which has terminated at the
% endpoints.  If end points have non-zero values its not necessarily bad.
% However the user may wish to try a wider range of kmax and kmin.
% Otherwise the method indicates complete inert components exist and/or
% immediately lost components exist.
flag=3;

% flag=0 if finite solution is found
if rho_c(1)==0 && rho_c(end)==0 
    flag=0; 
end

end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%---- FUNCTIONS --------------------------------------
%% discretize function
function [A,l]=discretize(t,kn,kx,n)

% this function will discretize the laplace transform integral in the form
% of linear equation g = A rho
%
% INPUT
% t = range of the known output
% kx = max(k), the maximum relevant value of k
% kn = min(k), the minimum relevant value of k
% nk = length of k
%
% OUTPUT
% l = vector log(k) equally spaced between kx and
% kn, logarithmically spaced
%
% A = m x n matrix where n is the length of t.  Aij is
% exp(-k_i*t_j) * w_i.   w_i is the weight dk, the logarithmic spacing.
%

m = length(t);

A=zeros(m,n);

lx = log(kx);
ln = log(kn);

l = ((1:n)-1/2)*(lx-ln)/n+ln;
w = ones(1,n)*(lx-ln)/n;

for i = 1:m
    A(i,:) = exp(-exp(l)*t(i)).*w;
end

end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% L1pdf.m
function L = L1pdf(n)
% L1pdf Compute discrete first derivative operator for a pdf.  The first
% and last rows of L are modified since L is operating on a pdf which is
% zero valued outside of the domain of the pdf.  
% 
% INPUT
% n: length of vector rho which L operates on
% OUTPUT
% L: discrete first derivative operator for operation on a pdf
%
% Computes the discrete approximation L to the first derivative operator
% on a regular grid with n points
% L is (n+1)-by-n.
%
% This code is a modified version of a code in the Regularization tools
% package by Per Christian Hansen, 02/05/98.
d=1;

c = [-1,1,zeros(1,d-1)];
nd = n-d;

for i=2:d, c = [0,c(1:d)] - [c(1:d),0]; end

L = sparse(nd,n);
for i=1:d+1
  L = L + sparse(1:nd,(1:nd)+i-1,c(i)*ones(1,nd),nd,n);
end

L=full(L);

% modify L to account for zeros in the distribution outside the range of k.
L1= [1 zeros(1,n-1)]; 
Lend=[zeros(1,n-1) -1];
L=[L1;L;Lend];

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function  [Xlambda,lambda,err,rgh] = Lcurve(b,A,L)
% This function solves the non-negative least squares minimization 
% problem for a range of the regularization parameters lambda.  The 
% output can be used to draw an L-curve.
%
% INPUT
% b: data vector (mass remaining values).  
% A: discrete constraint Laplace transform operator
% L: discrete roughness operator
%
% OUTPUT
% Xlambda: matrix containing the solution vectors x for each value of
%          lambda.  It has size nA by nlambda, where nA is the # of columns 
%          in A and nlambda is the length of the vector lambda. 
% lambda: vector of regularization parameters.
% err: the norm of residual error, Ax-b for each lambda, has length nlambda
% rgh: the norm of roughness vector Lx for each lambda, has length nlambda
%% PARAMETERS
npoints=200; % number of points in l-curve
smin_ratio=1e-4; % sets smallest regularization parameter.
smax_ratio=1e3; % sets largest regularization parameter
%% MODEL SETUP
[~,nA]=size(A);
Xlambda=zeros(nA,npoints);
err=zeros(npoints,1);
rgh=err;

% choose range of lambda's based on the generalized singular values of A
% and L
[~,~,~,C,S] = gsvd(A,L);
s=[diag(C); diag(S)];

% regularization parameters lambda vary from the largest singular value to
% to one that is many orders of magnitude smaller.
lambda_min = max([min(s),max(s)*smin_ratio]);
lambda_max = max([max(s).^2,max(s)*smax_ratio]);
lambda = logspace(log10(lambda_min),log10(lambda_max),npoints)';


%% Loop to solve optimization problem for each value of lambda
for i=1:npoints
   
     % convert the regularization problem to a normal least squares
     % problem for input to lsqnonneg.m
     [x_lambda,ERR,RGH] = tikhnn(A,L,b,lambda(i));
     err(i)=ERR; rgh(i)=RGH;  Xlambda(:,i)=x_lambda;

end


end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% tikhnn.m
function [x_lambda,rho,eta] = tikhnn(A,L,b,lambda)
% solves the non-negative least squares problem ...
% min(||b-Ax||^2 - lambda^2 ||Lx||^2)
%
% INPUT
% A: discrete laplace transform operator matrix
% L: roughness operator
% lambda: regularization parameter
% 
% OUTPUT
% x_lambda: vector solution x for the specified lamba
% rho: residual error norm ||b-Ax||
% eta: roughness norm ||Lx||


Ann=[A;lambda*L]; bnn=[b;zeros(size(L,1),1)];
    
% now solve the least squares non-negative problem
options=optimset('TolX',100*eps*norm(Ann,1)*length(Ann));
[xnn,~,~,exitflag,~,~] = lsqnonneg(Ann,bnn,options);
if exitflag==0
    warning('lsqnonneg:NoSolutions','lsqnonneg did not converge')
end

% semi-norm is eta, residual is rho
eta=norm(L*xnn)/sqrt(length(xnn));
rho=norm(A*xnn-b)/sqrt(length(b));
x_lambda=xnn;

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [g,Lambda_n,lambda_c] = findcorner(lambda,eta,rho,opt)

% Initialization.
% check for any repeats at each end and 
% remove curvatures due to tiny flucutations in rho, eta
% take only 6 sig figs
p=6;
ordereta=floor(log10(eta));
orderrho=floor(log10(rho));
etaH=10.^(p-ordereta-1).*eta;
rhoH=10.^(p-orderrho-1).*rho;
etaR=round(etaH);
rhoR=round(rhoH);
er=[etaR,rhoR];
[~,id]=unique(er,'rows');
id=sort(id);
eta=er(id,1)./10.^(p-ordereta(id)-1); 
rho=er(id,2)./10.^(p-orderrho(id)-1);
lambda=lambda(id);

% Estimate the derivative of the L-curve
switch(opt)
    case 'linear'
        leta=eta;
        lrho=rho;
    case 'log'
        leta=log(eta);
        lrho=log(rho);
end
dleta=diff(leta);
dleta1=dleta(1:end-1);
dleta2=dleta(2:end);
dlrho=diff(lrho);
dlrho1=dlrho(1:end-1);
dlrho2=dlrho(2:end);
dlambda=diff(lambda);
dlambda1=dlambda(1:end-1);
dlambda2=dlambda(2:end);


% get derivative at Lambda_n.  Lambda_n is the lambda at the center of the
% first and second derivative. 
Lambda_n=lambda(2:end-1)/2+(lambda(1:end-2)+lambda(3:end))/4;

% first derivative @ Lambda_n
dlogeta=(dleta1./dlambda1+dleta2./dlambda2)/2;
dlogrho=(dlrho1./dlambda1+dlrho2./dlambda2)/2;

% second derivative @ Lambda_n
ddlogeta=(dleta2./dlambda2-dleta1./dlambda1)./(dlambda1/2+dlambda2/2);
ddlogrho=(dlrho2./dlambda2-dlrho1./dlambda1)./(dlambda1/2+dlambda2/2);

% Let g = curvature.
g =   (dlogrho.*ddlogeta - ddlogrho.*dlogeta)./...
      (dlogrho.^2 + dlogeta.^2).^(1.5);

[~,gi]=max(g);
lambda_c=Lambda_n(gi);

end    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [lambda_pick,err_pick,rgh_pick,id]=manualcorner(lambda,err,rgh,figL)
% find the corner of an L-curve plotted on a figure.
% INPUT
% lambda: vector of regularization parameter values
% err: residual error norm vector corresponding to lambda 
% rgh: roughness norm vector corresponding to lambda
% figL: figure on which L-curve has been plotted
%
% OUTPUT
% lambda_pick: lambda associated with the user chosen corner
% err_pick: error associated with the user chosen corner
% rgh_pick: roughness associated with the user chosen corner
% id: index of the vectors lambda, err, rgh corresponding to the corner

figure(figL);

[err_g,rgh_g]=ginput(1);

dist=(err_g-err).^2+(rgh_g-rgh).^2;
[~,id]=min(dist);
rgh_pick=rgh(id);
err_pick=err(id);
lambda_pick=lambda(id);

end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [hL,hrho,htime]=plot_L(figL,figrho,figtime,err,rgh,err_c,rgh_c,...
                                lk,x,A,t,g,idx)
% plot the L-curve, rho(log(k)), g(t) and model fit to g(t).
% INPUT
% figL = figure number of L-curve
% figrho = figure number of solution rho(log(k))
% figtime = figure number of g(t)
% err = residual error norm vector
% rgh = roughness norm vector
% err_c = residual norm of the solution
% rgh_c = roughness norm of the solution
% lk = log(k) vector
% x = solution vector rho(log(k))
% A = discrete constrained laplace transform operator
% t = time vector
% g = data vector
%
% OUTPUT
% hL = handle to the L-curve plot
% hrho = handle to the rho(log(k)) plot
% htime = handle to the time decay plot
                            
            % Draw L-curve
            figure(figL)
            subplot(211)
            plot(err,rgh,'r',err(idx),rgh(idx),'b',err_c,rgh_c,'bo','markersize',8);
            xlabel('Residual Error Norm ||A\rho -b||')
            ylabel('Roughness Norm ||L \rho||')
            title('L-curve')
            subplot(212)
            hL=loglog(err,rgh,'r',err(idx),rgh(idx),'b',err_c,rgh_c,'bo','markersize',8);

            xlabel('Residual Error Norm ||A\rho -b||')
            ylabel('Roughness Norm ||L \rho||')


            
            % Draw distribution rho(log(k))
            figure(figrho)
            hrho=semilogx(exp(lk),x,'b');
            xlabel('log k')
            ylabel('\rho(log(k))')
            
            % Draw decay data fit
            gfit=A*x;
            figure(figtime)
            htime=plot(t,g,'o',t,gfit,'b');
            xlabel('time')
            ylabel('mass fration remaining')
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [rho,npool]=checkexponential(rho,lk,t1)
% analyze unregularized solution to count the number of pools present.  If
% only one rate pool is present, then solution is an exponential decay
%
% INPUT
% rho: unregularized solution
% lk: log(k) log of rates associated with solution rho
% t1: the first measured datapoint (first datapoint after t=0)
%
% OUTPUT
% rho: unregularized solution, modified in case a negligible pool is
% present
% npool: number of pools in the solution

% parameters
lim=.99;

% analyze unregularized inversion to identify whether there is one pool
% (exponential decay)
[mg,mk,~,~,~,~,~,~,npool]= ... 
    MultipoolAnalysis(lk,rho,t1);

% only one pool return rho to main program
if npool==1    
    return        
end
% if there are multiple pools, check if one is dominant having mass
% fraction > lim,  If so, consider other peaks in rho neglibible and
% remove them.
if sum(mg > lim)
    dlk=diff(lk(1:2));
    idk=find(mg > lim);
    bl=log(mk(idk));
    idl= lk > (bl+dlk) | lk < (bl-dlk);
    rho(idl)=0;
    rho=rho/mg(idk); % rescale the distribution
    
    % analyze the modified distribution
    [~,~,~,~,~,~,~,~,npool]= ...
        MultipoolAnalysis(lk,rho,t1);
end
end