# pertco2.R - Routine to compute coefficients for anthropogenic CO2 perturbatino approach
# James Orr
# LSCE, Saclay, France
# 28 Feb 2011

  library(seacarb)
# library(ggplot2)
# setwd("/homel/orr/R/gattuso/")

# 1) Set initial conditions (must be changed for MedSeA model)
# pCO2o = 280.0
  xCO2o = 280.0  #Mole fraction (ppm) measured in dry air

# Global ocean:
# Temp <- seq(-2, 30, by=1)
  Temp <- seq( 0, 30, by=1)
  Salt = 35
  TA = 2300e-6
  delpco2MAX = 200. 

# Mediterranean Sea:
  Temp <- seq(13, 30, by=1)
# Salt = 38
# TA = 2600e-6
# Salt <- 38
  Salt <- 38.20208
  TA   <- 73.7*Salt - 285.7 #From Scneider et al. (2007, GRL), namely their equation (1) for surface TA = f(salinity)
  TA <- TA * 1e-6
  delpco2MAX = 280.

# 2) Correct for atmospheric humidty (100%) at equilibrium, see Sarmiento et al. (1992; equation 9)
#    Compute initial field pCO2o, which varies with temperature (due to humidity correction)
  tk <- Temp + 273.15   #Absolute temperature (K)
  LNespa <- 20.1050 - 0.0097982*tk - 6163.10/tk
  espa <- exp(LNespa)
  pCO2o <- xCO2o * (1 - espa)   #Convert from mole fraction (ppm) to partial pressure (uatm), assumes Patm = 1 atm

# Fit of coefficients z0 and z1 will be degraded with higher deldicMAX
# deldicMAX = 50.    
# deldicMAX = 150.
# deldicMAX is no longer used; Use delpco2MAX instead (as specified above)

# 3) Compute corresponding reference DIC field [dic0 = f(T)]
  carb0 <- carb(flag=24, var1=pCO2o, var2=TA, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
  dic0 <- carb0$DIC

# 4) Increase DIC incrementally, computing pCO2 each increment
  DIC  <- numeric(0) ; pCO2 <- numeric(0) 
  delDIC  <- numeric(0) ; delpCO2 <- numeric(0) 
# for (i in seq(5, deldicMAX, 5)) {
    #carbdel <- carb(flag=15, var1=TA, var2=dic0+i*1e-6, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
  for (i in seq(5, delpco2MAX, 5)) {
    print(c("delpCO2 =", i))
    carbdel <- carb(flag=24, var1=pCO2o+i, var2=TA, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
    dic <- carbdel$DIC
    pco2 <- carbdel$pCO2
    DIC  <- rbind(DIC, dic)
    pCO2 <- rbind(pCO2, pco2)
    delDIC  <- rbind(delDIC, dic - dic0)
    delpCO2 <- rbind(delpCO2, pco2 - pCO2o)
  }

# 5) Compute linear regression coefficients (slope and intercept) as a function of temperature
  delDIC <- delDIC * 1e+6   #Convert from mol/kg to umol/kg
  ratio <- delpCO2 / delDIC

  Z0  <- numeric(0) ; Z1  <- numeric(0) 
  for (i in 1:length(Temp)) {
    z0 <- lm(ratio[,i] ~ delpCO2[,i])$coeff[[1]]  #Intercept
    z1 <- lm(ratio[,i] ~ delpCO2[,i])$coeff[[2]]  #Slope
    Z0 <- append(Z0, z0)
    Z1 <- append(Z1, z1)
  }

# 6) Fit each linear regression coefficient to a quadratic equation in Temperature
    t <- Temp
    t2 <- t^2
    s0 <- summary(lm(Z0 ~ t + t2))
    s1 <- summary(lm(Z1 ~ t + t2))
    print(s0)
    print(s1)

# 7) Retrieve coefficients from quadratic fits, and write them to output file

#   Coefficients, associated errors, and R^2 for Z0 = a0 + a1*t + a2*t^2
    a0 <- (lm(Z0 ~ t + t2))$coeff[[1]]
    a1 <- (lm(Z0 ~ t + t2))$coeff[[2]]
    a2 <- (lm(Z0 ~ t + t2))$coeff[[3]]
    ae0 <- s0$coefficients[[1,2]]      #Std. Error of a0
    ae1 <- s0$coefficients[[2,2]]      #Std. Error of a1
    ae2 <- s0$coefficients[[3,2]]      #Std. Error of a2
    aR2 <- s0$adj.r.squared            #Adjusted R^2

#   Coefficients, associated errors, and R^2 for Z1 = a0 + a1*t + a2*t^2
    b0 <- (lm(Z1 ~ t + t2))$coeff[[1]]
    b1 <- (lm(Z1 ~ t + t2))$coeff[[2]]
    b2 <- (lm(Z1 ~ t + t2))$coeff[[3]]
    be0 <- s1$coefficients[[1,2]]      #Std. Error of b0
    be1 <- s1$coefficients[[2,2]]      #Std. Error of b1
    be2 <- s1$coefficients[[3,2]]      #Std. Error of b2
    bR2 <- s1$adj.r.squared            #Adjusted R^2
   
#   all <- c(a0, ae0, a1, ae1, a2, ae2, aR2, b0, be0, b1, be1, b2, be2, bR2, "\n")
#   all <- dmeata.frame(all)
#   write.table(all, file="coeffs.dat", append = TRUE)
#   cat(all, file="coeffs.dat", append = TRUE)

#   Print out all coefficients in 1 file, including R2 values as well
    allp <- c(a0,"+/-",ae0, a1,"+/-",ae1, a2,"+/-",ae2, aR2, b0,"+/-",be0, b1,"+/-",be1, b2,"+/-",be2, bR2, "\n")
    cat(allp, file="coeffs.dat", append = TRUE)

# 8) Verify on Fitted vs. True results
    x0 <- a0 + a1*t +a2*t2
    x1 <- b0 + b1*t +b2*t2
    ddic = 50
    ddic = 100
    ddic = 150
    ddic = 175
#   ddic = 225
    carbd <- carb(flag=15, var1=TA, var2=dic0+ddic*1e-6, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
    dpCO2true <- carbd$pCO2 - pCO2o
    dpCO2fit <- (x0 * ddic) / (1 - x1*ddic)
    plot(x=Temp, y=dpCO2true)
    lines(x=Temp, y=dpCO2fit, col="blue")

#   Rough conclusion: relative errors always less than 0.5% when delpCO2 < 280 (i.e., atm pCO2 ~ 560 ppm)
    plot(x=Temp, y=100*(dpCO2fit - dpCO2true)/dpCO2true, col="blue")

