# pertco2_salt.R - Routine to compute coefficients for anthropogenic CO2 perturbation approach
# James Orr
# LSCE, Saclay, France
# 2 March 2011 (based on pertco2.R)

  library(seacarb)
# library(ggplot2)
# setwd("/homel/orr/R/gattuso/")

# 1) Set initial conditions (must be changed for MedSeA model)
# pCO2o = 280.0
  xCO2o = 280.0  #Mole fraction (ppm) measured in dry air

# Global ocean:
# Temp <- seq(-2, 30, by=1)
  Temp <- seq( 0, 30, by=1)
  Salt = 35
  TA = 2300e-6
  delpco2MAX = 200. 

# Mediterranean Sea:
  Temp <- seq(13, 30, by=1)
  Salt = 38
  TA = 2600e-6
  delpco2MAX = 280.

  salt <- seq(33.5, 40.0, by=0.5)
  temp <- seq(13, 30, by=1)
  hydro <- expand.grid(temp, salt)
  Temp <- hydro$Var1
  Salt <- hydro$Var2
  TA <- Salt
  TA <- (73.7*Salt - 285.7)*1e-6     #Schneider et al. (2007)
  TA2 <- (93.996*Salt - 1038.1)*1e-6 #Copin et Begovic (2002)


# 2) Correct for atmospheric humidty (100%) at equilibrium, see Sarmiento et al. (1992; equation 9)
#    Compute initial field pCO2o, which varies with temperature (due to humidity correction)
  tk <- Temp + 273.15   #Absolute temperature (K)
  LNespa <- 20.1050 - 0.0097982*tk - 6163.10/tk
  espa <- exp(LNespa)
  pCO2o <- xCO2o * (1 - espa)   #Convert from mole fraction (ppm) to partial pressure (uatm), assumes Patm = 1 atm

# Fit of coefficients z0 and z1 will be degraded with higher deldicMAX
# deldicMAX = 50.    
# deldicMAX = 150.
# deldicMAX is no longer used; Use delpco2MAX instead (as specified above)

# 3) Compute corresponding reference DIC field [dic0 = f(T)]
  carb0 <- carb(flag=24, var1=pCO2o, var2=TA, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
  dic0 <- carb0$DIC

# 4) Increase DIC incrementally, computing pCO2 each increment
  DIC  <- numeric(0) ; pCO2 <- numeric(0) 
  delDIC  <- numeric(0) ; delpCO2 <- numeric(0) 
  Tc  <- numeric(0) ; Sc <- numeric(0) 
# for (i in seq(5, deldicMAX, 5)) {
    #carbdel <- carb(flag=15, var1=TA, var2=dic0+i*1e-6, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
# for (i in seq(5, delpco2MAX, 5)) {
# delpco2RANGE <- c(seq(10, 100, 10)) 
  delpco2RANGE <- c(0.1, 1, seq(5, delpco2MAX, 5)) 
  delpco2RANGE <- c(seq(5, delpco2MAX, 5)) 
  for (i in delpco2RANGE) {
    print(c("delpCO2 =", i))
    carbdel <- carb(flag=24, var1=pCO2o+i, var2=TA, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
    dic <- carbdel$DIC
    pco2 <- carbdel$pCO2
    DIC  <- rbind(DIC, dic)
    pCO2 <- rbind(pCO2, pco2)
    delDIC  <- rbind(delDIC, dic - dic0)
    delpCO2 <- rbind(delpCO2, pco2 - pCO2o)
    tc <- carbdel$T
    sc <- carbdel$S
    Tc  <- rbind(Tc, tc)
    Sc  <- rbind(Sc, sc)
  }

# 5) Compute linear regression coefficients (slope and intercept) as a function of temperature
  delDIC <- delDIC * 1e+6   #Convert from mol/kg to umol/kg
  ratio <- delpCO2 / delDIC

  Z1  <- numeric(0) ; Z2  <- numeric(0) ;  Z3  <- numeric(0)
  nsalt = length(salt)
  ntemp = length(temp)
# for (it in 1:length(Temp)) {
  for (is in 1:nsalt) {
    for (it in 1:ntemp) {
      n <- (is - 1) * ntemp + it
      dp <- delpCO2[,n]
      ddic <- delDIC[,n]
      ddic2 <- ddic*ddic
      ddic3 <- ddic*ddic2
#     f <- lm(ratio[,n] ~ dp)         #Fit to ratio = a0 + a1*dp
#     f2 <- lm(ratio[,n] ~ dp + dp2)  #Fit to ratio = a0 + a1*dp + a2*dp2
#
#     Switch to polynomial of dpCO2 = f(dDIC)
      f <- lm(dp ~ 0 + ddic + ddic2 + ddic3)  #Fit to dpCO2 = 0 + a1*ddic + a2*ddic^2 + a3*ddic^3
      z1 <- f$coeff[[1]]  #ddic
      z2 <- f$coeff[[2]]  #Slope
      z3 <- f$coeff[[3]]  #coefficient on dp2 term
      Z1 <- append(Z1, z1)
      Z2 <- append(Z2, z2)
      Z3 <- append(Z3, z3)
    }
  }

# 6) Fit each linear regression coefficient to multivariate quadratic equations in T and S
    t <- Temp
    s <- Salt
    t2 <- t^2
    s2 <- s^2
    s3 <- s2 * s
    t3 <- t2 * t
    ts <- t*s
    t2s <- t2*s
    ts2 <- t*s2
#   s4 <- s^4
#   sinv <- 1/s
    fitZ1 <- lm(Z1 ~ t + s + t2 + s2 + t3 + s3 + ts + t2s + ts2)
    fitZ2 <- lm(Z2 ~ t + s + t2 + s2 + t3 + s3 + ts + t2s + ts2)
    fitZ3 <- lm(Z3 ~ t + s + t2 + s2 + t3 + s3 + ts + t2s + ts2)
    sum1 <- summary(fitZ1)
    sum2 <- summary(fitZ2)
    sum3 <- summary(fitZ3)
    print(sum1)
    print(sum2)
    print(sum3)

# 7) Retrieve coefficients from quadratic fits, and write them to output file

#   Z1: Coefficients, associated errors, and R^2 for Z1 = a0 + a1*t + a2*t^2
    a0 <- (fitZ1)$coeff[[1]]
    a1 <- (fitZ1)$coeff[[2]]
    a2 <- (fitZ1)$coeff[[3]]
    a3 <- (fitZ1)$coeff[[4]]
    a4 <- (fitZ1)$coeff[[5]]
    a5 <- (fitZ1)$coeff[[6]]
    a6 <- (fitZ1)$coeff[[7]]
    a7 <- (fitZ1)$coeff[[8]]
    a8 <- (fitZ1)$coeff[[9]]
    a9 <- (fitZ1)$coeff[[10]]
    ae0 <- sum1$coefficients[[1,2]]      #Std. Error of a0
    ae1 <- sum1$coefficients[[2,2]]      #Std. Error of a1
    ae2 <- sum1$coefficients[[3,2]]      #Std. Error of a2
    ae3 <- sum1$coefficients[[4,2]]      #Std. Error of a3
    ae4 <- sum1$coefficients[[5,2]]      #Std. Error of a4
    ae5 <- sum1$coefficients[[6,2]]      #Std. Error of a5
    ae6 <- sum1$coefficients[[7,2]]      #Std. Error of a6
    ae7 <- sum1$coefficients[[8,2]]      #Std. Error of a7
    ae8 <- sum1$coefficients[[9,2]]      #Std. Error of a8
    ae9 <- sum1$coefficients[[10,2]]     #Std. Error of a9
    aR2 <- sum1$adj.r.squared            #Adjusted R^2

#   Z2: Coefficients, associated errors, and R^2 for Z2 = a0 + a1*t + a2*t^2
    b0 <- (fitZ2)$coeff[[1]]
    b1 <- (fitZ2)$coeff[[2]]
    b2 <- (fitZ2)$coeff[[3]]
    b3 <- (fitZ2)$coeff[[4]]
    b4 <- (fitZ2)$coeff[[5]]
    b5 <- (fitZ2)$coeff[[6]]
    b6 <- (fitZ2)$coeff[[7]]
    b7 <- (fitZ2)$coeff[[8]]
    b8 <- (fitZ2)$coeff[[9]]
    b9 <- (fitZ2)$coeff[[10]]
    be0 <- sum2$coefficients[[1,2]]      #Std. Error of b0
    be1 <- sum2$coefficients[[2,2]]      #Std. Error of b1
    be2 <- sum2$coefficients[[3,2]]      #Std. Error of b2
    be3 <- sum2$coefficients[[4,2]]      #Std. Error of b3
    be4 <- sum2$coefficients[[5,2]]      #Std. Error of b4
    be5 <- sum2$coefficients[[6,2]]      #Std. Error of b5
    be6 <- sum2$coefficients[[7,2]]      #Std. Error of b6
    be7 <- sum2$coefficients[[8,2]]      #Std. Error of b7
    be8 <- sum2$coefficients[[9,2]]      #Std. Error of b8
    be9 <- sum2$coefficients[[10,2]]     #Std. Error of b9
    bR2 <- sum2$adj.r.squared            #Adjusted R^2
   
#   Z3: Coefficients, associated errors, and R^2 for Z3 = a0 + a1*t + a2*t^2
    c0 <- (fitZ3)$coeff[[1]]
    c1 <- (fitZ3)$coeff[[2]]
    c2 <- (fitZ3)$coeff[[3]]
    c3 <- (fitZ3)$coeff[[4]]
    c4 <- (fitZ3)$coeff[[5]]
    c5 <- (fitZ3)$coeff[[6]]
    c6 <- (fitZ3)$coeff[[7]]
    c7 <- (fitZ3)$coeff[[8]]
    c8 <- (fitZ3)$coeff[[9]]
    c9 <- (fitZ3)$coeff[[10]]
    ce0 <- sum3$coefficients[[1,2]]      #Std. Error of c0
    ce1 <- sum3$coefficients[[2,2]]      #Std. Error of c1
    ce2 <- sum3$coefficients[[3,2]]      #Std. Error of c2
    ce3 <- sum3$coefficients[[4,2]]      #Std. Error of c3
    ce4 <- sum3$coefficients[[5,2]]      #Std. Error of c4
    ce5 <- sum3$coefficients[[6,2]]      #Std. Error of c5
    ce6 <- sum3$coefficients[[7,2]]      #Std. Error of c6
    ce7 <- sum3$coefficients[[8,2]]      #Std. Error of c7
    ce8 <- sum3$coefficients[[9,2]]      #Std. Error of c8
    ce9 <- sum3$coefficients[[10,2]]     #Std. Error of c9
    cR2 <- sum3$adj.r.squared            #Adjusted R^2
   
#   all <- c(a0, ae0, a1, ae1, a2, ae2, aR2, b0, be0, b1, be1, b2, be2, bR2, "\n")
#   all <- data.frame(all)
#   write.table(all, file="coeffs.dat", append = TRUE)
#   cat(all, file="coeffs.dat", append = TRUE)

#   Print out all coefficients in 1 file, including R2 values as well/df
    Z1coefs <- c("Z1:", a0,"+/-",ae0, a1,"+/-",ae1, a2,"+/-",ae2, a3,"+/-",ae3, a4,"+/-",ae4, a5,"+/-",ae5, a6,"+/-",ae6, a7,"+/-",ae7, a8,"+/-",ae8, a9,"+/-", ae9, aR2, "\n")
    Z2coefs <- c("Z2:", b0,"+/-",be0, b1,"+/-",be1, b2,"+/-",be2, b3,"+/-",be3, b4,"+/-",be4, b5,"+/-",be5, b6,"+/-",be6, b7,"+/-",be7, b8,"+/-",be8, b9,"+/-", be9, bR2, "\n")
    Z3coefs <- c("Z3:", c0,"+/-",ce0, c1,"+/-",ce1, c2,"+/-",ce2, c3,"+/-",ce3, c4,"+/-",ce4, c5,"+/-",ce5, c6,"+/-",ce6, c7,"+/-",ce7, c8,"+/-",ce8, c9,"+/-", ce9, cR2, "\n")
#   allcoefs <- c(Z0coefs, Z1coefs)
    allcoefs <- c(Z1coefs, Z2coefs, Z3coefs)
    cat(allcoefs, file="coeffs_salt.dat", append = TRUE)

# 8) Verify on Fitted vs. True results
    x1 <- a0 + a1*t + a2*s + a3*t2 + a4*s2 + a5*t3 + a6*s3 + a7*ts + a8*t2s + a9*ts2
    x2 <- b0 + b1*t + b2*s + b3*t2 + b4*s2 + b5*t3 + b6*s3 + b7*ts + b8*t2s + b9*ts2
    x3 <- c0 + c1*t + c2*s + c3*t2 + c4*s2 + c5*t3 + c6*s3 + c7*ts + c8*t2s + c9*ts2

 #  ddic = 150
 #  carbd <- carb(flag=15, var1=TA, var2=dic0+ddic*1e-6, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
 #  dpCO2true <- carbd$pCO2 - pCO2o

    dpco2a <- 280
    carbd <- carb(flag=24, var1=pCO2o+dpco2a, var2=TA, S=Salt, T=Temp, P=0, Pt=0, Sit=0, k1k2="l", kf="dr", ks="d", pHscale="T")
    dpCO2true <- carbd$pCO2 - pCO2o
    dDICtrue <- carbd$DIC - dic0
    ddic <- dDICtrue * 1e+6
    ddic2 <- ddic*ddic
    ddic3 <- ddic*ddic2
#   dpCO2fit <- (x0 * ddic) / (1 - x1*ddic)    #This solution only for linear relationship in Sarmiento et al. (1992)
    dpCO2fit <- x1*ddic + x2*ddic2 + x3*ddic3  #This study
#   plot(x=Temp, y=dpCO2true)
#   lines(x=Temp, y=dpCO2fit, col="blue")
#   lines(x=Temp, y=x1*ddic, col="red")

#   Rough conclusion: relative errors always less than 0.5% when delpCO2 < 280 (i.e., atm pCO2 ~ 560 ppm)
    plot(x=Temp, y=100*(dpCO2fit - dpCO2true)/dpCO2true, col="blue")

#   Make contour plots of Z0 and Z1g
#   Z0p <- Z0 ; Z1p <- Z1
#   dim(Z0p) <- c(ntemp, nsalt) ; dim(Z1p) <- c(ntemp, nsalt)
#   contour(temp, salt, Z0p)
#   contour(temp, salt, Z1p)

#   Save out date to seek best statistical model for z1 = f(T, S) ; z2 = f(T, S) ;  z3 = f(T, S)
    Z1f <- data.frame(Temp, Salt, Z1)
    Z2f <- data.frame(Temp, Salt, Z2)
    Z3f <- data.frame(Temp, Salt, Z3)
    write.table(Z1f, file="z1.dat", row.names = FALSE)
    write.table(Z2f, file="z2.dat", row.names = FALSE)
    write.table(Z3f, file="z3.dat", row.names = FALSE)

    Z1coef <- sum1$coeff[,1]
    Z2coef <- sum2$coeff[,1]
    Z3coef <- sum3$coeff[,1]
    Zcoef  <- data.frame(Z1coef, Z2coef, Z3coef)
    write.table(Zcoef, file="Zcoef.txt")

#   Run tests to see significance of each of the terms in a Cubic equation with crossed terms
    Z1f2 <- Z1f^2
    Z1f <- data.frame(Z1f, Z1f^2)
    Z1f$Z1.1 <- NULL
    names(Z1f) <- c("t", "s", "Z1", "t2", "s2")
    fitZ1.lm  <- lm(Z1 ~ (.)^2, data=Z1f)
    fitZ1.glm <- glm(Z1 ~ (.)^2, data=Z1f)
    fitZ1.glm.step <- step(fitZ1.glm,direction="backward")
    s.lm <- summary(fitZ1.lm) ; s.gm <- summary(fitZ1.glm)
#   s.lm
#   s.gm
    anova(fitZ1.glm.step,test="F")

    Z2f2 <- Z2f^2
    Z2f <- data.frame(Z2f, Z2f^2)
    Z2f$Z2.1 <- NULL
    names(Z2f) <- c("t", "s", "Z2", "t2", "s2")
    fitZ2.lm  <- lm(Z2 ~ (.)^2, data=Z2f)
    fitZ2.glm <- glm(Z2 ~ (.)^2, data=Z2f)
    fitZ2.glm.step <- step(fitZ2.glm,direction="backward")
    s.lm <- summary(fitZ2.lm) ; s.gm <- summary(fitZ2.glm)
    s.lm
    s.gm
    anova(fitZ2.glm.step,test="F")

    Z3f2 <- Z3f^2
    Z3f <- data.frame(Z3f, Z3f^2)
    Z3f$Z3.1 <- NULL
    names(Z3f) <- c("t", "s", "Z3", "t2", "s2")
    fitZ3.lm  <- lm(Z3 ~ (.)^2, data=Z3f)
    fitZ3.glm <- glm(Z3 ~ (.)^2, data=Z3f)
    fitZ3.glm.step <- step(fitZ3.glm,direction="backward")
    s.lm <- summary(fitZ3.lm) ; s.gm <- summary(fitZ3.glm)
    s.lm
    s.gm
    anova(fitZ3.glm.step,test="F")

#   Make contour plot of predicted & true delpCO2
#   dptrue <- dpCO2true
#   dpfit <- dpCO2fit
#   dim(dptrue) <- c(ntemp, nsalt)
#   dim(dpfit) <- c(ntemp, nsalt)
#   contour(temp,salt, dpfit, col="red")
#   contour(temp,salt, dptrue, col="black", add = TRUE)
