# This script uses a lot of code from the packages FD and STEPCAM
# Please cite them correctly
# citation("FD")
# citation("STEPCAM")

# The code here is consequently distributed under the same license: GPL-2
# See https://cran.r-project.org/web/licenses/GPL-2

# Reduce the overload of the dbFD function
# Separate into two functions: one to calculate the ordination axes. 
# And the second function uses this axes to calculate FRic, FDiv, and FEve.
# Calculate the ordination axes only once and reuse them
ordinationAxes <- function(x, corr = c("sqrt", "cailliez", "lingoes", "none"), 
                           ord = c("podani", "metric"), w, 
                           asym.bin = NULL, messages = FALSE, stand.x = TRUE){
  tol <- .Machine$double.eps
  corr <- match.arg(corr)
  ord <- match.arg(ord)
  if (is.matrix(x) | is.data.frame(x)) {
    is.dist.x <- FALSE
    s.x <- nrow(x)
    t.x <- ncol(x)
    if (is.null(row.names(x))) 
      stop("'x' must have row names.", "\n")
    else x.rn <- row.names(x)
  }
  if (is.vector(x) | is.factor(x)) {
    is.dist.x <- FALSE
    s.x <- length(x)
    t.x <- 1
    if (is.null(names(x))) 
      stop("'x' must have names.", "\n")
    else x.rn <- names(x)
  }
  if (class(x)[1] == "dist" | class(x)[1] == "dissimilarity") {
    is.dist.x <- TRUE
    s.x <- attr(x, "Size")
    t.x <- 1
    if (is.null(attr(x, "Labels"))) 
      stop("'x' must have labels.", "\n")
    else x.rn <- attr(x, "Labels")
  }
  if (missing(w)) 
    w <- rep(1, t.x)/sum(rep(1, t.x))
  if (is.matrix(x) | is.data.frame(x)) {
    x <- data.frame(x)
    if (t.x >= 2) {
      x.class <- sapply(x, data.class)
      if (any(x.class == "character")) 
        x[, x.class == "character"] <- as.factor(x[, x.class == "character"])
      else x <- x
      if (all(x.class == "numeric") & all(!is.na(x))) {
        if (length(unique(w)) == 1) {
          x.s <- apply(x, 2, scale, center = TRUE, scale = stand.x)
          x.dist <- dist(x.s)
        }
        else {
          x.dist <- gowdis(x, w = w, ord = ord, asym.bin = asym.bin)
        }
      }
      else {
        x.dist <- gowdis(x, w = w, ord = ord, asym.bin = asym.bin)
      }
    }
    if (t.x == 1) {
      if (is.numeric(x[, 1])) {
        if (all(!is.na(x))) {
          x.s <- apply(x, 2, scale, center = TRUE, scale = stand.x)
          x.dist <- dist(x.s)
        }
        if (any(is.na(x))) {
          pos.NA <- which(is.na(x), arr.ind = TRUE)
          x <- na.omit(x)
          x.s <- apply(x, 2, scale, center = TRUE, scale = stand.x)
          x.dist <- dist(x.s)
          row.excl.ab <- pos.NA[, 1]
          a <- a[, -row.excl.ab]
          if (messages) 
            cat("Warning: Species with missing trait values have been excluded.", 
                "\n")
        }
      }
      if (is.factor(x[, 1]) | is.character(x[, 1])) {
        if (is.ordered(x[, 1])) 
          x <- x
        else x[, 1] <- as.factor(x[, 1])
        if (any(is.na(x))) {
          pos.NA <- which(is.na(x), arr.ind = TRUE)
          x <- na.omit(x)
          row.excl.ab <- pos.NA[, 1]
          a <- a[, -row.excl.ab]
          x.rn <- x.rn[-pos.NA]
          if (messages) 
            cat("Warning: Species with missing trait values have been excluded.", 
                "\n")
        }
        if (is.ordered(x[, 1])) {
          x.s <- data.frame(rank(x[, 1]))
          names(x.s) <- x.rn
          x.dist <- dist(x.s)
        }
        else {
          x.f <- as.factor(x[, 1])
          x.dummy <- diag(nlevels(x.f))[x.f, ]
          x.dummy.df <- data.frame(x.dummy, row.names = x.rn)
          sequence <- 1:10
          if (all(dist.bin != sequence[any(sequence)])) 
            stop("'dist.bin' must be an integer between 1 and 10.", 
                 "\n")
          x.dist <- dist.binary(x.dummy.df, method = dist.bin)
        }
      }
    }
  }
  if (is.vector(x) & is.numeric(x)) {
    if (any(is.na(x))) {
      pos.NA <- which(is.na(x))
      x <- na.omit(x)
      a <- a[, -pos.NA]
      x.rn <- x.rn[-pos.NA]
      if (messages) 
        cat("Warning: Species with missing trait values have been excluded.", 
            "\n")
    }
    else x <- x
    x.s <- scale(x, center = T, scale = stand.x)
    x.dist <- dist(x.s)
    x <- data.frame(x)
    dimnames(x) <- list(x.rn, "Trait")
  }
  if (is.vector(x) & is.character(x)) {
    x <- as.factor(x)
    if (any(is.na(x))) {
      pos.NA <- which(is.na(x))
      x <- na.omit(x)
      a <- a[, -pos.NA]
      x.rn <- x.rn[-pos.NA]
      if (messages) 
        cat("Warning: Species with missing trait values have been excluded.", 
            "\n")
    }
    else x <- x
    dimnames(x) <- list(x.rn, "Trait")
    x.dummy <- diag(nlevels(x))[x, ]
    x.dummy.df <- data.frame(x.dummy, row.names = x.rn)
    sequence <- 1:10
    if (all(dist.bin != sequence[any(sequence)])) 
      stop("'dist.bin' must be an integer between 1 and 10.", 
           "\n")
    x <- data.frame(x)
    x.dist <- dist.binary(x.dummy.df, method = dist.bin)
  }
  if (is.ordered(x)) {
    if (any(is.na(x))) {
      pos.NA <- which(is.na(x))
      x <- na.omit(x)
      a <- a[, -pos.NA]
      x.rn <- x.rn[-pos.NA]
      cat("Warning: Species with missing trait values have been excluded.", 
          "\n")
    }
    else x <- x
    x <- data.frame(x)
    dimnames(x) <- list(x.rn, "Trait")
    x.dist <- gowdis(x, w = w, ord = ord, asym.bin = asym.bin)
  }
  if (is.factor(x) & !is.ordered(x)) {
    if (any(is.na(x))) {
      pos.NA <- which(is.na(x))
      x <- na.omit(x)
      a <- a[, -pos.NA]
      x.rn <- x.rn[-pos.NA]
      if (messages) 
        cat("Warning: Species with missing trait values have been excluded.", 
            "\n")
    }
    else x <- x
    x.dummy <- diag(nlevels(x))[x, ]
    x.dummy.df <- data.frame(x.dummy, row.names = x.rn)
    sequence <- 1:10
    if (all(dist.bin != sequence[any(sequence)])) 
      stop("'dist.bin' must be an integer between 1 and 10.", 
           "\n")
    x.dist <- dist.binary(x.dummy.df, method = dist.bin)
    x <- data.frame(x)
    dimnames(x) <- list(x.rn, "Trait")
  }
  if (class(x)[1] == "dist" | class(x)[1] == "dissimilarity") {
    if (any(is.na(x))) 
      stop("When 'x' is a distance matrix, it cannot have missing values (NA).", 
           "\n")
    x.dist <- x
  }
  if (any(is.na(x.dist))) 
    stop("NA's in the distance matrix.", "\n")
  if (!is.dist.x) {
    no.traits <- apply(x, 1, function(v) length(v[!is.na(v)]))
    if (any(no.traits == 0)) 
      stop("At least one species has no trait data.", "\n")
  } 
  attr(x.dist, "Labels") <- x.rn
  if (is.euclid(x.dist)) 
    x.dist2 <- x.dist
  if (!is.euclid(x.dist)) {
    if (corr == "lingoes") {
      x.dist2 <- lingoes(x.dist)
      if (messages) 
        cat("Species x species distance matrix was not Euclidean. Lingoes correction was applied.", 
            "\n")
    }
    if (corr == "cailliez") {
      x.dist2 <- cailliez(x.dist)
      if (messages) 
        cat("Species x species distance matrix was not Euclidean. Cailliez correction was applied.", 
            "\n")
    }
    if (corr == "sqrt") {
      x.dist2 <- sqrt(x.dist)
      if (!is.euclid(x.dist2)) 
        stop("Species x species distance matrix was still is not Euclidean after 'sqrt' correction. Use another correction method.", 
             "\n")
      if (is.euclid(x.dist2)) 
        if (messages) 
          cat("Species x species distance matrix was not Euclidean. 'sqrt' correction was applied.", 
              "\n")
    }
    if (corr == "none") {
      x.dist2 <- quasieuclid(x.dist)
      if (messages) 
        cat("Species x species distance was not Euclidean, but no correction was applied. Only the PCoA axes with positive eigenvalues were kept.", 
            "\n")
    }
  }
  x.pco <- dudi.pco(x.dist2, scannf = FALSE, full = TRUE)
  return(x.pco) # Best is to return the whole object, because dbFD needs sometimes more than the axes alone?!
}


# Function to dermine the number of PCoA axes used and number of species. 
# (Both constant in STEPCAM and therefor only camculated once.)
detMnbsp <- function(x.pco, a){
  tol <- .Machine$double.eps
  a <- as.matrix(a)
  c <- nrow(a) # Number of communities
  traits <- x.pco$li 
  # Number of species per community where traits are present - do I need it later? Yes!
  nb.sp <- numeric(c)
  for (i in 1:c) {
    sp.pres <- which(a[i, ] > 0)
    traits.sp.pres <- traits[sp.pres, , drop = F]
    traits.sp.pres[traits.sp.pres != 0 & abs(traits.sp.pres) < tol] <- 0
    nb.sp[i] <- nrow(unique(traits.sp.pres))
  }
  min.nb.sp <- min(nb.sp)
  m.max <- min.nb.sp - 1 # Minimum number of species in one of the communities - 1
  #Warning <- FALSE
  if (min.nb.sp < 3) {
    nb.sp2 <- nb.sp[nb.sp > 2]
    m.max <- min(nb.sp2) - 1    
  }
  else {
    m.max <- m.max
  }
  m <- m.max
  Res <- list()
  Res[[1]] <- m
  Res[[2]] <- nb.sp
  return(Res)  
}

# Strip away radically everything except FRic, FDiv, and FEve
# Because this function needs to be much much faster!
# All communities are run in a for loop which cannot be vectorized
strippedDbFd <- function(x.pco, a, m, nb.sp){
  #tol <- .Machine$double.eps
  a <- as.matrix(a)
  c <- nrow(a) # Number of communities
  traits <- x.pco$li 
  Warning <- FALSE
  if (m < x.pco$nf){ # If there is a community with less species than ordination axes
    traits.FRic <- x.pco$li[, 1:m] # Use only the first m axes
  }
  if (m >= x.pco$nf){
    traits.FRic <- x.pco$li
  }
  FRic <- rep(NA, c)
  names(FRic) <- row.names(a)
  FEve <- FRic
  FDiv <- FRic
  #AbundRel <- a/rowSums(a)
  
  
  for (i in 1:c) { # For each community
    sppres <- which(a[i, ] > 0) # Present species
    S <- length(sppres)
    tr <- data.frame(traits[sppres, ]) # Axes coordinates of the present species
    tr.FRic <- data.frame(traits.FRic[sppres, ])
    # Will I need relative abundances of the species?
    ab <- as.matrix(a[i, sppres])
    abundrel <- ab/sum(ab)
    abund2 <- sapply( c(abundrel), function(x) x + abundrel)          
    abund2vect <- as.dist(abund2)
    
    # New part: check range of axes values, because if range is very small, convhulln will fail
    #apply(tr.FRic, 2, range)
    
    if (ncol(tr.FRic) > 1 & nb.sp[i] >= 3) { # If there are more than 3 species present
      if (Warning) 
        thresh <- 4
      if (!Warning) 
        thresh <- 3
      if (nb.sp[i] >= thresh) {
        # Option QJ is helpfull in case of planar hulls, Pp removes warning
        convhull <- convhulln(tr.FRic, c("QJ", "FA", "Pp")) 
        FRic[i] <- convhull$vol
      }
    }
    if (ncol(tr.FRic) == 1) {
      tr.range <- range(tr.FRic[, 1])
      t.range <- tr.range[2] - tr.range[1]
      FRic[i] <- t.range
    }
    
    if (nb.sp[i] >= 3) {
      tr.dist <- dist(tr) # pair-wise distance of ordination coordinates
      linkmst <- mst(tr.dist)
      mstvect <- as.dist(linkmst)
      #abund2 <- matrix(0, nrow = S, ncol = S)
      #for (q in 1:S) for (r in 1:S) abund2[q, r] <- abundrel[q] + abundrel[r]
      #  the *apply family is faster than the original code with more than three species
      # Move this outside of the loop 'cause its always the same:
      #abund2 <- sapply( c(abundrel), function(x) x + abundrel)          
      #abund2vect <- as.dist(abund2)
      #EW <- rep(0, S - 1)
      #flag <- 1
      #for (m in 1: ((S - 1) * S/2) ) {
      #  if (mstvect[m] != 0) {
      #    EW[flag] <- tr.dist[m]/(abund2vect[m])
      #    flag <- flag + 1
      #  }
      #}
      # Faster:
      EW <- c((tr.dist * mstvect) / abund2vect)
      EW <- EW[EW > 0]
      
      minPEW <- rep(0, S - 1)
      OdSmO <- 1/(S - 1)
      for (l in 1:(S - 1)) minPEW[l] <- min( (EW[l]/sum(EW)), OdSmO)
      # Slower:
      #sapply( EW/sum(EW), function(x) min( x, OdSmO ))      
      FEve[i] <- ((sum(minPEW)) - OdSmO)/(1 - OdSmO)
    }
    if (ncol(tr.FRic) > 1 & nb.sp[i] >= 3) {
      # Option QJ is helpfull in case of planar hulls, Pp removes warning
      vert0 <- convhulln(tr.FRic, c("Fx TO 'vert.txt'", "QJ", "Pp"))
      vert1 <- scan("vert.txt", quiet = T)
      vert2 <- vert1 + 1
      vertices <- vert2[-1]
      trvertices <- tr.FRic[vertices, ]
      baryv <- colMeans(trvertices) #apply(trvertices, 2, mean)
      #Faster:
      #distbaryv <- rep(0, S)
      #for (j in 1:S) distbaryv[j] <- ( sum( (tr.FRic[j, ] - baryv)^2) )^0.5      
      distbaryv <- sqrt( rowSums( (tr.FRic - baryv)^2) )
      
      meandB <- mean(distbaryv)
      devdB <- distbaryv - meandB
      abdev2 <- abundrel * devdB
      ababsdev2 <- abundrel * abs(devdB)
      FDiv[i] <- (sum(abdev2) + meandB)/(sum(ababsdev2) + meandB)
    }    
  }
  res <- list()
  res$FRic <- FRic
  res$FEve <- FEve
  res$FDiv <- FDiv
  return(res)  
}

# Max possible interpoint distance for S points (for FRicMax)
# Axes: Ordination axes
# S: Species richness of focal community
maxDistPoints <- function(Axes, S){ 
  # Code heavily borrowed by spacedman:
  # http://stackoverflow.com/questions/22152482/choose-n-most-distant-points-in-r
  subset <- Axes  
  alldist <- as.matrix(dist(subset))   
  while (nrow(subset) > S) {
    cdists <- rowSums(alldist)
    closest <- which(cdists == min(cdists))[1]
    subset <- subset[-closest,]
    alldist <- alldist[-closest,-closest]
  }
  return(subset)
}

# PREPARATION
# vectorized version should be faster than the original loop version!
# target = changed and will be always one value only 
# For 100 particles my version lasts only 3 % of the original running time!
myCalculateWeight <- function(params, target, sigM, 
                              dispVals, filtVals, compVals, 
                              numParticles, W){
  diff <- params[target] - cbind(dispVals, filtVals, compVals)[,target]
  vals <- W * dnorm(diff, mean = 0, sd = sigM)  
  return( 1/sum(vals) )
}

# Dispersal limitation
myFalloutDispersal2 <- function(new_community, dispersal_fallout, 
                                data_frequencies, IsDist){ 
  new_community_rows <- sample(nrow(new_community), dispersal_fallout,
                               # Removal chance inversely depends of frequency species
                               # Due to a lack of dispersal limitation, species that occur more frequently 
                               # in the species pool have a lower chance of being removed
                               prob = data_frequencies, 
                               replace = FALSE)
  if( IsDist ){
    new_community <- new_community[-new_community_rows,-new_community_rows]
  }
  else{ # Not a distance matrix
    new_community <- new_community[-new_community_rows,]
  }  
  return(new_community)
}

# Environmental filtering
myFalloutFiltering2 <- function(new_community, filtering_fallout, optimum, 
                                TraitsOptimumDist, IsDist){ 
  Sel <- c( charmatch(rownames(new_community), rownames(TraitsOptimumDist)), nrow(TraitsOptimumDist)) 
  # Species in new_community + optimum    
  distances_traits2 <- TraitsOptimumDist[Sel, Sel]
  distance_from_optimum <- distances_traits2[nrow(distances_traits2),] 
  distances_ordered <- rank( distance_from_optimum[-(nrow(new_community)+1)], ties.method = "random" )
  Remove <- which( distances_ordered > ( nrow(new_community) - filtering_fallout ) )
  if( IsDist ){
    new_community <- new_community[-Remove, -Remove]
  }
  else{
    new_community <- new_community[-Remove,]
  }
  return(new_community)
}

# Biotic interaction or limiting similarity
myFalloutCompetition2 <- function(new_community, competition_fallout, 
                                  SpeciesDist, IsDist){   
  for(y in 1:competition_fallout){ 
    Sel <- charmatch(rownames(new_community), rownames(SpeciesDist))
    trait_distances <- as.dist(SpeciesDist[Sel,Sel])
    m3 <- which(SpeciesDist[Sel,Sel] == min(trait_distances), arr.ind=TRUE) 
    a <- as.vector(m3[,1]) 
    La <- length(a)
    # What if two species pairs have the same distance? - Choose one randomly!
    if( La > 2){
      SampleMat <- matrix(1:La, nc = 2, byrow = TRUE)
      # Take randomly 1,2 or 3,4 or 5,6th values of a
      a <- a[ SampleMat[sample(La/2, 1),] ]
    }
    mina1 <- min(SpeciesDist[a[1], -a[2:1]]) 
    mina2 <- min(SpeciesDist[a[2], -a[2:1]])  
    # Can also be several times the same!
    min_overall <- which( c(mina1,mina2) == min(mina1,mina2) )  
    if(length(min_overall) > 1){
      min_overall <- sample(min_overall, 1)
    }
    species_out <- a[min_overall] 
    if( IsDist ){
      new_community <- new_community[-species_out, -species_out]
    }
    else{
      new_community <- new_community[-species_out, ]
    }
  }    
  return(new_community)
}

## the Kraft.generator function: function that runs (hybrid) STEPCAMs
mySTEPCAM2 <- function(params, species, TraitsOptimumDist, IsDist, taxa, data_frequencies, SpeciesDist){
  dispersal_fallout <- params[1] # species falling out through dispersal events
  filtering_fallout <- params[2] # species falling out through filtering events
  competition_fallout <- params[3] # species falling out through limiting similarity events
  
  if(is.na(dispersal_fallout)) dispersal_fallout <- 0  # just in case something went wrong
  if(is.na(competition_fallout)) competition_fallout <- 0
  if(is.na(filtering_fallout)) filtering_fallout <- 0
  
  new_community <- species # copy the traits
  if(dispersal_fallout > 0){ 
    new_community <- myFalloutDispersal2(new_community, dispersal_fallout, 
                                         data_frequencies, IsDist)
  }
  if(filtering_fallout > 0){ 
    new_community <- myFalloutFiltering2(new_community, filtering_fallout, 
                                         optimum, TraitsOptimumDist, IsDist)
  }
  if(competition_fallout > 0){      
    new_community <- myFalloutCompetition2(new_community, competition_fallout, SpeciesDist, IsDist)    
  }
  species_presences <- rep(0,taxa)
  species_presences[ charmatch(rownames(new_community), rownames(TraitsOptimumDist)) ] <- 1
  return(species_presences) # final modelled community
}

# This is the really time-consuming algorithm
myABC_SMC2 <- function(numParticles, species_fallout, taxa, esppres,  
                       summary_stats, community_number, species, abundances, 
                       data_frequencies, stopRate, Ord, 
                       IsDist, Axes, FRicMax, MaxCTMDis, Mnbsp, SD, ContinuePrevious){    
  # Removal chance inversely depends of frequency species
  data_frequencies <- 1/data_frequencies
  # Double check for infinite numbers (if a species is not present) 
  # and set them to 1 (high chance of removal)
  data_frequencies[!is.finite(data_frequencies)] <- 1
  # Observed CTM (either trait means or for distance-based STEPCAM axes ccordinates)
  optimum <- summary_stats[-c(1:3)] 
  dispVals <- 1:numParticles
  filtVals <- 1:numParticles
  compVals <- 1:numParticles
  fits <- 1:numParticles
  RichVec <- 1:numParticles
  EveVec <- 1:numParticles
  DivVec <- 1:numParticles
  OptVec <- 1:numParticles
  nextDisp <- dispVals
  nextFilt <- filtVals
  nextComp <- compVals
  weights <- rep(1, numParticles)
  nextWeights <- weights
  indices <- 1:numParticles
  sigma <- 1
  t <- 1
  f <- list.files(pattern = "particles_t=")
  # Add calculation of distance here because it is repeated millions of times during the fallout steps   
  if( !IsDist ){ # If no distance matrix
    # The distance among the mean of present traits (optimum) and all traits
    tr <- species[esppres,]  
    optimum <- colMeans(tr) 
    TraitsOptimum <- rbind(species, optimum)
    TraitsOptimumDist <- as.matrix(dist(TraitsOptimum))
    SpeciesDist <- as.matrix(dist(species)) 
    diag(SpeciesDist) <- NA
  }
  if( IsDist ){ # If distance matrix
    # The distance between the community centroid of the PCoA-axes and all species PCoA coordinates
    TraitsOptimum <- rbind(Axes, optimum)
    TraitsOptimumDist <- as.matrix(dist(TraitsOptimum))
    SpeciesDist <- as.matrix(species)
    diag(SpeciesDist) <- NA
    species <- as.matrix(species) 
  }
  # ABC steps
  #############
  if (length(f) > 0) { # Is there any previouse run?
    f <- mixedsort(f) # Sorting correctly eg t9 before t10 smart!
    t1 <- 1 + length(f)
    t <- t1
    d <- read.table(f[length(f)], header = F) # Read last run
    # Enable to continue a previous run with a smaller stopRate
    if(!ContinuePrevious){
      if (d[numParticles, 1] == numParticles) {
        d <- read.table(f[length(f) - 1], header = F)
        output <- list(Stoch = d[, 1], Filt = d[, 2], Comp = d[, 3])
        cat("Found previously finished run, loaded results from that run\n")
        flush.console()
        return(output)
      }
    }
    else{
      # In case of continueing a previous run, this step should be done only once
      d <- read.table(f[length(f) - 1], header = F)
      # Set FALSE to continue in the next while loop
      ContinuePrevious <- FALSE
      t <- length(f)
    }
    dispVals <- d[, 1] 
    filtVals <- d[, 2]
    compVals <- d[, 3]
    fits <- d[, 8]
    weights <- d[, 9]    
  }
  while (t < 50) { # Maximum 50 steps
    cat("\nGenerating Particles for iteration\t", t, "\n")
    cat("0--------25--------50--------75--------100\n")
    cat("*")
    flush.console()
    PRINT_FREQ <- 20
    numberAccepted <- 0
    if (t != 1) 
      weights <- weights/sum(weights) 
    threshold <- 200 * exp(-0.5 * t) # threshold changes with iterations
    stop_iteration <- 0
    changed <- 1
    tried <- 1
    while (numberAccepted < numParticles) {
      if (t == 1) {
        # This is really a crucial step -  generate random values for species that should be removed via 
        # dispersal, niche, or limiting similarity process !
        #####################################################
        params <- getRandomVals(species_fallout) # Random values for dispersal, niche & limiting similarity
      }
      else { # Use previous (t-1) values for dispersal, niche & limiting similarity
        params <- getFromPrevious(indices, weights, dispVals, 
                                  filtVals, compVals) # Mixes previous ABC run
        params <- perturb(params, sigma)
        changed <- params[4] 
        params <- params[1:3]
        if (sum(params) > species_fallout) {
          print("too much params after perturb!")
          flush.console()
          break
        }
      }
      # Here species are going to be removed according to random values for dispersal, niche & limiting similarity
      # Results in a vector of presence absence
      allcommunities <- mySTEPCAM2(params, species, TraitsOptimumDist, IsDist, taxa, data_frequencies, SpeciesDist)
      communities <- as.data.frame(t(allcommunities))
      present_species <- which(communities > 0)
      FDmod <- strippedDbFd(Ord, communities, m = Mnbsp[[1]], nb.sp = Mnbsp[[2]]) 
      FRic <- FDmod$FRic
      FEve <- FDmod$FEve
      FDiv <- FDmod$FDiv
      if( !IsDist ){
        TraitMeansMod <- colMeans(species[present_species, ], na.rm = TRUE) # Trait means of the modeled community
      }
      else{
        TraitMeansMod <- colMeans(Axes[present_species, ], na.rm = TRUE) # Axes means of the modeled community
      }
      # optimum are the observed and trait_means the modeled values
      optimum_plus_trait_means <- rbind(optimum, TraitMeansMod) 
      # Distance between observed optimum and modeled optimum
      mean_optimum <- dist(optimum_plus_trait_means)[1]   
      
      # Calculate fit
      IndFits <- c(abs(summary_stats[1] - FRic) / SD$FRic, 
                   abs(summary_stats[2] - FEve)  / SD$FEve,
                   abs(summary_stats[3] - FDiv) / SD$FDiv,
                   mean_optimum / SD$CTM )
      fit <- sum(IndFits)
      
      # If the single particle is below the threshold
      if (fit < threshold) {
        numberAccepted <- numberAccepted + 1
        nextDisp[numberAccepted] = params[1]
        nextFilt[numberAccepted] = params[2]
        nextComp[numberAccepted] = params[3]
        fits[numberAccepted] = fit
        RichVec[numberAccepted] = IndFits[1] 
        EveVec[numberAccepted] = IndFits[2]
        DivVec[numberAccepted] = IndFits[3]
        OptVec[numberAccepted] = IndFits[4]
        if (t == 1) {
          nextWeights[numberAccepted] = 1
        }
        else {
          # This only affects the next t
          nextWeights[numberAccepted] = myCalculateWeight(params, changed, sigma, 
                                                          dispVals, filtVals, compVals, 
                                                          numParticles, weights)
        }
        if( (numberAccepted) %% (numParticles/PRINT_FREQ) == 0 ) {
          cat("**")
          flush.console()
        }
      }
      tried <- tried + 1
      if (tried > (1/stopRate) && tried > 5) {
        if (numberAccepted/tried < stopRate) { 
          stop_iteration <- 1
          break
        }
      }
    }
    dispVals <- nextDisp
    filtVals <- nextFilt
    compVals <- nextComp
    weights <- nextWeights
    output <- cbind(dispVals, filtVals, compVals, RichVec, EveVec, DivVec, OptVec, fits, weights)
    file_name <- paste("particles_t=", t, ".txt", sep = "", collapse = NULL)
    write.table(output, file_name, row.names = F, col.names = F) # Why no column names?
    cat(" ", mean(dispVals), mean(filtVals), mean(compVals), 
        "\t", "accept rate = ", numberAccepted/(tried - 1), 
        numberAccepted, tried, "\n")
    nextWeights <- rep(1, numParticles)
    nextDisp <- 1:numParticles
    nextFilt <- 1:numParticles
    nextComp <- 1:numParticles
    t <- t + 1
    if (stop_iteration == 1) {
      break
    }
  }
  if (t >= 2) {
    d <- read.table(paste("particles_t=", t - 2, ".txt", sep = "", collapse = NULL), header = F)
  }
  else {
    if (t >= 1) {
      d <- read.table(paste("particles_t=", t - 1, ".txt", sep = "", collapse = NULL), header = F)
    }
    else {
      d <- read.table(paste("particles_t=", t, ".txt", sep = "", collapse = NULL), header = F)
    }
  }
  output <- list(Stoch = d[, 1], Filt = d[, 2], Comp = d[, 3])
  return(output)
}

# Version to get rid of the first column in data_species, because it is just the species names
# SD: Either FALSE or a number of random communities with the same number of species than the plot number
# SD will be used for standardizing the summary statistics (otherwise in a range of [0,1])
# This step may last long, more than 1000 is not recommanded
dbSTEPCAM_ABC <- function(data_abundances, data_species, numParticles, 
                          plot_number, stopRate, SD = 100, 
                          corr = "sqrt", ContinuePrevious = FALSE){ 
  # Check species names in data_abundances and data_species 
  # Order them equally and exclude missing species
  NamesDataAb <- colnames(data_abundances)
  NamesDataSp <- rownames(as.matrix(data_species))
  SpeciesNames <- sort(unique( NamesDataAb,  NamesDataSp))
  Pres <- data.frame( data_abundances = SpeciesNames %in% NamesDataAb, data_species = SpeciesNames %in% NamesDataSp, 
                      row.names = SpeciesNames)
  PresBoth <- rowSums(Pres)
  KeepSpecies <- row.names(Pres)[PresBoth == 2]
  DataSpMatch <- match(KeepSpecies, NamesDataSp) # The order and which species of data_species
  DataAbMatch <- match(KeepSpecies, NamesDataAb) # The order and which species of data_abundances
  data_abundances <- data_abundances[, DataAbMatch] # Order and exclude
  
  if( is.data.frame(data_species) | is.matrix(data_species) ){
    IsDist <- FALSE
    taxa <- nrow(data_species) # Total number of species
    data_species <- data_species[DataSpMatch,] # Order and exclude
  }
  else{
    IsDist <- TRUE
    Tmp <- as.matrix(data_species)
    Tmp <- Tmp[DataSpMatch, DataSpMatch]
    taxa <- nrow(Tmp) # Total number of species
    data_species <- as.dist(Tmp) # Order and exclude
  }
  n_plots <- nrow(data_abundances) # Number of plots 
  observed_abundances <- data_abundances[plot_number, ] # Focal community
  esppres <- which(observed_abundances > 0) # Which species is present?
  S <- length(esppres) # Number of present species  
  # How many species need to be omitted of the regional species pool in order to reach richness of the focal plot?
  species_fallout <- taxa - S 
  # How often is each species present in the regional pool?
  IncidenceMatrix <- ifelse(data_abundances > 0, 1, 0)
  data_frequencies <- colSums(IncidenceMatrix)  
  # Scale all indices to the interval [0, 1]. Than the total Fit will have bounds too [0, 4].
  # FEve and FDiv are anyway in [0, 1], 
  # FRic can be standardised by dividing by the whole functional volumne
  # If not distance matrix:
  if(!IsDist){
    data_species <- scale(data_species) # Scale traits to unit variance
    # Here all CTM values
    # N O T  Y E T  W I T H I N  [0,1] ! ! !
    # If SD is calculated based on simulation, then this two lines are not necessary any more:
    #optimum <- t(apply(IncidenceMatrix, 1, function(x) colMeans(data_species[x > 0, ], na.rm = TRUE) ))
    #TraitMeansObs <- optimum[plot_number,] # CTM of the focal community (in ABC_SMC it is calculated again!)
    TraitMeansObs <- colMeans(data_species[ IncidenceMatrix[plot_number,] > 0, ], na.rm = TRUE)
    #average_optimums <- colMeans(optimum, na.rm = FALSE) # Average optimum of a communities with all species
    #optimums_plus_average <- rbind(optimum, average_optimums)
    # Shouldn't this be sd instead?
    #mean_multi_trait_difference <- mean(as.matrix(dist(optimums_plus_average))[n_plots + 1, c(1:n_plots)])
    # PCoA of the traits (which are first converted to a distance matrix)
    Ord <- ordinationAxes(data_species, stand.x = FALSE, corr = corr)
    Mnbsp <- detMnbsp(Ord, IncidenceMatrix[plot_number, , drop = FALSE])    
    Axes <- Ord$li # If not distance matrix, should not be necessary
    # Maybe there are more species than axes:
    AxesThresh <- min(Mnbsp[[1]], ncol(Axes)) # Used for FRicMax
    # Given random communities with S species, 
    # what is the he maximum distance among all trait means that can be achieved?
    #############################################################################
    SpeciesDist <- as.matrix(dist(data_species))
    MaxCTMDis <- rep(0, ncol(SpeciesDist))
    for(i in 1:ncol(SpeciesDist)){
      # The population of possible CTM differences:
      # For each species, identify the S closest species
      # Also, determine its most distant species
      # For this most distant species, get the S closest species
      # For this two groups, calculate the distance between their column means
      TmpSp <- SpeciesDist[,i]
      Group1 <- order(TmpSp)[1:S]
      Group2 <- order(SpeciesDist[, which.max(TmpSp)])[1:S]
      MaxCTMDis[i] <- dist( rbind( colMeans(data_species[Group1,]), colMeans(data_species[Group2,]) ) )[1]
    }
    MaxCTMDis <- max(MaxCTMDis, na.rm = TRUE)
  }   
  if(IsDist){    
    # PCoA of the distance matrix
    Ord <- ordinationAxes(data_species, stand.x = FALSE)
    Axes <- as.matrix(Ord$li) # mean etc does not work with data.frame
    Mnbsp <- detMnbsp(Ord, IncidenceMatrix[plot_number, , drop = FALSE])
    # Maybe there are more species than axes:
    AxesThresh <- min(Mnbsp[[1]], ncol(Axes)) # Used for FRicMax
    # If SD is calculated based on simulation, then this two lines are not necessary any more:
    #optimum <- t(apply(IncidenceMatrix, 1, function(x) colMeans(Axes[x > 0,]))) # List of axes coordinates
    #TraitMeansObs <- optimum[plot_number,] # CTM of the focal community
    TraitMeansObs <- colMeans(Axes[ IncidenceMatrix[plot_number,] > 0, ], na.rm = TRUE)
    # Given random communities with S species, 
    # what is the he maximum distance among all community centroids that can be achieved?
    #####################################################################################
    AxesCombn <- combn(ncol(Axes[,1:AxesThresh]), 2)
    ExtSpecies <- unique(unlist(apply(AxesCombn, 2, function(x) chull(Axes[,x]))))
    # What is, if there are more extreme points than taxa in the focal community?
    # Brute force methode by excluding extreme points until observed richness is reached. 
    # For each possible way to exclude extreme, the maximum value that distance between observed 
    # and modeled centroid can achieve will be calculated
    AxesDist <- as.matrix(dist(Axes))
    Diff <- length(ExtSpecies) - S # Difference corners of ordination and present species richness
    if(Diff > 1){
      # May cause problems if there are many extreme of the functional space but little observed richness!
      Crit <- choose(length(ExtSpecies), S) # Calculate how many combination are possible
      if(Crit > 10000){ 
        # If there are more than 10000 possibilities to excluding extreme points until observed richness is reached,
        # than just a subsample should be used
        Y <- 10000
        Cn <- matrix(0, nrow =  Diff, ncol = Y)        
        for(z in 1:Y){
          Cn[,z] <- sample(ExtSpecies, Diff)
        }        
      }
      else{
        Cn <- combn(length(ExtSpecies), S) # Indices of extreme that are going to be kept
        Y <- ncol(Cn)
      }
    }
    else{
      Y <- 1
      Cn <- FALSE
    }
    MaxCTMDis <- rep(0, Y)
    for(y in 1:Y){
      ExtremeCentroids <- matrix(0, ncol = ncol(Axes), nrow = length(ExtSpecies))
      for(i in 1:length(ExtSpecies)){ # Use loop because it is done only once and its complex        
        if(is.matrix(Cn)){
          Extreme <- ExtSpecies[ -Cn[,y] ][i] 
        }
        else{
          Extreme <- ExtSpecies[i] 
        }
        # Exclude some of the extreme in order to have same richness than observed one
        TmpSpecies <- AxesDist[Extreme, ]
        CloseSpecies <- order(TmpSpecies)[1:S] # Which are the S closest species to that corner? (including itself)
        ExtremeCentroids[i,] <- colMeans(Axes[CloseSpecies, ]) # Centroid of this S species
      }
      # Maximum value that distance between observed and modeled centroid can achieve
      MaxCTMDis[y] <- max( dist(ExtremeCentroids) )     
    }
    MaxCTMDis <- max(MaxCTMDis, na.rm = TRUE) 
    # Should be always smaller than the maximum distance between individual species in the trait space
  }
  
  # Maxima of FRic?
  #########################################################################################################
  AxesCombn <- combn(ncol(Axes[,1:AxesThresh]), 2) 
  # Already calculated in IsDist block above - but how to avoid
  # But for this step, time does not matter too much!
  ExtSpecies <- unique(unlist(apply(AxesCombn, 2, function(x) chull(Axes[,x]))))
  Diff <- length(ExtSpecies) - S # Difference corners of ordination and present species richness
  if(Diff < 0){ # More or equal species present than extreme points
    FRicMax <- convhulln( Axes[ExtSpecies, 1:AxesThresh], c("QJ","FA", "Pp"))$vol
  }
  else{
    FRicMax <- convhulln( maxDistPoints(Axes[, 1:AxesThresh], S), c("QJ","FA", "Pp"))$vol # Can last a while
  }  
  #########################################################################################################
  
  # Infer standard deviation of the summary statistic based on a population of random possible values
  # Either generate random communities or, if numerical possible, use an exhaustive search
  # Depending on the number of SD, this step will last hours
  #########################################################################################################
  if(is.numeric(SD)){
    CritSD <- choose(taxa, S) # How many communities are posible?
    if(CritSD < SD){ # Exhaustive solution using all possibilities
      CombnSD <- combn(taxa, S)
      RanComm <- matrix(0, nrow = CritSD, ncol = taxa)
      for(i in 1:CritSD){
        RanComm[i,CombnSD[,i]] <- 1
      }      
    }
    else{
      RanComm <- matrix(0, nrow =  SD, ncol = taxa)
      colnames(RanComm) <- colnames(data_abundances) 
      for(s in 1:SD){         
        RanComm[s, sample(taxa,S)] <- 1
      }
    }
    # Store it as SD    
    RanFD <- strippedDbFd(Ord, RanComm, m = Mnbsp[[1]], nb.sp = rep(Mnbsp[[2]], SD))      
    # SD of CTM
    if(!IsDist){
      # Calculate pair-wise distances between centroids of all random communities 
      RanCTM <- matrix(0, nrow = SD, ncol = SD)
      for(i in 1:SD){
        RanCTM[,i] <- apply(RanComm, 1, function(x) dist( 
          rbind( colMeans(data_species[RanComm[i,]>0,])  , colMeans(data_species[x>0,])) 
        )[1] )
      }
      diag(RanCTM) <- NA # Remove comparisons between the same community      
    }
    if(IsDist){
      RanCTM <- matrix(0, nrow = SD, ncol = SD)
      for(i in 1:SD){
        RanCTM[,i] <- apply(RanComm, 1, function(x) dist( 
          rbind( colMeans(Axes[RanComm[i,]>0, 1:AxesThresh])  , colMeans(Axes[x>0, 1:AxesThresh])) 
        )[1] )
      }
      diag(RanCTM) <- NA # Remove comparisons between the same community     
    }
    SD <- lapply(RanFD, sd) # Calculate standard deviations
    SD$CTM <- sd(RanCTM, na.rm = TRUE)
  }
  #########################################################################################################
  
  # Observed FD
  FDobs <- strippedDbFd(Ord, IncidenceMatrix[plot_number, , drop = FALSE], m = Mnbsp[[1]], nb.sp = Mnbsp[[2]]) 
  
  # Functional indices & CTM, plus values to standardize differences in FRic and CTM in [0,1]
  summary_stats <- c(FDobs$FRic, FDobs$FEve, FDobs$FDiv, TraitMeansObs)
  
  # From here on will be slow because communities are going to be assembled thousands of times...
  output <- myABC_SMC2(numParticles, #
                       species_fallout, # Number of species that will be removed from the regional pool
                       taxa, # Total number of species
                       esppres, # Local community
                       summary_stats, # Functional indices & CTM
                       community_number = plot_number, # Number of the local community
                       species = data_species, # Scaled traits
                       abundances = data_abundances, # Whole community matrix
                       data_frequencies, # Frequency of presence of all species
                       stopRate, # Acceptence rate
                       Ord, # Ordination axes
                       IsDist, # Distance or trait matrix?
                       Axes, FRicMax, MaxCTMDis, Mnbsp, SD, ContinuePrevious) 
  return(output)
}


