%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Supplementary Material: Examples Script
%
% Description: provides examples of twin-test and using
% remineralisation rates to infer flux curves from the manuscript.
%
% To accompany Biogeosciences Manuscript: 'Can organic matter flux profiles
% be diagnosed using remineralisation rates derived from observed tracers
% and modelled ocean transport rates?'
%
% Loaded Variables...
%   - WORBE2_A:     GENIE Transport Matrix
%   - I:            sparse identity function
%   - v_index:      information for vectorising functions
%   - nb:           number of wet grid-boxes (vector size)
%   - Index_vars:   Indices for vectors
%       .Ii         Ocean Interior
%       .Ib         Ocean Surface
%       .Icol       Water Columns
%   - WORBE2_data:  GENIE synthetic tracer data (SYN)
%       .PO4        dissolved phosphate (mol kg-1)                
%       .PO4REMIN   total phosphate remineralisation (mol kg-1)
%       .SAL        salinity (psu)
%       .T          temperature (deg C)
%       .DOP        dissolved organic phosphorus (mol kg-1)
%       .CONV       convection cost function (unitless)
%       .M          grid-box mass (kg)
%
% Derived Varibles...
%   - WORBE2_ISS:   Interior source/sink results
%   - curves:       Output of fitting flux curves
%       .remin      Calculated Martin Curve Exponent
%       .r2         R2 of linear fit
%
%
% Jamie D. Wilson (wilsonjd@cardiff.ac.uk) 02/03/2015

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Twin-Test Example (section 4.1.1 of manuscript)
% load data
load('Supplementary_Matlab_Files.mat')

% calculate remin. rates for twin-test
WORBE2_ISS.PO4=-(WORBE2_A-I)*WORBE2_data.PO4;
WORBE2_ISS.SAL=-(WORBE2_A-I)*WORBE2_data.SAL;
WORBE2_ISS.T=-(WORBE2_A-I)*WORBE2_data.T;

% display twin-test results
% calculate and display errors
temp=(WORBE2_ISS.PO4(Index_vars.Ii)-WORBE2_data.PO4REMIN(Index_vars.Ii))./WORBE2_data.PO4REMIN(Index_vars.Ii);
temp(isinf(temp))=[];
fprintf('\n Mean proportion of error: %f +/- %f (1 s.d.) (mol kg-1) \n',...
    nanmean(temp),...
    nanstd(temp))
fprintf('\n Median proportion of error: %f (mol kg-1) \n',median(temp))
clear temp

% plotting
figure; subplot(2,1,1); hold on; box;
plot(WORBE2_data.PO4REMIN(Index_vars.Ii)*1e9,WORBE2_ISS.PO4(Index_vars.Ii)*1e9,'k.')
L1=line(sort(WORBE2_data.PO4REMIN(Index_vars.Ii)*1e9),sort(WORBE2_data.PO4REMIN(Index_vars.Ii)*1e9));
set(L1,'color',[.6 .6 .6])
set(gca,'YLim',get(gca,'XLim'))
xlabel({'Model Remineralisation','(nmol PO_{4} kg^{-1} dt^{-1})'});
ylabel({'Inverse ISS','(nmol PO_{4} kg^{-1} dt^{-1})'});

subplot(2,1,2); hold on; box;
hist(WORBE2_ISS.PO4(Index_vars.Ii)*1e9-WORBE2_data.PO4REMIN(Index_vars.Ii)*1e9,50);
xlabel({'Inverse ISS - Model Remin.','(nmol PO_{4} kg^{-1} dt^{-1})'});
ylabel('Frequency')
set(get(gca,'child'),'FaceColor',[.6 .6 .6])

%% Inferring Flux Curves from Remin. Rates (section 5.1 of manuscript)

% initialise arrays
curves.remin=zeros(934,1);
curves.r2=zeros(934,1);

% loop to calculate flux curve in each water column
for n=1:numel(Index_vars.Ib)
    
    xdata=genie_depth(Index_vars.Icol==n);
    ydata=WORBE2_data.PO4REMIN(Index_vars.Icol==n).*M(Index_vars.Icol==n);
    ztdata=genie_delta_depth(Index_vars.Icol==n);
    
    if numel(xdata)>5           % at least 3 data points (minus top and bottom)
        
        xdata=xdata(2:end-1); % remove surface and above-sediment boxes
        ydata=ydata(2:end-1);
        ztdata=ztdata(2:end-1);
        
        [B,BINT,R,RINT,STATS]=regress(log10(ydata./ztdata),[ones(numel(xdata),1),log10(xdata)]); % linear regression of log10(data)
        curves.remin(n,1)=B(2);
        curves.r2(n,1)=STATS(1);
        
    else 
        curves.remin(n,1)=0;
        curves.r2(n,1)=0;
    end
    
    clear xdata ydata ztdata
end
clear n

% plotting
figure;
C=zeros(nb,1);
C(Index_vars.Ib)=curves.remin;
v2f(C,v_index.i,v_index.j,v_index.rk);
imagesc(genie_lon,genie_lat,squeeze(ans(1,:,:)));
axis xy
caxis([-2.5 -1.5])
CB=colorbar;
ylabel(CB,'Martin Curve Exponent (b)')




