###########################################################################################
##    Interactions among temperature, moisture, and oxygen concentrations in 
##                   controlling decomposition rates.
#     R code to reproduce all results
###########################################################################################
# Code by: Carlos A. Sierra
# csierra@bgc-jena.mpg.de
###########################################################################################

# Load required R packages
library(SoilR)
library(FME)

# Set working directory
setwd("~/SOIL-R/Manuscripts/TWO_incubations")

#Read the data
IAll=read.csv("DataCode/IncubationsAll.csv") #Data from all incubation columns
co2=read.csv("DataCode/CO2incubations.csv") #Data aggregated by treatment

###########################################################################################
# 1. Comparison of treatment means

#Create a function to subset data by treatment combination
subst2=function(t,m,o){
  tmp1=subset(IAll,Temperature==t)
  tmp2=subset(tmp1,Moisture==m)
  tmp3=subset(tmp2,Oxygen==o,select=c(-Temperature,-Moisture,-Oxygen))
  return(tmp3)
}

clm=levels(IAll$SoilColumn) #Vector to identify soil columns by name

#Calculate cummulative respiration for the 35 day period for each column. First fits a spline and integrate area under the curve.
Resp=NULL
for(i in 1:length(clm)){
  set=subset(IAll,SoilColumn==clm[i])
  sp=splinefun(set$Day,set$CO2efflux)
  intg=integrate(sp,lower=1,upper=35)$value
  Ri=data.frame(Column=clm[i],Respiration=intg,Temperature=set[1,4],Moisture=set[1,5],Oxygen=set[1,6])
  Resp=rbind(Resp,Ri)
}

# Figure 1
#pdf(file="~/SOIL-R/Manuscripts/TWO_incubations/Figures/allbox.pdf")
par(mar=c(4,4,1,0.1))
boxplot(Respiration~Temperature*Moisture*Oxygen,data=Resp, 
        ylab=expression(paste("Total respired C",O[2],"-C (", mu, "g)")),xlab="Treatment (Temperature-Moisture)",cex.axis=0.6,
        names=rep(c("25-15","35-15","25-30","35-30","25-60","35-60","25-90","35-90"),2))
abline(v=8.5,lty=3)
legend(2,1600, "Low oxygen (1%)", bty="n")
legend(10,1600, "High oxygen (20%)", bty="n")
#dev.off()

# Comparison of treatment means (ANOVA)
summary(aov(lm(Respiration~Temperature*Moisture*Oxygen,data=Resp)))

#Other interesting plots not in publication
par(mfrow=c(2,2))
boxplot(Respiration~Moisture,data=Resp, 
        ylab="Total respiration",xlab=" ",cex.axis=0.5)

boxplot(Respiration~Temperature*Moisture,data=Resp, 
        ylab=" ",xlab=" ",cex.axis=0.5)

boxplot(Respiration~Moisture*Oxygen,data=Resp, 
        ylab="Total Respiration",xlab="Treatment",cex.axis=0.5)

boxplot(Respiration~Temperature*Oxygen,data=Resp, 
        ylab="",xlab="Treatment",cex.axis=0.5)
par(mfrow=c(1,1))

###########################################################################################
# 2. Optimization for each treatment

# Calculation of initial C stocks
soilweight=450 #grams
TOC=c(0.04683975, 0.04703255, 0.04687287) #proportion of organic C in a gram of soil sample. Values from 3 replicates
C0=soilweight*mean(TOC) #in grams of carbon

#Function to subset data from specific combinations of temperature, moisture, oxygen
subst=function(t,m,o){
  tmp1=subset(co2,Temperature==t)
  tmp2=subset(tmp1,Moisture==m)
  tmp3=subset(tmp2,Oxygen==o,select=c(Day,CO2efflux,sd))
  return(tmp3)
}

#Function to fit two-pool parallel model to data
TwopOptm=function(Temperature,Moisture,Oxygen, Cconc=NULL,
                  inipars=c(k1=1/15,k2=1/100,gamma=0.1),met="Marq",
                  up=c(1.5,1,1),low=c(0,0,0)){
  
  days=seq(0,max(co2$Day)) #Incubation days
  Obs=data.frame(Day=seq(1,35,1),
                 CO2efflux=cumsum(spline(subst(Temperature,Moisture,Oxygen)[,1],(subst(Temperature,Moisture,Oxygen)[,2]*1e-06*10),xout=seq(1,35,1))$y),
                 sd=(spline(subst(Temperature,Moisture,Oxygen)[,1],cumsum(subst(Temperature,Moisture,Oxygen)[,2]*1e-06*10),xout=seq(1,35,1))$y))
  
  # Create a function that takes parameter values for a 2-pool model
  # and returns the accumulated release of C from respiration
  eCO2func=function(pars){
    mod=TwopParallelModel(
      t=days,
      ks=pars[1:2],
      gam=0,
      C0=Cconc*c(pars[3], 1-pars[3]), 
      In=0,
      pass=TRUE
    )
    Rt=getAccumulatedRelease(mod)
    return(data.frame(Day=days,CO2efflux=rowSums(Rt)))
  }
  
  # Create a cost function that takes the data and the parameters and calculates residuals
  eCO2cost=function(pars){
    modelOutput=eCO2func(pars)
    return(modCost(model=modelOutput, obs=Obs, err="sd",x="Day"))
  }
  
  # Run the optimization algorithm
  eCO2fit=modFit(f=eCO2cost,p=inipars,method=met,
                 upper=up,lower=low)
  
  #Best parameter set
  best.fit=eCO2fit$par
  print(best.fit)
  
  #Run the model again with best parameter set
  #fitmod=eCO2func(eCO2fit$par)
  fitmod=TwopParallelModel(t=days, ks=best.fit[1:2], 
                           gam=0,
                           C0=Cconc*c(best.fit[3], 1-best.fit[3]), 
                           In=0)
  fitCum=getAccumulatedRelease(fitmod)
  fitC=getC(fitmod)
  fitR=getReleaseFlux(fitmod)
  
  modelOut=as.data.frame(cbind(days,fitC,fitR,fitCum))
  names(modelOut)<-c("days","C1","C2","R1","R2","Rc1","Rc2")
  
  matplot(days, rowSums(fitCum), type="l",lty=1,xlab="Days", ylab="Respired C [g C day-1]",ylim=c(0,max(Obs[,2])))
  points(Obs[,1],Obs[,2])
  arrows(Obs[,1],Obs[,2]-Obs[,3],Obs[,1],Obs[,2]+Obs[,3],angle=90,
         length=0.1,code=3)
  legend("topright",c("Observations", "Predictions"),lty=1,bty="n")
  
  #Run a Bayesian optimization to estimate uncertainty ranges
  var0=eCO2fit$var_ms_unweighted
  
  eCO2mcmc=modMCMC(f=eCO2cost, p=eCO2fit$par, niter=5000, jump=NULL,  
                   var0=NULL, lower=low,
                   upper=up,burninlength = 1000)
  
  #Look at the output
  mcmc.summary=summary(eCO2mcmc)
  optim.summary=rbind(best.fit,mcmc.summary)
  rownames(optim.summary)[1]<-"best"
    
  pairs(eCO2mcmc)
  
  # Plot uncertainty in predictions
  predRange=sensRange(func=eCO2func, parInput=eCO2mcmc$par)
   
  plot(summary(predRange),xlab="Days",
       ylab=expression(paste("Respired C",O[2]," Efflux (g C ",d^-1, ")")),main="")
  points(Obs[,1:2])
  arrows(Obs[,1],Obs[,2]-Obs[,3],Obs[,1],Obs[,2]+Obs[,3],angle=90,
         length=0.1,code=3)

  return(list(optim.summary,TWO=c(Temperature,Moisture,Oxygen)))
}

### Run optimizations separately:
f251520=TwopOptm(Temperature = 25, Moisture = 15, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f25151=TwopOptm(Temperature = 25, Moisture = 15, Oxygen = 1, Cconc = C0,
                inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f253020=TwopOptm(Temperature = 25, Moisture = 30, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=1/2,k2=1/10000,gamma=0.00001))

f25301=TwopOptm(Temperature = 25, Moisture = 30, Oxygen = 1, Cconc = C0,
                inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f256020=TwopOptm(Temperature = 25, Moisture = 60, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f25601=TwopOptm(Temperature = 25, Moisture = 60, Oxygen = 1, Cconc = C0,
                inipars=c(k1=0.2,k2=0.00001,gamma=0.5),up=c(0.3,0.001,1))

f259020=TwopOptm(Temperature = 25, Moisture = 90, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=0.2,k2=1/10000,gamma=0.5),up=c(0.3,0.001,1))

f25901=TwopOptm(Temperature = 25, Moisture = 90, Oxygen = 1, Cconc = C0,
                inipars=c(k1=1/5,k2=1/10,gamma=0.5))

f351520=TwopOptm(Temperature = 35, Moisture = 15, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=0.2,k2=1/10000,gamma=0.5),up=c(0.3,0.001,1))

f35151=TwopOptm(Temperature = 35, Moisture = 15, Oxygen = 1, Cconc = C0,
                inipars=c(k1=0.2,k2=1/10000,gamma=0.5),up=c(0.3,0.001,1))

f353020=TwopOptm(Temperature = 35, Moisture = 30, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f35301=TwopOptm(Temperature = 35, Moisture = 30, Oxygen = 1, Cconc = C0,
                inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f356020=TwopOptm(Temperature = 35, Moisture = 60, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=0.6,k2=1e-05,gamma=2e-04))

f35601=TwopOptm(Temperature = 35, Moisture = 60, Oxygen = 1, Cconc = C0,
                inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f359020=TwopOptm(Temperature = 35, Moisture = 90, Oxygen = 20, Cconc = C0,
                 inipars=c(k1=1/2,k2=1/10,gamma=0.5))

f35901=TwopOptm(Temperature = 35, Moisture = 90, Oxygen = 1, Cconc = C0,
                inipars=c(k1=0.2,k2=1/10000,gamma=0.5),up=c(0.3,0.001,1))

## Prepare output for plotting
Ws=c(15,30,60,90) # Moisture range
pick=2 # 1: classical optimization, 2: bayesian optimization

k1T25=c(f251520[[1]][pick,1],f253020[[1]][pick,1],f256020[[1]][pick,1],f259020[[1]][pick,1])
k1T35=c(f351520[[1]][pick,1],f353020[[1]][pick,1],f356020[[1]][pick,1],f359020[[1]][pick,1])
k1T25O1=c(f25151[[1]][pick,1],f25301[[1]][pick,1],f25601[[1]][pick,1],f25901[[1]][pick,1])
k1T35O1=c(f35151[[1]][pick,1],f35301[[1]][pick,1],f35601[[1]][pick,1],f35901[[1]][pick,1])

k2T25=c(f251520[[1]][pick,2],f253020[[1]][pick,2],f256020[[1]][pick,2],f259020[[1]][pick,2])
k2T35=c(f351520[[1]][pick,2],f353020[[1]][pick,2],f356020[[1]][pick,2],f359020[[1]][pick,2])
k2T25O1=c(f25151[[1]][pick,2],f25301[[1]][pick,2],f25601[[1]][pick,2],f25901[[1]][pick,2])
k2T35O1=c(f35151[[1]][pick,2],f35301[[1]][pick,2],f35601[[1]][pick,2],f35901[[1]][pick,2])

gT25=c(f251520[[1]][pick,3],f253020[[1]][pick,3],f256020[[1]][pick,3],f259020[[1]][pick,3])
gT35=c(f351520[[1]][pick,3],f353020[[1]][pick,3],f356020[[1]][pick,3],f359020[[1]][pick,3])
gT25O1=c(f25151[[1]][pick,3],f25301[[1]][pick,3],f25601[[1]][pick,3],f25901[[1]][pick,3])
gT35O1=c(f35151[[1]][pick,3],f35301[[1]][pick,3],f35601[[1]][pick,3],f35901[[1]][pick,3])

#q25 -> low
k1T25l=c(f251520[[1]][6,1],f253020[[1]][6,1],f256020[[1]][6,1],f259020[[1]][6,1])
k1T35l=c(f351520[[1]][6,1],f353020[[1]][6,1],f356020[[1]][6,1],f359020[[1]][6,1])
k1T25O1l=c(f25151[[1]][6,1],f25301[[1]][6,1],f25601[[1]][6,1],f25901[[1]][6,1])
k1T35O1l=c(f35151[[1]][6,1],f35301[[1]][6,1],f35601[[1]][6,1],f35901[[1]][6,1])

k2T25l=c(f251520[[1]][6,2],f253020[[1]][6,2],f256020[[1]][6,2],f259020[[1]][6,2])
k2T35l=c(f351520[[1]][6,2],f353020[[1]][6,2],f356020[[1]][6,2],f359020[[1]][6,2])
k2T25O1l=c(f25151[[1]][6,2],f25301[[1]][6,2],f25601[[1]][6,2],f25901[[1]][6,2])
k2T35O1l=c(f35151[[1]][6,2],f35301[[1]][6,2],f35601[[1]][6,2],f35901[[1]][6,2])

gT25l=c(f251520[[1]][6,3],f253020[[1]][6,3],f256020[[1]][6,3],f259020[[1]][6,3])
gT35l=c(f351520[[1]][6,3],f353020[[1]][6,3],f356020[[1]][6,3],f359020[[1]][6,3])
gT25O1l=c(f25151[[1]][6,3],f25301[[1]][6,3],f25601[[1]][6,3],f25901[[1]][6,3])
gT35O1l=c(f35151[[1]][6,3],f35301[[1]][6,3],f35601[[1]][6,3],f35901[[1]][6,3])

#q75 -> up
k1T25u=c(f251520[[1]][8,1],f253020[[1]][8,1],f256020[[1]][8,1],f259020[[1]][8,1])
k1T35u=c(f351520[[1]][8,1],f353020[[1]][8,1],f356020[[1]][8,1],f359020[[1]][8,1])
k1T25O1u=c(f25151[[1]][8,1],f25301[[1]][8,1],f25601[[1]][8,1],f25901[[1]][8,1])
k1T35O1u=c(f35151[[1]][8,1],f35301[[1]][8,1],f35601[[1]][8,1],f35901[[1]][8,1])

k2T25u=c(f251520[[1]][8,2],f253020[[1]][8,2],f256020[[1]][8,2],f259020[[1]][8,2])
k2T35u=c(f351520[[1]][8,2],f353020[[1]][8,2],f356020[[1]][8,2],f359020[[1]][8,2])
k2T25O1u=c(f25151[[1]][8,2],f25301[[1]][8,2],f25601[[1]][8,2],f25901[[1]][8,2])
k2T35O1u=c(f35151[[1]][8,2],f35301[[1]][8,2],f35601[[1]][8,2],f35901[[1]][8,2])

gT25u=c(f251520[[1]][8,3],f253020[[1]][8,3],f256020[[1]][8,3],f259020[[1]][8,3])
gT35u=c(f351520[[1]][8,3],f353020[[1]][8,3],f356020[[1]][8,3],f359020[[1]][8,3])
gT25O1u=c(f25151[[1]][8,3],f25301[[1]][8,3],f25601[[1]][8,3],f25901[[1]][8,3])
gT35O1u=c(f35151[[1]][8,3],f35301[[1]][8,3],f35601[[1]][8,3],f35901[[1]][8,3])

# Figure 2
#pdf(file="~/SOIL-R/Manuscripts/TWO_incubations/Figures/paramEstimation.pdf")
par(mfrow=c(3,2),mar=c(2,4,2,0.5))
plot(Ws,k1T25,type="b",ylim=c(0,0.8), main="20% oxygen", xlab="",ylab=expression(k[1]))
lines(Ws,k1T35,type="b",col=2)
arrows(Ws,k1T25l,Ws,k1T25u,angle=90,length=0.1,code=3)
arrows(Ws,k1T35l,Ws,k1T35u,angle=90,length=0.1,code=3,col=2)

plot(Ws,k1T25O1,type="b",ylim=c(0,0.8), main="1% oxygen", xlab="", ylab="")
lines(Ws,k1T35O1,type="b",col=2)
arrows(Ws,k1T25O1l,Ws,k1T25O1u,angle=90,length=0.1,code=3)
arrows(Ws,k1T35O1l,Ws,k1T35O1u,angle=90,length=0.1,code=3,col=2)
legend("topright",c("25 Celcius","35 Celcius"),lty=1,col=1:2,pch=1,bty="n")

plot(Ws,k2T25,type="b",ylim=c(0,.00003), xlab="",ylab=expression(k[2]))
lines(Ws,k2T35,type="b",col=2)
arrows(Ws,k2T25l,Ws,k2T25u,angle=90,length=0.1,code=3)
arrows(Ws,k2T35l,Ws,k2T35u,angle=90,length=0.1,code=3,col=2)

plot(Ws,k2T25O1,type="b",ylim=c(0,0.00003),xlab="",ylab="")
lines(Ws,k2T35O1,type="b",col=2)
arrows(Ws,k2T25O1l,Ws,k2T25O1u,angle=90,length=0.1,code=3)
arrows(Ws,k2T35O1l,Ws,k2T35O1u,angle=90,length=0.1,code=3,col=2)

par(mar=c(4,4,1,0.5))
plot(Ws,gT25,type="b",ylim=c(0,0.001),ylab=expression(gamma),xlab="WFPS")
lines(Ws,gT35,type="b",col=2)
arrows(Ws,gT25l,Ws,gT25u,angle=90,length=0.1,code=3)
arrows(Ws,gT35l,Ws,gT35u,angle=90,length=0.1,code=3,col=2)

plot(Ws,gT25O1,type="b",ylim=c(0,0.001),ylab="",xlab="WFPS")
lines(Ws,gT35O1,type="b",col=2)
arrows(Ws,gT25O1l,Ws,gT25O1u,angle=90,length=0.1,code=3)
arrows(Ws,gT35O1l,Ws,gT35O1u,angle=90,length=0.1,code=3,col=2)
par(mfrow=c(1,1))
#dev.off()

###########################################################################################
# 3. Optimization of all treatments simulteneously

#Create new objective function
objFunc<-function(pars,obs=co2){
  fT=fT.Q10(Temp=obs$Temperature, Q10=pars[2])
  fM=obs$Moisture/(pars[3]+obs$Moisture)
  fO=obs$Oxygen/(pars[4]+obs$Oxygen)
  
  Pred=C0*(pars[1]*(1-exp(-pars[5]*fT*fM*fO*obs$Day))+(1-pars[1])*(1-exp(-pars[6]*fT*fM*fO*obs$Day)))
  Obs=obs$CO2efflux
  Err=obs$sd
  Res=Pred-Obs
  res=Res/Err

  return(res)
}

#Classical optimization
inipars=c(gamma=0.001,Q10=1.4,Ks=3,Ko=0.01,k1=0.1,k2=0.0001)
best=modFit(f=objFunc,p=inipars,method="Pseudo",lower=c(0,0,1,0,0,0),upper=c(1,3,5,1,0.8,0.001))

#Bayesian optimization
fm=modMCMC(f=objFunc,p=c(coef(best)),var0=best$ms,niter=10000,burninlength = 2000,lower=c(0,0,1,0,0,0),upper=c(1,3,5,1,1,0.1))

#Figure 3
#pdf(file="~/SOIL-R/Manuscripts/TWO_incubations/Figures/pairs.pdf")
pairs(fm,pch=".",nsample=1000)
#dev.off()

## Extract optimized parameters and plot sensitivities with uncertainties
Q10m=summary(fm)[1,2]
Ksm=summary(fm)[1,3]
Kom=summary(fm)[1,4]

Temp=seq(0,40,1)
W=seq(0,90,0.01)
O=seq(0,20,0.001)

dfT=function(Temp, Q10){fT.Q10(Temp=Temp, Q10=Q10)}
dfW=function(W, Ks){W/(Ks+W)}
dfO=function(O, Ko){O/(Ko+O)}

dfTdT=function(Temp,Q10){fT.Q10(Temp=Temp, Q10=Q10)*(1/10)}
dfWdW=function(W,Ks){Ks/((Ks+W)^2)}
dfOdO=function(O,Ko){Ko/((Ko+O)^2)}


# Figure 4
pdf(file="~/SOIL-R/Manuscripts/TWO_incubations/Figures/sensitivities.pdf")
par(mfrow=c(3,2), mar=c(5,4,1,1))
plot(Temp,dfT(Temp,Q10m),type="l", xlab="Temperature (Celcius)", ylab="f(T)",ylim=c(0,20),xlim=c(20,40),col=2)
polygon(x=c(Temp,rev(Temp)), y=c(dfT(Temp,summary(fm)[5,2]),rev(dfT(Temp,summary(fm)[7,2]))),col="gray",border=NA)
lines(Temp,dfT(Temp,Q10m),col=2)

plot(Temp,dfTdT(Temp,Q10m),type="l", xlab="Temperature (Celcius)", ylab=expression(paste(partialdiff,"f(T)","/",partialdiff,"T")),ylim=c(0,2),xlim=c(20,40),col=2)
polygon(x=c(Temp,rev(Temp)), y=c(dfTdT(Temp,summary(fm)[5,2]),rev(dfTdT(Temp,summary(fm)[7,2]))),col="gray",border=NA)
lines(Temp,dfTdT(Temp,Q10m),col=2)

plot(W,dfW(W,Ksm),type="l", xlab="Water-filled pore space (%)", ylab="f(W)",ylim=c(0,1),col=2)
polygon(x=c(W,rev(W)), y=c(dfW(W,summary(fm)[5,3]),rev(dfW(W,summary(fm)[7,3]))),col="gray",border=NA)
lines(W,dfW(W,Ksm),col=2)

plot(W,dfWdW(W,Ksm),type="l", xlab="Water-filled pore space (%)", ylab=expression(paste(partialdiff,"f(W)","/",partialdiff,"W")),ylim=c(0,1),col=2)
polygon(x=c(W,rev(W)), y=c(dfWdW(W,summary(fm)[5,3]),rev(dfWdW(W,summary(fm)[7,3]))),col="gray",border=NA)
lines(W,dfWdW(W,Ksm),col=2)

plot(O,dfO(O,Kom),type="l", xlab="Oxygen concentration (%)", ylab="f(O)",ylim=c(0,1),col=2)
polygon(x=c(O,rev(O)), y=c(dfO(O,summary(fm)[5,4]),rev(dfO(O,summary(fm)[7,4]))),col="gray",border=NA)
lines(O,dfO(O,Kom),col=2)

plot(O,dfOdO(O,Kom),type="l", xlab="Oxygen concentration (%)", ylab=expression(paste(partialdiff,"f(O)","/",partialdiff,"O")),ylim=c(0,1),col=2)
polygon(x=c(O,rev(O)), y=c(dfOdO(O,summary(fm)[5,4]),rev(dfOdO(O,summary(fm)[7,4]))),col="gray",border=NA)
lines(O,dfOdO(O,Kom),col=2)
par(mfrow=c(1,1))
dev.off()


###########################################################################################
#End