module biolight 

  implicit none

  interface bio_light
    module procedure bio_light1
    module procedure bio_light2
    module procedure bio_light3
  end interface


  private
  
  public :: bio_light
  
contains
  
  !----------------------------------------------------------------------------!
  !     calculate the light considering self shadowing
  !----------------------------------------------------------------------------!
  subroutine bio_light1(kb,hi,hsrf,swrad,ecoa,kpar,PPI)

    use bioparam,    only : Nnorm,               &
                            idx_dia            , &
                            idx_flag           , &
                            idx_cyano          , &
                            idx_det            , &
                            att_par_1, att_par_2
    !--------------------------------------------------------------------------!
    !     cmod variables
    !--------------------------------------------------------------------------!
    integer(4), intent(in) :: kb
    real(8), intent(in)    :: hi(:)             ! (ke)
    real(8), intent(in)    :: hsrf
    real(8), intent(in)    :: swrad

    !--------------------------------------------------------------------------!
    !     biomod variables
    !--------------------------------------------------------------------------!
    real(8), intent(in)    :: ecoa(:,0:)         ! (ke,0:ntbio)  old values
    real(8), intent(out)   :: PPI(:)            ! (1:ke)
    real(8), intent(out)   :: kpar(:)            ! (1:ke)

    !--------------------------------------------------------------------------!
    !     local variables
    !--------------------------------------------------------------------------!
    real(8) :: I0norm          ! light
    real(8) :: ss
    real(8) :: ssz             ! light path
    real(8) :: shade
    real(8) :: hzk, exparg, fac

    !--------------------------------------------------------------------------!
    ! constants
    !--------------------------------------------------------------------------!
    real(8), parameter :: max_exp = 20.0_8, half = 0.5_8, zero = 0.0_8

    !--------------------------------------------------------------------------!
    ! loop counters
    !--------------------------------------------------------------------------!
    integer(4) :: k

    !--------------------------------------------------------------------------!
    ! level depths
    !--------------------------------------------------------------------------!
    real(8) :: zw              ! from surface to bottom of k'th layer
    real(8) :: zt              ! from surface to center of k'th layer
    
    !--------------------------------------------------------------------------!
    !     calculate the light considering self shadowing
    !--------------------------------------------------------------------------!

    I0norm = half*swrad

    !--------------------------------------------------------------------------!
    ! surface layer
    !--------------------------------------------------------------------------!
    k  = 1

    !--------------------------------------------------------------------------!
    ! calculate depth of (center,bottom) of this layer
    !--------------------------------------------------------------------------!
    zw = hsrf
    zt = half*zw

    !--------------------------------------------------------------------------!
    !           biology self shading only in this loop
    !--------------------------------------------------------------------------!

    shade =                              &
            ecoa(k,idx_dia            ) + &
            ecoa(k,idx_flag           ) + &
            ecoa(k,idx_cyano          ) + &
            ecoa(k,idx_det            ) + &
            zero

    fac    = Nnorm*att_par_2

    ss     = shade*zt
    ssz    = att_par_1 + fac*ss/zw
    kpar(k)= ssz
    PPI(k) = I0norm*exp(-ssz*zt)

    !--------------------------------------------------------------------------!
    ! layers below surface
    !--------------------------------------------------------------------------!
    do k=2,kb
      hzk = hi(k)

      !------------------------------------------------------------------------!
      ! calculate depth of (center,bottom) of this layer
      !------------------------------------------------------------------------!
      zt = zw + half*hzk
      zw = zw + hzk

      !------------------------------------------------------------------------!
      !           biology self shading only in this loop
      !------------------------------------------------------------------------!

      shade =                              &
              ecoa(k,idx_dia            ) + &
              ecoa(k,idx_flag           ) + &
              ecoa(k,idx_cyano          ) + &
              ecoa(k,idx_det            ) + &
              zero

      ss  = ss + shade*hzk
      ssz = att_par_1 + fac*ss/zw
      kpar(k)=ssz
      exparg = ssz*zt

      if (exparg > max_exp) then
        PPI(k) = zero
      else
        PPI(k) = I0norm*exp(-exparg)
      endif

    enddo ! k

  end subroutine bio_light1

  !============================================================================!

  subroutine bio_light2(kb,hi,hsrf,swrad,ecoa,dispv,kpar,PPI)
    !---------------------------------------------------------------------------
    !
    ! Calculation of PAR in the water column. This routine parameterizes the 
    ! turbidity by using the vertical eddy diffusion and water depth
    !
    !---------------------------------------------------------------------------
    use bioparam,    only : Nnorm,               &
                            idx_dia            , &
                            idx_flag           , &
                            idx_cyano          , &
                            idx_det            , &
                            att_par_1, att_par_2, att_par_3, att_depth,        &
                            att_vdiff    
    !- directives 
    implicit none

    !- arguments
    integer(4), intent(in) :: kb
    real(8), intent(in)    :: hi(:), hsrf, swrad, ecoa(:,0:), dispv(:)
    real(8), intent(out)   :: kpar(:)            ! (1:ke)
    real(8), intent(out)   :: PPI(:)

    !- locals
    real(8)    :: I0norm, ss, ssz, shade, hzk, exparg, fac, avv, dep, zw, zt
    integer(4) :: k

    !- constants
    real(8), parameter :: max_exp = 20.0_8, half = 0.5_8, zero = 0.0_8
    real(8), parameter :: two = 2.0_8, zero8 = 0.08_8
    real(8), parameter :: avv_max = 100.0_8, avv_min = 0.0001_8
    real(8), parameter :: dep_max = 256.0_8, dep_min = 8.0_8

    !--------------------------------------------------------------------------!
    !     calculate the light considering self shadowing
    !--------------------------------------------------------------------------!
    I0norm = half*swrad

    !--------------------------------------------------------------------------!
    ! surface layer
    !--------------------------------------------------------------------------!
    k  = 1

    !--------------------------------------------------------------------------!
    ! calculate depth of (center,bottom) of this layer
    !--------------------------------------------------------------------------!
    zw = hsrf
    zt = half*zw

    !--------------------------------------------------------------------------!
    ! calculate the average vertical  diffusivity and the  depth
    !--------------------------------------------------------------------------!
    !fixme: this is not the way to calculate the mean!
    avv=sum(dispv(1:kb))/kb
    avv=max(min(avv,avv_max),avv_min)
    dep=sum(hi(1:kb))
    dep=max(min(dep,dep_max),dep_min)

    !--------------------------------------------------------------------------!
    !           biology self shading only in this loop
    !--------------------------------------------------------------------------!
    shade =                              &
            ecoa(k,idx_dia            ) + &
            ecoa(k,idx_flag           ) + &
            ecoa(k,idx_cyano          ) + &
            ecoa(k,idx_det            ) + &
            zero

    fac    = Nnorm*att_par_2

    ss     = shade*zt
    ssz    = att_par_1 + fac*ss/zw + att_par_3*(log10(avv/att_vdiff)-          &
                                                log(dep/att_depth)/log(two))
    ssz = max(zero8,ssz)
    kpar(k)=ssz
    PPI(k) = I0norm*exp(-ssz*zt)

    !--------------------------------------------------------------------------!
    ! layers below surface
    !--------------------------------------------------------------------------!
    do k=2,kb
      hzk = hi(k)

      !------------------------------------------------------------------------!
      ! calculate depth of (center,bottom) of this layer
      !------------------------------------------------------------------------!
      zt = zw + half*hzk
      zw = zw + hzk

      !------------------------------------------------------------------------!
      !           biology self shading only in this loop
      !------------------------------------------------------------------------!

      shade =                              &
              ecoa(k,idx_dia            ) + &
              ecoa(k,idx_flag           ) + &
              ecoa(k,idx_cyano          ) + &
              ecoa(k,idx_det            ) + &
              zero

      ss  = ss + shade*hzk
      !fixme: change the formulations of this expression
      ssz = att_par_1 + fac*ss/zw + att_par_3*(log10(avv/att_vdiff)-           &
                                               log(dep/att_depth)/log(two))
      ssz = max(zero8,ssz)
      kpar(k)=ssz
      exparg = ssz*zt

      if (exparg > max_exp) then
        PPI(k) = zero
      else
        PPI(k) = I0norm*exp(-exparg)
      endif

    enddo ! k 

  end subroutine

  !----------------------------------------------------------------------------!
  !     new light calculation including chlorophyll calculation
  !----------------------------------------------------------------------------!

  subroutine bio_light3(kb,hi,hsrf,swrad,ecoa,s_k,light_o,kpar,PPI)
    
    use bioparam,    only : Nnorm, idx_dia, idx_flag, idx_cyano, idx_det,      &
                            att_wat, att_chl, att_det, att_ldon, att_cdom,     &
                            sal_coeff, sal_exp, idx_ldon
    !--------------------------------------------------------------------------!
    !     cmod variables
    !--------------------------------------------------------------------------!
    integer(4), intent(in) :: kb
    real(8), intent(in)    :: hi(:)             ! (ke)
    real(8), intent(in)    :: hsrf
    real(8), intent(in)    :: swrad

    !--------------------------------------------------------------------------!
    !     biomod variables
    !--------------------------------------------------------------------------!
    real(8), intent(in)    :: ecoa(:,0:)         ! (ke,0:ntbio)  old values
    real(8), intent(out)   :: PPI(:)             ! (1:ke)
    real(8), intent(out)   :: kpar(:)            ! (1:ke)
    real(8), intent(in)    :: light_o(:)         ! (1:ke)
    real(8), intent(in)    :: s_k(:)             ! (1:ke)

    !- params
    real(8), parameter  :: min_chl_to_n = 2.00_8   !mg Chl/mmol N
    real(8), parameter  :: max_chl_to_n = 3.00_8   !mg Chl/mmol N 
    real(8), parameter  :: light_max = 30.0_8      !W/m2
    real(8), parameter  :: mweig_N = 14.0_8        !molecular weight of N

    !--------------------------------------------------------------------------!
    !     local variables
    !--------------------------------------------------------------------------!
    real(8) :: I0norm          ! light
    real(8) :: ss
    real(8) :: shade
    real(8) :: hzk, exparg, fac

    real(8) :: r_chl_to_n, total_phyto, frac
    real(8) :: lightk, chlk  


    !--------------------------------------------------------------------------!
    ! constants
    !--------------------------------------------------------------------------!
    real(8), parameter :: max_exp = 20.0_8, half = 0.5_8, zero = 0.0_8
    real(8), parameter :: one = 1.0_8

    !--------------------------------------------------------------------------!
    ! loop counters
    !--------------------------------------------------------------------------!
    integer(4) :: k

    !--------------------------------------------------------------------------!
    ! level depths
    !--------------------------------------------------------------------------!
    real(8) :: zw              ! from surface to bottom of k'th layer
    real(8) :: zt              ! from surface to center of k'th layer
    
    !--------------------------------------------------------------------------!
    !     calculate the light considering self shadowing
    !--------------------------------------------------------------------------!
    frac = max_chl_to_n / (min_chl_to_n*light_max)
    fac = Nnorm * mweig_N      ! factor to calculate mg N m**-3 concentrations
    I0norm = half*swrad

    !--------------------------------------------------------------------------!
    ! surface layer
    !--------------------------------------------------------------------------!

    k  = 1

    !--------------------------------------------------------------------------!
    !     chlorophyll calculation
    !--------------------------------------------------------------------------!


    total_phyto = (ecoa(k,idx_dia)+ecoa(k,idx_flag)+                           &
                     ecoa(k,idx_cyano) )*Nnorm

    if(light_o(k) < zero)then
      chlk = zero
    else
      r_chl_to_n = max( min_chl_to_n, max_chl_to_n*(one-frac*light_o(k)) )
      !- calc chl
      chlk  = r_chl_to_n*total_phyto
    endif

    !--------------------------------------------------------------------------!
    ! calculate depth of (center,bottom) of this layer
    !--------------------------------------------------------------------------!
    zw = hsrf
    zt = half*zw

    !--------------------------------------------------------------------------!
    !           biology self shading only in this loop
    !--------------------------------------------------------------------------!

    shade = att_chl * chlk                                                     &
          + att_det * ecoa(k,idx_det) * fac                                    &
          + att_ldon * ecoa(k,idx_ldon) * fac                                  &
          + att_cdom * (sal_coeff * s_k(k) ** sal_exp)

    ss      = shade*zt
    kpar(k) = att_wat+ss/zw
    PPI(k)  = I0norm*exp(-kpar(k)*zt)

    !--------------------------------------------------------------------------!
    ! layers below surface
    !--------------------------------------------------------------------------!
    
    do k=2,kb
      hzk = hi(k)

    !--------------------------------------------------------------------------!
    !     chlorophyll calculation
    !--------------------------------------------------------------------------!

      total_phyto = (ecoa(k,idx_dia)+ecoa(k,idx_flag)+                           &
                     ecoa(k,idx_cyano) )*Nnorm

      if(light_o(k) < zero)then
        chlk = zero
      else
        r_chl_to_n = max( min_chl_to_n, max_chl_to_n*(one-frac*light_o(k)) )
        !- calc chl
        chlk  = r_chl_to_n*total_phyto
      endif



      !------------------------------------------------------------------------!
      ! calculate depth of (center,bottom) of this layer
      !------------------------------------------------------------------------!
      zt = zw + half*hzk
      zw = zw + hzk

      !------------------------------------------------------------------------!
      !           biology self shading only in this loop
      !------------------------------------------------------------------------!

      shade = att_chl * chlk                                                   &
            + att_det * ecoa(k,idx_det) * fac                                  &
            + att_ldon * ecoa(k,idx_ldon) * fac                                &
            + att_cdom * (sal_coeff * s_k(k) ** sal_exp)

       ss  = ss + shade*hzk
       kpar(k) = att_wat+ss/zw

      exparg = kpar(k)*zt

      if (exparg > max_exp) then
        PPI(k) = zero
      else
        PPI(k) = I0norm*exp(-exparg)
      endif

    enddo ! k

  end subroutine bio_light3

  !============================================================================!

end module biolight
