module bionetcdf_input

  use constants,  only : zero
  
  implicit none
  
  private
  
  public :: netcdf_init, ReadNetcdfClimaloads, ReadNetcdfBounds,               &
            ReadNETCDFatm_dep
  
#if defined (ncdf)
  private :: get_standard_name, get_standard_name_ben, get_standard_name_init, &
             get_standard_name_init_ben, get_standard_name_bounds,             &
             get_standard_name_atmos_dep, get_standard_name_river_infl
#endif


contains 

#if defined (ncdf)

  subroutine netcdf_init (iact, mmx, nmx, kmx, msrf, mcol, kh, ntbio, ntben,   &
                          ptc, benthos, iu06)
    ! Read biological tracers from netcdf files:

    use exits,            only : exitme
    USE bioparam,         ONLY : Nnorm_inv,           &
                                 idx_oxy            , &
                                 NOR
    use netcdf_interface, only : ncdf_open,ncdf_request_varid,ncdf_read_var,   &
                                 ncdf_close_file 

    implicit none

    integer(4), intent(in)    :: mmx, nmx, kmx, iu06
    integer(4), intent(in)    :: iact, ntbio, ntben
    integer(4), intent(in)    :: msrf(0:,0:), mcol(0:), kh(0:)
    ! real(8),    intent(in)    :: Nnorm, Onorm
    real(8),    intent(inout) :: benthos(0:,1:) 
    real(8),    intent(inout) :: ptc(0:,1:)

    integer(4)     :: ncid, varid(ntbio)
    integer(4)     :: varid_ben(ntben)
    integer(4)     :: biotr, m, n, k, msf, mm
    character(3)   :: c3
    character(60)  :: fnam
    character(120) :: std_name(ntbio)
    character(120) :: std_name_ben(ntben)
    real(8)        :: init_data(nmx,mmx,kmx,1)
    real(8)        :: init_ben(nmx,mmx,1,1)
               
    ! Get variable names by calling a subroutine, thus allowing 
    ! for different order of state vars
    call get_standard_name_init(std_name)
       
    !Area number as string for filename 
    if (iact < 10) then
      write(c3,'(i1)') iact
    elseif (iact < 100) then
      write(c3,'(i2)') iact
    else
      call exitme(7,'Cannot handle more than 99 areas.')
    endif
    fnam ='init'//trim(c3)//'.nc'
    
    write(iu06,*) 'Reading bio init netCDF file: '//trim(fnam)
    call ncdf_open(fnam,ncid)
       
    do biotr=1,ntbio
      if ( .not. (trim(std_name(biotr)) .eq. '') ) then
        call ncdf_request_varid(ncid,std_name(biotr),varid(biotr))
        call ncdf_read_var(ncid,varid(biotr),1,init_data, mmx, nmx,kmx)
        
        !rearange grid
        do m=1,mmx
          do n=1,nmx
            ! k=1 unrolled:
            msf = msrf(m,n)
            if (msf <= 0) cycle
            ptc(msf,biotr) = init_data(n,mmx+1-m,1,1)
            do k=2,kh(msf)
              mm = mcol(msf)+k-2
              ptc(mm,biotr) = init_data(n,mmx+1-m,k,1)
            enddo
          enddo
        enddo
      else
        write(iu06,'(a53,i3)') 'Warning, standard_name empty: not reading bio tracer ', biotr
        ptc(:,biotr) = zero
      endif
    enddo
    call ncdf_close_file(ncid)

    ! No need to permute input here because msrf/mcol is already permuted
    ! from cmod_main
    ptc(:,:) = ptc(:,:)*Nnorm_inv
    ptc(:,idx_oxy)     = ptc(:,idx_oxy)*NOR ! instead of *Nnorm/Onorm
        
    !Benthos as netcdf input
    fnam ='ben_init'//trim(c3)//'.nc'
    call get_standard_name_init_ben(std_name_ben)
    write(iu06,*) 'Reading ben init netCDF file: '//trim(fnam)
    call ncdf_open(fnam,ncid)
       
    do biotr = 1,ntben
      if ( .not. (trim(std_name_ben(biotr)) .eq. '') ) then
        call ncdf_request_varid(ncid,std_name_ben(biotr),varid_ben(biotr))
        call ncdf_read_var(ncid,varid_ben(biotr),1,init_ben, mmx, nmx,1)
        
        !rearange grid
        do m=1,mmx
          do n=1,nmx
            mm = msrf(m,n)
            benthos(mm,biotr)=init_ben(n,mmx+1-m,1,1)
          enddo
        enddo
      else
        write(iu06,'(a53,i3)') 'Warning, standard_name empty: not reading ben tracer ', biotr
        benthos(:,biotr) = zero
      endif
    enddo
    call ncdf_close_file(ncid)
    
    benthos(:,:) = benthos(:,:)*Nnorm_inv

  end subroutine netcdf_init

!===============================================================================

  subroutine ReadNetcdfClimaloads( day, nr, iu06, ia, load )    

    use bioparam,         only : Nnorm_inv, NOR,      &
                                 idx_oxy            , &
                                 ntbio
    use exits,            only : exitme
    use netcdf_interface, only : ncdf_open,ncdf_request_varid,ncdf_read_column,&
                                 ncdf_close_file 

    implicit none

    integer(4), intent(in)   :: day, nr, iu06, ia
    real(8), intent(inout)   :: load(:,:,0:)    ! (1:ntracers,1:kmax,0:nr)

   
    character(2)  :: n_river
    character(3)  :: c3    
    character(60) :: fnam
    integer(4)    :: river,biotr
    character(120):: std_name(ntbio)
    integer(4)    :: ncid,varid
    
    real(8)       :: river_data(1,day)
      
    !Area number as string for filename 
    if (ia < 10) then
       write(c3,'(i1)') ia
    elseif (ia < 100) then
       write(c3,'(i2)') ia
    else
       call exitme(7,'Cannot handle more than 99 areas.')
    endif

    !Get variable names
    call get_standard_name_river_infl(std_name)
      
    do river=1,nr      
      !River number as string for filename
      if(river < 10) then
        write(n_river,'(i1)') river
      elseif(river < 100) then
        write(n_river,'(i2)') river
      else
        call exitme(7,'Cannot handle more than 99 river.')
      endif
         
      fnam ='river'//trim(n_river)//'_'//trim(c3)//'.nc'
      write(iu06,*) 'Reading river clim netCDF file: ', fnam
      call ncdf_open(fnam,ncid)
      
      do biotr=1,ntbio
        if ( .not. (trim(std_name(biotr)) .eq. '') ) then
          write(iu06,*) 'read ', std_name(biotr)
          call ncdf_request_varid(ncid,std_name(biotr),varid)
          call ncdf_read_column(ncid,varid,day,river_data,1,1,1)
          load(2+biotr,1,river)=river_data(1,day) * Nnorm_inv
        else
          write(iu06,'(a60,i3)') 'Warning, standard_name empty: not reading inflow for tracer ', biotr
          load(2+biotr,1,river) = zero
        endif
      enddo
      call ncdf_close_file(ncid)
      load(2+idx_oxy,1,river)=load(2+idx_oxy,1,river)*NOR

      ! NEUMANND: all loads are multiplied by Nnorm_inv (=1/Nnorm);
      !           oxygen loads are then multuplied by NOR
      ! OLD CODE: 
      !   load(3:ntbio+1,1,river) = load(3:ntbio+1,1,river) / Nnorm
      !   load(ntbio+2,1,river)   = load(ntbio+2,1,river) / Onorm
      !  where ntbio was the index of oxygen
    enddo

  end subroutine ReadNetcdfClimaloads
  
!===============================================================================

  subroutine ReadNetcdfBounds (nz, iu06, ntbio, kmx, rwzke_a)

    use bioparam,         only : Nnorm_inv,           &
                                 idx_oxy            , &
                                 NOR
    use cmod_params,      only : mainarea, krz, mmx, nmx 
    use netcdf_interface, only : ncdf_open,ncdf_request_varid,ncdf_read_column,&
                                 ncdf_close_file 

    implicit none

    integer(4), intent(in)    :: nz, iu06, ntbio, kmx
    real(8),    intent(inout) :: rwzke_a(:,:,0:,:)
    
    character(60)  :: fnam
    character(120) :: std_name(ntbio)
    integer(4)     :: biotr, n, ibnd, jbnd
    integer(4)     :: ncid, varid(ntbio)

    ! Read from nc files 
    fnam ='bounds.nc'   
    write(iu06,*) 'Reading bounds netCDF file: ', fnam 
    call ncdf_open(fnam,ncid)

    !Get variable names
    call get_standard_name_bounds(std_name) 
    do biotr=1,ntbio
      if ( .not. (trim(std_name(biotr)) .eq. '') ) then
        call ncdf_request_varid(ncid,std_name(biotr),varid(biotr))
        do n=1,nz
          ibnd = mmx(mainarea)+1-krz(mainarea)%p(1,n)
          jbnd = krz(mainarea)%p(2,n)
          call ncdf_read_column(ncid,varid(biotr),12,rwzke_a(biotr,:,n,:),ibnd,jbnd,kmx)
        enddo
        ! we ignore the special treatment for oxygen here and do it below
        rwzke_a(biotr,:,1:,:) = rwzke_a(biotr,:,1:,:)*Nnorm_inv
      else
        write(iu06,'(a56,i3)') 'Warning, standard_name empty: not reading bounds tracer ', biotr
        rwzke_a(biotr,:,1:,:) = zero
      endif
    enddo  
    ! special treatment for oxygen
      rwzke_a(idx_oxy,:,1:,:)=rwzke_a(idx_oxy,:,1:,:)*NOR
    ! NEUMANND:  OLD VERSION
    !  where ntbio == idx_oxy
    ! rwzke_a(1:ntbio-1,:,1:,:) = rwzke_a(1:ntbio-1,:,1:,:)/Nnorm
    ! rwzke_a(ntbio,:,1:,:)     = rwzke_a(ntbio,:,1:,:)/Onorm
    call ncdf_close_file(ncid)

  end subroutine ReadNetcdfBounds

!===============================================================================

  subroutine ReadNETCDFatm_dep(iu06, ia,                         &
                             srfnredflx_const,                &
                             srfnoxflx_const,                &
                             srfpoxflx_const,                &
                             nmx,mmx,                         &
                               srfammflx,                     &
                               srfnitflx,                     &
                               srfphosflx,                     &
                               srfamm_with_totatmos_Nflx,                     &
                               srfnit_with_totatmos_Nflx,                     &
                               srfamm_with_agri_Nflx,                     &
                               srfnit_with_agri_Nflx,                     &
                               msrf)
      
    use netcdf_interface, only : ncdf_open, ncdf_close_file,                  &
                                 ncdf_request_varid, ncdf_read_var
    use exits,            only : exitme
    USE bioparam,         only : tag_atmos_dep,       &
                                 idx_amm            , &
                                 idx_nit            , &
                                 idx_phos           , &
                                 idx_amm_with_totatmos_N, &
                                 idx_nit_with_totatmos_N, &
                                 idx_amm_with_agri_N, &
                                 idx_nit_with_agri_N, &
                                 ntbio
   
    
    implicit none
    
    integer(4), intent(in)   :: iu06, ia, nmx, mmx, msrf(0:,0:)
    real(8),    intent(in)   :: srfnredflx_const
    real(8),    intent(in)   :: srfnoxflx_const
    real(8),    intent(in)   :: srfpoxflx_const
    real(8),    intent(out)  :: srfammflx(0:) ! atmospheric deposition flux: ammonium
    real(8),    intent(out)  :: srfnitflx(0:) ! atmospheric deposition flux: nitrate
    real(8),    intent(out)  :: srfphosflx(0:) ! atmospheric deposition flux: phosphate
    real(8),    intent(out)  :: srfamm_with_totatmos_Nflx(0:) ! atmospheric deposition flux: ammonium; containing 
    real(8),    intent(out)  :: srfnit_with_totatmos_Nflx(0:) ! atmospheric deposition flux: nitrate; containing 
    real(8),    intent(out)  :: srfamm_with_agri_Nflx(0:) ! atmospheric deposition flux: ammonium; containing 
    real(8),    intent(out)  :: srfnit_with_agri_Nflx(0:) ! atmospheric deposition flux: nitrate; containing 
   
   ! TODO
    integer(4)    :: ncid, m, n, msf
    character(3)  :: c3    
    character(60) :: fnam
    character(120):: std_name(ntbio)
    
    integer(4)    :: varid_nred
    real(8)       :: srfnredflx_4d(nmx,mmx,1,1)
    character(100):: nredname
    
    integer(4)    :: varid_nox
    real(8)       :: srfnoxflx_4d(nmx,mmx,1,1)
    character(100):: noxname
    
    integer(4)    :: varid_pox
    real(8)       :: srfpoxflx_4d(nmx,mmx,1,1)
    character(100):: poxname
    
    integer(4)    :: varid_nred_with_totatmos_N
    real(8)       :: srfnred_with_totatmos_Nflx_4d(nmx,mmx,1,1)
    character(100):: nred_with_totatmos_Nname
    
    integer(4)    :: varid_nox_with_totatmos_N
    real(8)       :: srfnox_with_totatmos_Nflx_4d(nmx,mmx,1,1)
    character(100):: nox_with_totatmos_Nname
    
    integer(4)    :: varid_nred_with_agri_N
    real(8)       :: srfnred_with_agri_Nflx_4d(nmx,mmx,1,1)
    character(100):: nred_with_agri_Nname
    
    integer(4)    :: varid_nox_with_agri_N
    real(8)       :: srfnox_with_agri_Nflx_4d(nmx,mmx,1,1)
    character(100):: nox_with_agri_Nname
    
    
    !Area number as string for filename 
    if (ia < 10) then
       write(c3,'(i1)') ia
    elseif (ia < 100) then
       write(c3,'(i2)') ia
    else
       call exitme(7,'Cannot handle more than 99 areas.')
    endif
    
    
    ! INITIALIZATION
    ! Distinguish initialization values of colored and uncolored tracers
      srfammflx(0:) = srfnredflx_const
      srfnitflx(0:) = srfnoxflx_const
      srfphosflx(0:) = srfpoxflx_const
    ! For ERGOM CGT users: One could also use the "celements" + "containingstracers"
    !                      loops instead of the "tracers" loop. However, one needed
    !                      to replace the tags. E.g. <tracerBelow> in the tracers
    !                      loop is <ctBelow> in the containingtracers loop. Since
    !                      the code written above for uncolored tracers is quite 
    !                      similar (note the different standard values assigned), it
    !                      is reasonable to choose the loop type for which the least
    !                      changes are needed.
    ! Alternatives:
    !   <  celements color=ship; atmosdep=1>
    !     <  containingtracers vertLoc=SUR>
    ! 
    ! START: If you see no code below, the code was generated without "N from ship"
    ! END: If you see no code above, the code was generated without "N from ship"
    ! Alternatives:
    !   <  /containingtracers>
    !     <  /celements>
    
    
    ! READING DATA
    !Get variable names
    call get_standard_name_atmos_dep(std_name)

    fnam ='atm_dep_'//trim(c3)//'.nc'
    write(iu06,*) 'Reading atmos dep netCDF file: '//trim(fnam)
    call ncdf_open(fnam, ncid)
  
  
    ! We take the standard values defined above in two situations:
    !  - the standard name is empty (no variable to read in from the file)
    !  - the variable is a tagged one but tagging atmos dep is off
    if ( ( .not. (trim(std_name(idx_amm)) .eq. '') ) .and.        &
         ( (('none' .eq. 'none') .or. tag_atmos_dep) )         &
        ) then
      write(iu06,*) 'read ', std_name(idx_amm)
      ! get data
      call ncdf_request_varid(ncid,std_name(idx_amm),varid_nred)
      call ncdf_read_var(ncid,varid_nred,1,srfnredflx_4d, mmx, nmx, 1)
      ! rearange grid
      do m=1,mmx
        do n=1,nmx
          ! k=1 unrolled:
          msf = msrf(m,n)
          if (msf <= 0) cycle
          srfammflx(msf) = srfnredflx_4d(n,mmx+1-m,1,1)
        enddo
      enddo
    else
      write(iu06,*) 'Warning, standard_name empty: not reading atmos dep for tracer amm; ...'
      write(iu06,'(a53,f10.5)') '   ... using default value instead: ', srfammflx(0)
    endif
    
    if ( ( .not. (trim(std_name(idx_nit)) .eq. '') ) .and.        &
         ( (('none' .eq. 'none') .or. tag_atmos_dep) )         &
        ) then
      write(iu06,*) 'read ', std_name(idx_nit)
      ! get data
      call ncdf_request_varid(ncid,std_name(idx_nit),varid_nox)
      call ncdf_read_var(ncid,varid_nox,1,srfnoxflx_4d, mmx, nmx, 1)
      ! rearange grid
      do m=1,mmx
        do n=1,nmx
          ! k=1 unrolled:
          msf = msrf(m,n)
          if (msf <= 0) cycle
          srfnitflx(msf) = srfnoxflx_4d(n,mmx+1-m,1,1)
        enddo
      enddo
    else
      write(iu06,*) 'Warning, standard_name empty: not reading atmos dep for tracer nit; ...'
      write(iu06,'(a53,f10.5)') '   ... using default value instead: ', srfnitflx(0)
    endif
    
    if ( ( .not. (trim(std_name(idx_phos)) .eq. '') ) .and.        &
         ( (('none' .eq. 'none') .or. tag_atmos_dep) )         &
        ) then
      write(iu06,*) 'read ', std_name(idx_phos)
      ! get data
      call ncdf_request_varid(ncid,std_name(idx_phos),varid_pox)
      call ncdf_read_var(ncid,varid_pox,1,srfpoxflx_4d, mmx, nmx, 1)
      ! rearange grid
      do m=1,mmx
        do n=1,nmx
          ! k=1 unrolled:
          msf = msrf(m,n)
          if (msf <= 0) cycle
          srfphosflx(msf) = srfpoxflx_4d(n,mmx+1-m,1,1)
        enddo
      enddo
    else
      write(iu06,*) 'Warning, standard_name empty: not reading atmos dep for tracer phos; ...'
      write(iu06,'(a53,f10.5)') '   ... using default value instead: ', srfphosflx(0)
    endif
    
    if ( ( .not. (trim(std_name(idx_amm_with_totatmos_N)) .eq. '') ) .and.        &
         ( (('nred' .eq. 'none') .or. tag_atmos_dep) )         &
        ) then
      write(iu06,*) 'read ', std_name(idx_amm_with_totatmos_N)
      ! get data
      call ncdf_request_varid(ncid,std_name(idx_amm_with_totatmos_N),varid_nred_with_totatmos_N)
      call ncdf_read_var(ncid,varid_nred_with_totatmos_N,1,srfnred_with_totatmos_Nflx_4d, mmx, nmx, 1)
      ! rearange grid
      do m=1,mmx
        do n=1,nmx
          ! k=1 unrolled:
          msf = msrf(m,n)
          if (msf <= 0) cycle
          srfamm_with_totatmos_Nflx(msf) = srfnred_with_totatmos_Nflx_4d(n,mmx+1-m,1,1)
        enddo
      enddo
    else
      write(iu06,*) 'Warning, standard_name empty: not reading atmos dep for tracer amm_with_totatmos_N; ...'
      write(iu06,'(a53,f10.5)') '   ... using default value instead: ', srfamm_with_totatmos_Nflx(0)
    endif
    
    if ( ( .not. (trim(std_name(idx_nit_with_totatmos_N)) .eq. '') ) .and.        &
         ( (('nox' .eq. 'none') .or. tag_atmos_dep) )         &
        ) then
      write(iu06,*) 'read ', std_name(idx_nit_with_totatmos_N)
      ! get data
      call ncdf_request_varid(ncid,std_name(idx_nit_with_totatmos_N),varid_nox_with_totatmos_N)
      call ncdf_read_var(ncid,varid_nox_with_totatmos_N,1,srfnox_with_totatmos_Nflx_4d, mmx, nmx, 1)
      ! rearange grid
      do m=1,mmx
        do n=1,nmx
          ! k=1 unrolled:
          msf = msrf(m,n)
          if (msf <= 0) cycle
          srfnit_with_totatmos_Nflx(msf) = srfnox_with_totatmos_Nflx_4d(n,mmx+1-m,1,1)
        enddo
      enddo
    else
      write(iu06,*) 'Warning, standard_name empty: not reading atmos dep for tracer nit_with_totatmos_N; ...'
      write(iu06,'(a53,f10.5)') '   ... using default value instead: ', srfnit_with_totatmos_Nflx(0)
    endif
    
    if ( ( .not. (trim(std_name(idx_amm_with_agri_N)) .eq. '') ) .and.        &
         ( (('nred' .eq. 'none') .or. tag_atmos_dep) )         &
        ) then
      write(iu06,*) 'read ', std_name(idx_amm_with_agri_N)
      ! get data
      call ncdf_request_varid(ncid,std_name(idx_amm_with_agri_N),varid_nred_with_agri_N)
      call ncdf_read_var(ncid,varid_nred_with_agri_N,1,srfnred_with_agri_Nflx_4d, mmx, nmx, 1)
      ! rearange grid
      do m=1,mmx
        do n=1,nmx
          ! k=1 unrolled:
          msf = msrf(m,n)
          if (msf <= 0) cycle
          srfamm_with_agri_Nflx(msf) = srfnred_with_agri_Nflx_4d(n,mmx+1-m,1,1)
        enddo
      enddo
    else
      write(iu06,*) 'Warning, standard_name empty: not reading atmos dep for tracer amm_with_agri_N; ...'
      write(iu06,'(a53,f10.5)') '   ... using default value instead: ', srfamm_with_agri_Nflx(0)
    endif
    
    if ( ( .not. (trim(std_name(idx_nit_with_agri_N)) .eq. '') ) .and.        &
         ( (('nox' .eq. 'none') .or. tag_atmos_dep) )         &
        ) then
      write(iu06,*) 'read ', std_name(idx_nit_with_agri_N)
      ! get data
      call ncdf_request_varid(ncid,std_name(idx_nit_with_agri_N),varid_nox_with_agri_N)
      call ncdf_read_var(ncid,varid_nox_with_agri_N,1,srfnox_with_agri_Nflx_4d, mmx, nmx, 1)
      ! rearange grid
      do m=1,mmx
        do n=1,nmx
          ! k=1 unrolled:
          msf = msrf(m,n)
          if (msf <= 0) cycle
          srfnit_with_agri_Nflx(msf) = srfnox_with_agri_Nflx_4d(n,mmx+1-m,1,1)
        enddo
      enddo
    else
      write(iu06,*) 'Warning, standard_name empty: not reading atmos dep for tracer nit_with_agri_N; ...'
      write(iu06,'(a53,f10.5)') '   ... using default value instead: ', srfnit_with_agri_Nflx(0)
    endif
    

    
    call ncdf_close_file(ncid)

  end subroutine ReadNETCDFatm_dep  

!===============================================================================

  subroutine get_standard_name(std_name)
  
    use bioparam, only :                                 &
                         idx_amm            ,            &
                         idx_nit            ,            &
                         idx_phos           ,            &
                         idx_sil            ,            &
                         idx_dia            ,            &
                         idx_flag           ,            &
                         idx_cyano          ,            &
                         idx_mez            ,            &
                         idx_miz            ,            &
                         idx_det            ,            &
                         idx_dets           ,            &
                         idx_ldon           ,            &
                         idx_oxy            ,            &
                         idx_amm_with_totatmos_N,            &
                         idx_nit_with_totatmos_N,            &
                         idx_dia_with_totatmos_N,            &
                         idx_flag_with_totatmos_N,            &
                         idx_cyano_with_totatmos_N,            &
                         idx_mez_with_totatmos_N,            &
                         idx_miz_with_totatmos_N,            &
                         idx_det_with_totatmos_N,            &
                         idx_ldon_with_totatmos_N,            &
                         idx_amm_with_agri_N,            &
                         idx_nit_with_agri_N,            &
                         idx_dia_with_agri_N,            &
                         idx_flag_with_agri_N,            &
                         idx_cyano_with_agri_N,            &
                         idx_mez_with_agri_N,            &
                         idx_miz_with_agri_N,            &
                         idx_det_with_agri_N,            &
                         idx_ldon_with_agri_N,            &
                         ntbio 
    use exits,    only : exitme
  
    implicit none
  
    character(len=120), intent(out) :: std_name(:)
    
    call exitme('bionetcdf_input: get_standard_name is deprecated')
    
    std_name(idx_amm            )  = 'mole_concentration_of_ammonium_in_seawater'
    std_name(idx_nit            )  = 'mole_concentration_of_nitrate_in_seawater'
    std_name(idx_phos           )  = 'mole_concentration_of_phosphate_in_seawater'
    std_name(idx_sil            )  = 'mole_concentration_of_silicate_in_seawater'
    std_name(idx_dia            )  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_in_seawater'
    std_name(idx_flag           )  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_in_seawater'
    std_name(idx_cyano          )  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_in_seawater'
    std_name(idx_mez            )  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_in_seawater'
    std_name(idx_miz            )  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_in_seawater'
    std_name(idx_det            )  = 'mole_concentration_of_detritus_expressed_as_nitrogen_in_seawater'
    std_name(idx_dets           )  = 'mole_concentration_of_detritus_expressed_as_silicate_in_seawater'
    std_name(idx_ldon           )  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_expressed_as_nitrogen_in_seawater'
    std_name(idx_oxy            )  = 'mole_concentration_of_oxygen_in_seawater'
    std_name(idx_amm_with_totatmos_N)  = 'mole_concentration_of_ammonium_from_atmospheric_deposition_in_seawater'
    std_name(idx_nit_with_totatmos_N)  = 'mole_concentration_of_nitrate_from_atmospheric_deposition_in_seawater'
    std_name(idx_dia_with_totatmos_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
    std_name(idx_flag_with_totatmos_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
    std_name(idx_cyano_with_totatmos_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
    std_name(idx_mez_with_totatmos_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
    std_name(idx_miz_with_totatmos_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
    std_name(idx_det_with_totatmos_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
    std_name(idx_ldon_with_totatmos_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_atmospheric_deposition_expressed_as_nitrogen_in_seawater'
    std_name(idx_amm_with_agri_N)  = 'mole_concentration_of_ammonium_from_agricultural_activities_in_seawater'
    std_name(idx_nit_with_agri_N)  = 'mole_concentration_of_nitrate_from_agricultural_activities_in_seawater'
    std_name(idx_dia_with_agri_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
    std_name(idx_flag_with_agri_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
    std_name(idx_cyano_with_agri_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
    std_name(idx_mez_with_agri_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
    std_name(idx_miz_with_agri_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
    std_name(idx_det_with_agri_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
    std_name(idx_ldon_with_agri_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_agricultural_activities_expressed_as_nitrogen_in_seawater'
  
  end subroutine get_standard_name

!===============================================================================

  subroutine get_standard_name_ben(std_name_ben)
  
    use bioparam, only :                       &
                         bidx_nitr           , &
                         bidx_sili           , &
                         bidx_nitr_with_totatmos_N, &
                         bidx_nitr_with_agri_N, &
                         ntben
    use exits,    only : exitme
  
    implicit none
  
    character(len=120), intent(out) :: std_name_ben(:) 
    
    call exitme('bionetcdf_input: get_standard_name_ben is deprecated')
    
    ! NEUMANND: changed name of sediment output variable to CF-standardname
    std_name_ben(bidx_nitr) = 'mole_concentration_of_detritus_expressed_as_nitrogen_in_seawater'
    std_name_ben(bidx_sili) = 'mole_concentration_of_detritus_expressed_as_silicate_in_seawater'
    std_name_ben(bidx_nitr_with_totatmos_N) = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
    std_name_ben(bidx_nitr_with_agri_N) = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
  
  end subroutine get_standard_name_ben

!===============================================================================

  subroutine get_standard_name_init(std_name)
  
    use bioparam, only :  use_nc_varname_nml_init,                      &
                          tag_river_infl, tag_atmos_dep,              &
                          read_netcdf_init_notag,                    &
                         idx_amm            ,                         &
                         idx_nit            ,                         &
                         idx_phos           ,                         &
                         idx_sil            ,                         &
                         idx_dia            ,                         &
                         idx_flag           ,                         &
                         idx_cyano          ,                         &
                         idx_mez            ,                         &
                         idx_miz            ,                         &
                         idx_det            ,                         &
                         idx_dets           ,                         &
                         idx_ldon           ,                         &
                         idx_oxy            ,                         &
                         idx_amm_with_totatmos_N,                         &
                         idx_nit_with_totatmos_N,                         &
                         idx_dia_with_totatmos_N,                         &
                         idx_flag_with_totatmos_N,                         &
                         idx_cyano_with_totatmos_N,                         &
                         idx_mez_with_totatmos_N,                         &
                         idx_miz_with_totatmos_N,                         &
                         idx_det_with_totatmos_N,                         &
                         idx_ldon_with_totatmos_N,                         &
                         idx_amm_with_agri_N,                         &
                         idx_nit_with_agri_N,                         &
                         idx_dia_with_agri_N,                         &
                         idx_flag_with_agri_N,                         &
                         idx_cyano_with_agri_N,                         &
                         idx_mez_with_agri_N,                         &
                         idx_miz_with_agri_N,                         &
                         idx_det_with_agri_N,                         &
                         idx_ldon_with_agri_N,                         &
                         nc_varname_nml_init_amm            ,            &
                         nc_varname_nml_init_nit            ,            &
                         nc_varname_nml_init_phos           ,            &
                         nc_varname_nml_init_sil            ,            &
                         nc_varname_nml_init_dia            ,            &
                         nc_varname_nml_init_flag           ,            &
                         nc_varname_nml_init_cyano          ,            &
                         nc_varname_nml_init_mez            ,            &
                         nc_varname_nml_init_miz            ,            &
                         nc_varname_nml_init_det            ,            &
                         nc_varname_nml_init_dets           ,            &
                         nc_varname_nml_init_ldon           ,            &
                         nc_varname_nml_init_oxy            ,            &
                         nc_varname_nml_init_amm_with_totatmos_N,            &
                         nc_varname_nml_init_nit_with_totatmos_N,            &
                         nc_varname_nml_init_dia_with_totatmos_N,            &
                         nc_varname_nml_init_flag_with_totatmos_N,            &
                         nc_varname_nml_init_cyano_with_totatmos_N,            &
                         nc_varname_nml_init_mez_with_totatmos_N,            &
                         nc_varname_nml_init_miz_with_totatmos_N,            &
                         nc_varname_nml_init_det_with_totatmos_N,            &
                         nc_varname_nml_init_ldon_with_totatmos_N,            &
                         nc_varname_nml_init_amm_with_agri_N,            &
                         nc_varname_nml_init_nit_with_agri_N,            &
                         nc_varname_nml_init_dia_with_agri_N,            &
                         nc_varname_nml_init_flag_with_agri_N,            &
                         nc_varname_nml_init_cyano_with_agri_N,            &
                         nc_varname_nml_init_mez_with_agri_N,            &
                         nc_varname_nml_init_miz_with_agri_N,            &
                         nc_varname_nml_init_det_with_agri_N,            &
                         nc_varname_nml_init_ldon_with_agri_N,            &
                         ntbio 
  
    implicit none
  
    character(len=120), intent(out) :: std_name(:)
    
    if (use_nc_varname_nml_init) then
      std_name(idx_amm            )  = nc_varname_nml_init_amm            
      std_name(idx_nit            )  = nc_varname_nml_init_nit            
      std_name(idx_phos           )  = nc_varname_nml_init_phos           
      std_name(idx_sil            )  = nc_varname_nml_init_sil            
      std_name(idx_dia            )  = nc_varname_nml_init_dia            
      std_name(idx_flag           )  = nc_varname_nml_init_flag           
      std_name(idx_cyano          )  = nc_varname_nml_init_cyano          
      std_name(idx_mez            )  = nc_varname_nml_init_mez            
      std_name(idx_miz            )  = nc_varname_nml_init_miz            
      std_name(idx_det            )  = nc_varname_nml_init_det            
      std_name(idx_dets           )  = nc_varname_nml_init_dets           
      std_name(idx_ldon           )  = nc_varname_nml_init_ldon           
      std_name(idx_oxy            )  = nc_varname_nml_init_oxy            
      std_name(idx_amm_with_totatmos_N)  = nc_varname_nml_init_amm_with_totatmos_N
      std_name(idx_nit_with_totatmos_N)  = nc_varname_nml_init_nit_with_totatmos_N
      std_name(idx_dia_with_totatmos_N)  = nc_varname_nml_init_dia_with_totatmos_N
      std_name(idx_flag_with_totatmos_N)  = nc_varname_nml_init_flag_with_totatmos_N
      std_name(idx_cyano_with_totatmos_N)  = nc_varname_nml_init_cyano_with_totatmos_N
      std_name(idx_mez_with_totatmos_N)  = nc_varname_nml_init_mez_with_totatmos_N
      std_name(idx_miz_with_totatmos_N)  = nc_varname_nml_init_miz_with_totatmos_N
      std_name(idx_det_with_totatmos_N)  = nc_varname_nml_init_det_with_totatmos_N
      std_name(idx_ldon_with_totatmos_N)  = nc_varname_nml_init_ldon_with_totatmos_N
      std_name(idx_amm_with_agri_N)  = nc_varname_nml_init_amm_with_agri_N
      std_name(idx_nit_with_agri_N)  = nc_varname_nml_init_nit_with_agri_N
      std_name(idx_dia_with_agri_N)  = nc_varname_nml_init_dia_with_agri_N
      std_name(idx_flag_with_agri_N)  = nc_varname_nml_init_flag_with_agri_N
      std_name(idx_cyano_with_agri_N)  = nc_varname_nml_init_cyano_with_agri_N
      std_name(idx_mez_with_agri_N)  = nc_varname_nml_init_mez_with_agri_N
      std_name(idx_miz_with_agri_N)  = nc_varname_nml_init_miz_with_agri_N
      std_name(idx_det_with_agri_N)  = nc_varname_nml_init_det_with_agri_N
      std_name(idx_ldon_with_agri_N)  = nc_varname_nml_init_ldon_with_agri_N
    else
      std_name(idx_amm            )  = 'mole_concentration_of_ammonium_in_seawater'
      std_name(idx_nit            )  = 'mole_concentration_of_nitrate_in_seawater'
      std_name(idx_phos           )  = 'mole_concentration_of_phosphate_in_seawater'
      std_name(idx_sil            )  = 'mole_concentration_of_silicate_in_seawater'
      std_name(idx_dia            )  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_in_seawater'
      std_name(idx_flag           )  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_in_seawater'
      std_name(idx_cyano          )  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_in_seawater'
      std_name(idx_mez            )  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_in_seawater'
      std_name(idx_miz            )  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_in_seawater'
      std_name(idx_det            )  = 'mole_concentration_of_detritus_expressed_as_nitrogen_in_seawater'
      std_name(idx_dets           )  = 'mole_concentration_of_detritus_expressed_as_silicate_in_seawater'
      std_name(idx_ldon           )  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_expressed_as_nitrogen_in_seawater'
      std_name(idx_oxy            )  = 'mole_concentration_of_oxygen_in_seawater'
      std_name(idx_amm_with_totatmos_N)  = 'mole_concentration_of_ammonium_from_atmospheric_deposition_in_seawater'
      std_name(idx_nit_with_totatmos_N)  = 'mole_concentration_of_nitrate_from_atmospheric_deposition_in_seawater'
      std_name(idx_dia_with_totatmos_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name(idx_flag_with_totatmos_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name(idx_cyano_with_totatmos_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name(idx_mez_with_totatmos_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name(idx_miz_with_totatmos_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name(idx_det_with_totatmos_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name(idx_ldon_with_totatmos_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_atmospheric_deposition_expressed_as_nitrogen_in_seawater'
      std_name(idx_amm_with_agri_N)  = 'mole_concentration_of_ammonium_from_agricultural_activities_in_seawater'
      std_name(idx_nit_with_agri_N)  = 'mole_concentration_of_nitrate_from_agricultural_activities_in_seawater'
      std_name(idx_dia_with_agri_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name(idx_flag_with_agri_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name(idx_cyano_with_agri_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name(idx_mez_with_agri_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name(idx_miz_with_agri_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name(idx_det_with_agri_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name(idx_ldon_with_agri_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_agricultural_activities_expressed_as_nitrogen_in_seawater'
    endif
    
    if ( read_netcdf_init_notag .or. .not. tag_river_infl ) then
    endif
    
    if ( read_netcdf_init_notag .or. .not. tag_atmos_dep ) then
    endif
  
  end subroutine get_standard_name_init

!===============================================================================

  subroutine get_standard_name_init_ben(std_name_ben)
  
    use bioparam, only :  use_nc_varname_nml_init_ben,                  &
                          tag_river_infl, tag_atmos_dep,              &
                          read_netcdf_init_notag,                    &
                         bidx_nitr           ,                        &
                         bidx_sili           ,                        &
                         bidx_nitr_with_totatmos_N,                        &
                         bidx_nitr_with_agri_N,                        &
                         nc_varname_nml_init_ben_nitr           ,           &
                         nc_varname_nml_init_ben_sili           ,           &
                         nc_varname_nml_init_ben_nitr_with_totatmos_N,           &
                         nc_varname_nml_init_ben_nitr_with_agri_N,           &
                         ntben 
  
    implicit none
  
    character(len=120), intent(out) :: std_name_ben(:)
    
    if (use_nc_varname_nml_init_ben) then
      std_name_ben(bidx_nitr           ) = nc_varname_nml_init_ben_nitr           
      std_name_ben(bidx_sili           ) = nc_varname_nml_init_ben_sili           
      std_name_ben(bidx_nitr_with_totatmos_N) = nc_varname_nml_init_ben_nitr_with_totatmos_N
      std_name_ben(bidx_nitr_with_agri_N) = nc_varname_nml_init_ben_nitr_with_agri_N
    else
      ! currently: ben_init files do not follow cf conventions
      std_name_ben(bidx_nitr) = 'det            '
      std_name_ben(bidx_sili) = 'dets           '
      std_name_ben(bidx_nitr_with_totatmos_N) = 'det_with_totatmos_N'
      std_name_ben(bidx_nitr_with_agri_N) = 'det_with_agri_N'
      
      ! better:
      ! std_name_sed(bidx_nitr           )  = 'mole_concentration_of_nitrogen_in_the_sediment'
      ! std_name_sed(bidx_sili           )  = 'mole_concentration_of_silicate_in_the_sediment'
      ! std_name_sed(bidx_nitr_with_totatmos_N)  = 'mole_concentration_of_nitrogen_from_atmospheric_deposition_in_the_sediment'
      ! std_name_sed(bidx_nitr_with_agri_N)  = 'mole_concentration_of_nitrogen_from_agricultural_activities_in_the_sediment'
    endif
    
    if ( read_netcdf_init_notag .or. .not. tag_river_infl ) then
    endif
    
    if ( read_netcdf_init_notag .or. .not. tag_atmos_dep ) then
    endif
  
  end subroutine get_standard_name_init_ben

!===============================================================================

  subroutine get_standard_name_bounds(std_name_bounds)
  
    use bioparam, only :  use_nc_varname_nml_bounds,                    &
                          tag_river_infl, tag_atmos_dep,              &
                          read_netcdf_bounds_notag,                  &
                         idx_amm            ,                         &
                         idx_nit            ,                         &
                         idx_phos           ,                         &
                         idx_sil            ,                         &
                         idx_dia            ,                         &
                         idx_flag           ,                         &
                         idx_cyano          ,                         &
                         idx_mez            ,                         &
                         idx_miz            ,                         &
                         idx_det            ,                         &
                         idx_dets           ,                         &
                         idx_ldon           ,                         &
                         idx_oxy            ,                         &
                         idx_amm_with_totatmos_N,                         &
                         idx_nit_with_totatmos_N,                         &
                         idx_dia_with_totatmos_N,                         &
                         idx_flag_with_totatmos_N,                         &
                         idx_cyano_with_totatmos_N,                         &
                         idx_mez_with_totatmos_N,                         &
                         idx_miz_with_totatmos_N,                         &
                         idx_det_with_totatmos_N,                         &
                         idx_ldon_with_totatmos_N,                         &
                         idx_amm_with_agri_N,                         &
                         idx_nit_with_agri_N,                         &
                         idx_dia_with_agri_N,                         &
                         idx_flag_with_agri_N,                         &
                         idx_cyano_with_agri_N,                         &
                         idx_mez_with_agri_N,                         &
                         idx_miz_with_agri_N,                         &
                         idx_det_with_agri_N,                         &
                         idx_ldon_with_agri_N,                         &
                         nc_varname_nml_bounds_amm            ,            &
                         nc_varname_nml_bounds_nit            ,            &
                         nc_varname_nml_bounds_phos           ,            &
                         nc_varname_nml_bounds_sil            ,            &
                         nc_varname_nml_bounds_dia            ,            &
                         nc_varname_nml_bounds_flag           ,            &
                         nc_varname_nml_bounds_cyano          ,            &
                         nc_varname_nml_bounds_mez            ,            &
                         nc_varname_nml_bounds_miz            ,            &
                         nc_varname_nml_bounds_det            ,            &
                         nc_varname_nml_bounds_dets           ,            &
                         nc_varname_nml_bounds_ldon           ,            &
                         nc_varname_nml_bounds_oxy            ,            &
                         nc_varname_nml_bounds_amm_with_totatmos_N,            &
                         nc_varname_nml_bounds_nit_with_totatmos_N,            &
                         nc_varname_nml_bounds_dia_with_totatmos_N,            &
                         nc_varname_nml_bounds_flag_with_totatmos_N,            &
                         nc_varname_nml_bounds_cyano_with_totatmos_N,            &
                         nc_varname_nml_bounds_mez_with_totatmos_N,            &
                         nc_varname_nml_bounds_miz_with_totatmos_N,            &
                         nc_varname_nml_bounds_det_with_totatmos_N,            &
                         nc_varname_nml_bounds_ldon_with_totatmos_N,            &
                         nc_varname_nml_bounds_amm_with_agri_N,            &
                         nc_varname_nml_bounds_nit_with_agri_N,            &
                         nc_varname_nml_bounds_dia_with_agri_N,            &
                         nc_varname_nml_bounds_flag_with_agri_N,            &
                         nc_varname_nml_bounds_cyano_with_agri_N,            &
                         nc_varname_nml_bounds_mez_with_agri_N,            &
                         nc_varname_nml_bounds_miz_with_agri_N,            &
                         nc_varname_nml_bounds_det_with_agri_N,            &
                         nc_varname_nml_bounds_ldon_with_agri_N,            &
                         ntbio 
  
    implicit none
  
    character(len=120), intent(out) :: std_name_bounds(:)
    
    if (use_nc_varname_nml_bounds) then
      std_name_bounds(idx_amm            )  = nc_varname_nml_bounds_amm            
      std_name_bounds(idx_nit            )  = nc_varname_nml_bounds_nit            
      std_name_bounds(idx_phos           )  = nc_varname_nml_bounds_phos           
      std_name_bounds(idx_sil            )  = nc_varname_nml_bounds_sil            
      std_name_bounds(idx_dia            )  = nc_varname_nml_bounds_dia            
      std_name_bounds(idx_flag           )  = nc_varname_nml_bounds_flag           
      std_name_bounds(idx_cyano          )  = nc_varname_nml_bounds_cyano          
      std_name_bounds(idx_mez            )  = nc_varname_nml_bounds_mez            
      std_name_bounds(idx_miz            )  = nc_varname_nml_bounds_miz            
      std_name_bounds(idx_det            )  = nc_varname_nml_bounds_det            
      std_name_bounds(idx_dets           )  = nc_varname_nml_bounds_dets           
      std_name_bounds(idx_ldon           )  = nc_varname_nml_bounds_ldon           
      std_name_bounds(idx_oxy            )  = nc_varname_nml_bounds_oxy            
      std_name_bounds(idx_amm_with_totatmos_N)  = nc_varname_nml_bounds_amm_with_totatmos_N
      std_name_bounds(idx_nit_with_totatmos_N)  = nc_varname_nml_bounds_nit_with_totatmos_N
      std_name_bounds(idx_dia_with_totatmos_N)  = nc_varname_nml_bounds_dia_with_totatmos_N
      std_name_bounds(idx_flag_with_totatmos_N)  = nc_varname_nml_bounds_flag_with_totatmos_N
      std_name_bounds(idx_cyano_with_totatmos_N)  = nc_varname_nml_bounds_cyano_with_totatmos_N
      std_name_bounds(idx_mez_with_totatmos_N)  = nc_varname_nml_bounds_mez_with_totatmos_N
      std_name_bounds(idx_miz_with_totatmos_N)  = nc_varname_nml_bounds_miz_with_totatmos_N
      std_name_bounds(idx_det_with_totatmos_N)  = nc_varname_nml_bounds_det_with_totatmos_N
      std_name_bounds(idx_ldon_with_totatmos_N)  = nc_varname_nml_bounds_ldon_with_totatmos_N
      std_name_bounds(idx_amm_with_agri_N)  = nc_varname_nml_bounds_amm_with_agri_N
      std_name_bounds(idx_nit_with_agri_N)  = nc_varname_nml_bounds_nit_with_agri_N
      std_name_bounds(idx_dia_with_agri_N)  = nc_varname_nml_bounds_dia_with_agri_N
      std_name_bounds(idx_flag_with_agri_N)  = nc_varname_nml_bounds_flag_with_agri_N
      std_name_bounds(idx_cyano_with_agri_N)  = nc_varname_nml_bounds_cyano_with_agri_N
      std_name_bounds(idx_mez_with_agri_N)  = nc_varname_nml_bounds_mez_with_agri_N
      std_name_bounds(idx_miz_with_agri_N)  = nc_varname_nml_bounds_miz_with_agri_N
      std_name_bounds(idx_det_with_agri_N)  = nc_varname_nml_bounds_det_with_agri_N
      std_name_bounds(idx_ldon_with_agri_N)  = nc_varname_nml_bounds_ldon_with_agri_N
    else
      std_name_bounds(idx_amm            )  = 'mole_concentration_of_ammonium_in_seawater'
      std_name_bounds(idx_nit            )  = 'mole_concentration_of_nitrate_in_seawater'
      std_name_bounds(idx_phos           )  = 'mole_concentration_of_phosphate_in_seawater'
      std_name_bounds(idx_sil            )  = 'mole_concentration_of_silicate_in_seawater'
      std_name_bounds(idx_dia            )  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_flag           )  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_cyano          )  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_mez            )  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_miz            )  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_det            )  = 'mole_concentration_of_detritus_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_dets           )  = 'mole_concentration_of_detritus_expressed_as_silicate_in_seawater'
      std_name_bounds(idx_ldon           )  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_oxy            )  = 'mole_concentration_of_oxygen_in_seawater'
      std_name_bounds(idx_amm_with_totatmos_N)  = 'mole_concentration_of_ammonium_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_nit_with_totatmos_N)  = 'mole_concentration_of_nitrate_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_dia_with_totatmos_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_flag_with_totatmos_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_cyano_with_totatmos_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_mez_with_totatmos_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_miz_with_totatmos_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_det_with_totatmos_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_bounds(idx_ldon_with_totatmos_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_atmospheric_deposition_expressed_as_nitrogen_in_seawater'
      std_name_bounds(idx_amm_with_agri_N)  = 'mole_concentration_of_ammonium_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_nit_with_agri_N)  = 'mole_concentration_of_nitrate_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_dia_with_agri_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_flag_with_agri_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_cyano_with_agri_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_mez_with_agri_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_miz_with_agri_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_det_with_agri_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_bounds(idx_ldon_with_agri_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_agricultural_activities_expressed_as_nitrogen_in_seawater'
    endif
    
    if ( read_netcdf_bounds_notag .or. .not. tag_river_infl ) then
    endif
    
    if ( read_netcdf_bounds_notag .or. .not. tag_atmos_dep ) then
    endif
  
  end subroutine get_standard_name_bounds

!===============================================================================

  subroutine get_standard_name_river_infl(std_name_river)
  
    use bioparam, only :  use_nc_varname_nml_river_infl,                &
                          tag_river_infl,                             &
                         idx_amm            ,                         &
                         idx_nit            ,                         &
                         idx_phos           ,                         &
                         idx_sil            ,                         &
                         idx_dia            ,                         &
                         idx_flag           ,                         &
                         idx_cyano          ,                         &
                         idx_mez            ,                         &
                         idx_miz            ,                         &
                         idx_det            ,                         &
                         idx_dets           ,                         &
                         idx_ldon           ,                         &
                         idx_oxy            ,                         &
                         idx_amm_with_totatmos_N,                         &
                         idx_nit_with_totatmos_N,                         &
                         idx_dia_with_totatmos_N,                         &
                         idx_flag_with_totatmos_N,                         &
                         idx_cyano_with_totatmos_N,                         &
                         idx_mez_with_totatmos_N,                         &
                         idx_miz_with_totatmos_N,                         &
                         idx_det_with_totatmos_N,                         &
                         idx_ldon_with_totatmos_N,                         &
                         idx_amm_with_agri_N,                         &
                         idx_nit_with_agri_N,                         &
                         idx_dia_with_agri_N,                         &
                         idx_flag_with_agri_N,                         &
                         idx_cyano_with_agri_N,                         &
                         idx_mez_with_agri_N,                         &
                         idx_miz_with_agri_N,                         &
                         idx_det_with_agri_N,                         &
                         idx_ldon_with_agri_N,                         &
                         nc_varname_nml_river_infl_amm            ,            &
                         nc_varname_nml_river_infl_nit            ,            &
                         nc_varname_nml_river_infl_phos           ,            &
                         nc_varname_nml_river_infl_sil            ,            &
                         nc_varname_nml_river_infl_dia            ,            &
                         nc_varname_nml_river_infl_flag           ,            &
                         nc_varname_nml_river_infl_cyano          ,            &
                         nc_varname_nml_river_infl_mez            ,            &
                         nc_varname_nml_river_infl_miz            ,            &
                         nc_varname_nml_river_infl_det            ,            &
                         nc_varname_nml_river_infl_dets           ,            &
                         nc_varname_nml_river_infl_ldon           ,            &
                         nc_varname_nml_river_infl_oxy            ,            &
                         nc_varname_nml_river_infl_amm_with_totatmos_N,            &
                         nc_varname_nml_river_infl_nit_with_totatmos_N,            &
                         nc_varname_nml_river_infl_dia_with_totatmos_N,            &
                         nc_varname_nml_river_infl_flag_with_totatmos_N,            &
                         nc_varname_nml_river_infl_cyano_with_totatmos_N,            &
                         nc_varname_nml_river_infl_mez_with_totatmos_N,            &
                         nc_varname_nml_river_infl_miz_with_totatmos_N,            &
                         nc_varname_nml_river_infl_det_with_totatmos_N,            &
                         nc_varname_nml_river_infl_ldon_with_totatmos_N,            &
                         nc_varname_nml_river_infl_amm_with_agri_N,            &
                         nc_varname_nml_river_infl_nit_with_agri_N,            &
                         nc_varname_nml_river_infl_dia_with_agri_N,            &
                         nc_varname_nml_river_infl_flag_with_agri_N,            &
                         nc_varname_nml_river_infl_cyano_with_agri_N,            &
                         nc_varname_nml_river_infl_mez_with_agri_N,            &
                         nc_varname_nml_river_infl_miz_with_agri_N,            &
                         nc_varname_nml_river_infl_det_with_agri_N,            &
                         nc_varname_nml_river_infl_ldon_with_agri_N,            &
                         ntbio 
  
    implicit none
  
    character(len=120), intent(out) :: std_name_river(:)
    
    if (use_nc_varname_nml_river_infl) then
      std_name_river(idx_amm            )  = nc_varname_nml_river_infl_amm            
      std_name_river(idx_nit            )  = nc_varname_nml_river_infl_nit            
      std_name_river(idx_phos           )  = nc_varname_nml_river_infl_phos           
      std_name_river(idx_sil            )  = nc_varname_nml_river_infl_sil            
      std_name_river(idx_dia            )  = nc_varname_nml_river_infl_dia            
      std_name_river(idx_flag           )  = nc_varname_nml_river_infl_flag           
      std_name_river(idx_cyano          )  = nc_varname_nml_river_infl_cyano          
      std_name_river(idx_mez            )  = nc_varname_nml_river_infl_mez            
      std_name_river(idx_miz            )  = nc_varname_nml_river_infl_miz            
      std_name_river(idx_det            )  = nc_varname_nml_river_infl_det            
      std_name_river(idx_dets           )  = nc_varname_nml_river_infl_dets           
      std_name_river(idx_ldon           )  = nc_varname_nml_river_infl_ldon           
      std_name_river(idx_oxy            )  = nc_varname_nml_river_infl_oxy            
      std_name_river(idx_amm_with_totatmos_N)  = nc_varname_nml_river_infl_amm_with_totatmos_N
      std_name_river(idx_nit_with_totatmos_N)  = nc_varname_nml_river_infl_nit_with_totatmos_N
      std_name_river(idx_dia_with_totatmos_N)  = nc_varname_nml_river_infl_dia_with_totatmos_N
      std_name_river(idx_flag_with_totatmos_N)  = nc_varname_nml_river_infl_flag_with_totatmos_N
      std_name_river(idx_cyano_with_totatmos_N)  = nc_varname_nml_river_infl_cyano_with_totatmos_N
      std_name_river(idx_mez_with_totatmos_N)  = nc_varname_nml_river_infl_mez_with_totatmos_N
      std_name_river(idx_miz_with_totatmos_N)  = nc_varname_nml_river_infl_miz_with_totatmos_N
      std_name_river(idx_det_with_totatmos_N)  = nc_varname_nml_river_infl_det_with_totatmos_N
      std_name_river(idx_ldon_with_totatmos_N)  = nc_varname_nml_river_infl_ldon_with_totatmos_N
      std_name_river(idx_amm_with_agri_N)  = nc_varname_nml_river_infl_amm_with_agri_N
      std_name_river(idx_nit_with_agri_N)  = nc_varname_nml_river_infl_nit_with_agri_N
      std_name_river(idx_dia_with_agri_N)  = nc_varname_nml_river_infl_dia_with_agri_N
      std_name_river(idx_flag_with_agri_N)  = nc_varname_nml_river_infl_flag_with_agri_N
      std_name_river(idx_cyano_with_agri_N)  = nc_varname_nml_river_infl_cyano_with_agri_N
      std_name_river(idx_mez_with_agri_N)  = nc_varname_nml_river_infl_mez_with_agri_N
      std_name_river(idx_miz_with_agri_N)  = nc_varname_nml_river_infl_miz_with_agri_N
      std_name_river(idx_det_with_agri_N)  = nc_varname_nml_river_infl_det_with_agri_N
      std_name_river(idx_ldon_with_agri_N)  = nc_varname_nml_river_infl_ldon_with_agri_N
    else
      std_name_river(idx_amm            )  = 'mole_concentration_of_ammonium_in_seawater'
      std_name_river(idx_nit            )  = 'mole_concentration_of_nitrate_in_seawater'
      std_name_river(idx_phos           )  = 'mole_concentration_of_phosphate_in_seawater'
      std_name_river(idx_sil            )  = 'mole_concentration_of_silicate_in_seawater'
      std_name_river(idx_dia            )  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_flag           )  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_cyano          )  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_mez            )  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_miz            )  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_det            )  = 'mole_concentration_of_detritus_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_dets           )  = 'mole_concentration_of_detritus_expressed_as_silicate_in_seawater'
      std_name_river(idx_ldon           )  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_oxy            )  = 'mole_concentration_of_oxygen_in_seawater'
      std_name_river(idx_amm_with_totatmos_N)  = 'mole_concentration_of_ammonium_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_nit_with_totatmos_N)  = 'mole_concentration_of_nitrate_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_dia_with_totatmos_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_flag_with_totatmos_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_cyano_with_totatmos_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_mez_with_totatmos_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_miz_with_totatmos_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_det_with_totatmos_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_atmospheric_deposition_in_seawater'
      std_name_river(idx_ldon_with_totatmos_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_atmospheric_deposition_expressed_as_nitrogen_in_seawater'
      std_name_river(idx_amm_with_agri_N)  = 'mole_concentration_of_ammonium_from_agricultural_activities_in_seawater'
      std_name_river(idx_nit_with_agri_N)  = 'mole_concentration_of_nitrate_from_agricultural_activities_in_seawater'
      std_name_river(idx_dia_with_agri_N)  = 'mole_concentration_of_diatoms_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_river(idx_flag_with_agri_N)  = 'mole_concentration_of_flagellates_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_river(idx_cyano_with_agri_N)  = 'mole_concentration_of_cyanobacteria_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_river(idx_mez_with_agri_N)  = 'mole_concentration_of_zooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_river(idx_miz_with_agri_N)  = 'mole_concentration_of_protozooplankton_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_river(idx_det_with_agri_N)  = 'mole_concentration_of_detritus_expressed_as_nitrogen_from_agricultural_activities_in_seawater'
      std_name_river(idx_ldon_with_agri_N)  = 'mole_concentration_of_labile_dissolved_organic_nitrogen_from_agricultural_activities_expressed_as_nitrogen_in_seawater'
    endif
    
    if ( .not. tag_river_infl ) then
    endif
  
  end subroutine get_standard_name_river_infl

!===============================================================================

  subroutine get_standard_name_atmos_dep(std_name_atmos)
  
    use bioparam, only :  use_nc_varname_nml_atmos_dep,                 &
                          tag_atmos_dep,                              &
                         idx_amm            ,                         &
                         idx_nit            ,                         &
                         idx_phos           ,                         &
                         idx_sil            ,                         &
                         idx_dia            ,                         &
                         idx_flag           ,                         &
                         idx_cyano          ,                         &
                         idx_mez            ,                         &
                         idx_miz            ,                         &
                         idx_det            ,                         &
                         idx_dets           ,                         &
                         idx_ldon           ,                         &
                         idx_oxy            ,                         &
                         idx_amm_with_totatmos_N,                         &
                         idx_nit_with_totatmos_N,                         &
                         idx_dia_with_totatmos_N,                         &
                         idx_flag_with_totatmos_N,                         &
                         idx_cyano_with_totatmos_N,                         &
                         idx_mez_with_totatmos_N,                         &
                         idx_miz_with_totatmos_N,                         &
                         idx_det_with_totatmos_N,                         &
                         idx_ldon_with_totatmos_N,                         &
                         idx_amm_with_agri_N,                         &
                         idx_nit_with_agri_N,                         &
                         idx_dia_with_agri_N,                         &
                         idx_flag_with_agri_N,                         &
                         idx_cyano_with_agri_N,                         &
                         idx_mez_with_agri_N,                         &
                         idx_miz_with_agri_N,                         &
                         idx_det_with_agri_N,                         &
                         idx_ldon_with_agri_N,                         &
                         nc_varname_nml_atmos_dep_amm            ,            &
                         nc_varname_nml_atmos_dep_nit            ,            &
                         nc_varname_nml_atmos_dep_phos           ,            &
                         nc_varname_nml_atmos_dep_sil            ,            &
                         nc_varname_nml_atmos_dep_dia            ,            &
                         nc_varname_nml_atmos_dep_flag           ,            &
                         nc_varname_nml_atmos_dep_cyano          ,            &
                         nc_varname_nml_atmos_dep_mez            ,            &
                         nc_varname_nml_atmos_dep_miz            ,            &
                         nc_varname_nml_atmos_dep_det            ,            &
                         nc_varname_nml_atmos_dep_dets           ,            &
                         nc_varname_nml_atmos_dep_ldon           ,            &
                         nc_varname_nml_atmos_dep_oxy            ,            &
                         nc_varname_nml_atmos_dep_amm_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_nit_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_dia_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_flag_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_cyano_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_mez_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_miz_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_det_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_ldon_with_totatmos_N,            &
                         nc_varname_nml_atmos_dep_amm_with_agri_N,            &
                         nc_varname_nml_atmos_dep_nit_with_agri_N,            &
                         nc_varname_nml_atmos_dep_dia_with_agri_N,            &
                         nc_varname_nml_atmos_dep_flag_with_agri_N,            &
                         nc_varname_nml_atmos_dep_cyano_with_agri_N,            &
                         nc_varname_nml_atmos_dep_mez_with_agri_N,            &
                         nc_varname_nml_atmos_dep_miz_with_agri_N,            &
                         nc_varname_nml_atmos_dep_det_with_agri_N,            &
                         nc_varname_nml_atmos_dep_ldon_with_agri_N,            &
                         ntbio 
  
    implicit none
  
    character(len=120), intent(out) :: std_name_atmos(:)
    
    if (use_nc_varname_nml_atmos_dep) then
      std_name_atmos(idx_amm            )  = nc_varname_nml_atmos_dep_amm            
      std_name_atmos(idx_nit            )  = nc_varname_nml_atmos_dep_nit            
      std_name_atmos(idx_phos           )  = nc_varname_nml_atmos_dep_phos           
      std_name_atmos(idx_sil            )  = nc_varname_nml_atmos_dep_sil            
      std_name_atmos(idx_dia            )  = nc_varname_nml_atmos_dep_dia            
      std_name_atmos(idx_flag           )  = nc_varname_nml_atmos_dep_flag           
      std_name_atmos(idx_cyano          )  = nc_varname_nml_atmos_dep_cyano          
      std_name_atmos(idx_mez            )  = nc_varname_nml_atmos_dep_mez            
      std_name_atmos(idx_miz            )  = nc_varname_nml_atmos_dep_miz            
      std_name_atmos(idx_det            )  = nc_varname_nml_atmos_dep_det            
      std_name_atmos(idx_dets           )  = nc_varname_nml_atmos_dep_dets           
      std_name_atmos(idx_ldon           )  = nc_varname_nml_atmos_dep_ldon           
      std_name_atmos(idx_oxy            )  = nc_varname_nml_atmos_dep_oxy            
      std_name_atmos(idx_amm_with_totatmos_N)  = nc_varname_nml_atmos_dep_amm_with_totatmos_N
      std_name_atmos(idx_nit_with_totatmos_N)  = nc_varname_nml_atmos_dep_nit_with_totatmos_N
      std_name_atmos(idx_dia_with_totatmos_N)  = nc_varname_nml_atmos_dep_dia_with_totatmos_N
      std_name_atmos(idx_flag_with_totatmos_N)  = nc_varname_nml_atmos_dep_flag_with_totatmos_N
      std_name_atmos(idx_cyano_with_totatmos_N)  = nc_varname_nml_atmos_dep_cyano_with_totatmos_N
      std_name_atmos(idx_mez_with_totatmos_N)  = nc_varname_nml_atmos_dep_mez_with_totatmos_N
      std_name_atmos(idx_miz_with_totatmos_N)  = nc_varname_nml_atmos_dep_miz_with_totatmos_N
      std_name_atmos(idx_det_with_totatmos_N)  = nc_varname_nml_atmos_dep_det_with_totatmos_N
      std_name_atmos(idx_ldon_with_totatmos_N)  = nc_varname_nml_atmos_dep_ldon_with_totatmos_N
      std_name_atmos(idx_amm_with_agri_N)  = nc_varname_nml_atmos_dep_amm_with_agri_N
      std_name_atmos(idx_nit_with_agri_N)  = nc_varname_nml_atmos_dep_nit_with_agri_N
      std_name_atmos(idx_dia_with_agri_N)  = nc_varname_nml_atmos_dep_dia_with_agri_N
      std_name_atmos(idx_flag_with_agri_N)  = nc_varname_nml_atmos_dep_flag_with_agri_N
      std_name_atmos(idx_cyano_with_agri_N)  = nc_varname_nml_atmos_dep_cyano_with_agri_N
      std_name_atmos(idx_mez_with_agri_N)  = nc_varname_nml_atmos_dep_mez_with_agri_N
      std_name_atmos(idx_miz_with_agri_N)  = nc_varname_nml_atmos_dep_miz_with_agri_N
      std_name_atmos(idx_det_with_agri_N)  = nc_varname_nml_atmos_dep_det_with_agri_N
      std_name_atmos(idx_ldon_with_agri_N)  = nc_varname_nml_atmos_dep_ldon_with_agri_N
    else
      std_name_atmos(idx_amm            )  = ''
      std_name_atmos(idx_nit            )  = ''
      std_name_atmos(idx_phos           )  = ''
      std_name_atmos(idx_sil            )  = ''
      std_name_atmos(idx_dia            )  = ''
      std_name_atmos(idx_flag           )  = ''
      std_name_atmos(idx_cyano          )  = ''
      std_name_atmos(idx_mez            )  = ''
      std_name_atmos(idx_miz            )  = ''
      std_name_atmos(idx_det            )  = ''
      std_name_atmos(idx_dets           )  = ''
      std_name_atmos(idx_ldon           )  = ''
      std_name_atmos(idx_oxy            )  = ''
      std_name_atmos(idx_amm_with_totatmos_N)  = ''
      std_name_atmos(idx_nit_with_totatmos_N)  = ''
      std_name_atmos(idx_dia_with_totatmos_N)  = ''
      std_name_atmos(idx_flag_with_totatmos_N)  = ''
      std_name_atmos(idx_cyano_with_totatmos_N)  = ''
      std_name_atmos(idx_mez_with_totatmos_N)  = ''
      std_name_atmos(idx_miz_with_totatmos_N)  = ''
      std_name_atmos(idx_det_with_totatmos_N)  = ''
      std_name_atmos(idx_ldon_with_totatmos_N)  = ''
      std_name_atmos(idx_amm_with_agri_N)  = ''
      std_name_atmos(idx_nit_with_agri_N)  = ''
      std_name_atmos(idx_dia_with_agri_N)  = ''
      std_name_atmos(idx_flag_with_agri_N)  = ''
      std_name_atmos(idx_cyano_with_agri_N)  = ''
      std_name_atmos(idx_mez_with_agri_N)  = ''
      std_name_atmos(idx_miz_with_agri_N)  = ''
      std_name_atmos(idx_det_with_agri_N)  = ''
      std_name_atmos(idx_ldon_with_agri_N)  = ''
    ! NEUMANND: todo, remove pox in future
      std_name_atmos(idx_amm) = 'tendency_of_ocean_mole_content_of_reduced_nitrogen_due_to_deposition'
      std_name_atmos(idx_nit) = 'tendency_of_ocean_mole_content_of_oxidized_nitrogen_due_to_deposition'
      std_name_atmos(idx_amm_with_totatmos_N) = 'tendency_of_ocean_mole_content_of_reduced_nitrogen_from_atmospheric_deposition_due_to_deposition'
      std_name_atmos(idx_nit_with_totatmos_N) = 'tendency_of_ocean_mole_content_of_oxidized_nitrogen_from_atmospheric_deposition_due_to_deposition'
      std_name_atmos(idx_amm_with_agri_N) = 'tendency_of_ocean_mole_content_of_reduced_nitrogen_from_agricultural_activities_due_to_deposition'
      std_name_atmos(idx_nit_with_agri_N) = 'tendency_of_ocean_mole_content_of_oxidized_nitrogen_from_agricultural_activities_due_to_deposition'
    endif
    
    if ( .not. tag_atmos_dep ) then
    endif
  
  end subroutine get_standard_name_atmos_dep

!===============================================================================

#else
  !-----------------------------------------------------------------------------
  !
  ! Please note: The subroutines below are just interfaces that enables 
  !              compilation of the source code even without netcdf. We will 
  !              allow unused variables to be present in our code for these 
  !              intent(in) and inten(inout) arguments here for now (at least
  !              as long IO is not completely revised)
  !
  !-----------------------------------------------------------------------------
!===============================================================================

  subroutine netcdf_init (iact, mmx, nmx, kmx, msrf, mcol, kh, ntbio, ntben,   &
                          ptc, benthos, iu06)

    implicit none

    integer(4), intent(in)    :: mmx, nmx, kmx, iu06
    integer(4), intent(in)    :: iact, ntbio, ntben
    integer(4), intent(in)    :: msrf(0:,0:), mcol(0:), kh(0:)
    ! real(8),    intent(in)    :: Nnorm, Onorm
    real(8),    intent(inout) :: benthos(0:,1:) 
    real(8),    intent(inout) :: ptc(0:,1:)

    write(iu06,*) 'WARNING: No netcdf-support for inital fields ...'
    write(iu06,*) '   ...continue without                          '
  end subroutine netcdf_init
 
!===============================================================================

  subroutine ReadNetcdfClimaloads( day, nr, iu06, ia, load )    

    implicit none

    integer(4), intent(in)   :: day, nr, iu06, ia
    real(8), intent(inout)   :: load(:,:,0:)
    
    write(iu06,*) 'WARNING: No netcdf-support for riverloads. ...  '
    write(iu06,*) '   ...continue without                          '
  end subroutine ReadNetcdfClimaloads

!===============================================================================

  subroutine ReadNetcdfBounds (nz, iu06, ntbio, kmx, rwzke_a)

    implicit none

    integer(4), intent(in)    :: nz, iu06, ntbio, kmx
    real(8),    intent(inout) :: rwzke_a(:,:,0:,:)
    
    write(iu06,*) 'WARNING: No netcdf-support for open boundaries ...  '
    write(iu06,*) '   ...continue without                              '
  end subroutine ReadNetcdfBounds

!===============================================================================

  subroutine ReadNETCDFatm_dep(iu06, ia,                         &
                             srfnredflx_const,                &
                             srfnoxflx_const,                &
                             srfpoxflx_const,                &
                             nmx,mmx,                         &
                               srfammflx,                     &
                               srfnitflx,                     &
                               srfphosflx,                     &
                               srfamm_with_totatmos_Nflx,                     &
                               srfnit_with_totatmos_Nflx,                     &
                               srfamm_with_agri_Nflx,                     &
                               srfnit_with_agri_Nflx,                     &
                               msrf)
      
    implicit none
    
    integer(4), intent(in)    :: iu06, ia, nmx, mmx, msrf(0:,0:)
    real(8),    intent(in)    :: srfnredflx_const
    real(8),    intent(in)    :: srfnoxflx_const
    real(8),    intent(in)    :: srfpoxflx_const
    real(8),    intent(out)   :: srfammflx(0:)
    real(8),    intent(out)   :: srfnitflx(0:)
    real(8),    intent(out)   :: srfphosflx(0:)
    real(8),    intent(out)   :: srfamm_with_totatmos_Nflx(0:)
    real(8),    intent(out)   :: srfnit_with_totatmos_Nflx(0:)
    real(8),    intent(out)   :: srfamm_with_agri_Nflx(0:)
    real(8),    intent(out)   :: srfnit_with_agri_Nflx(0:)
   
    write(iu06,*) 'WARNING: No netcdf-support for atmospheric deposition ...'
    write(iu06,*) '   ...continue with default values                       '
    srfammflx(0:) =  srfnredflx_const
    srfnitflx(0:) =  srfnoxflx_const
    srfphosflx(0:) =  srfpoxflx_const
    ! see above a note "For ERGOM CGT users".
  end subroutine ReadNETCDFatm_dep  

#endif

end module bionetcdf_input
