!==============================================================================!
!
! parameters for the biogeochemical model
!
!==============================================================================!

module bioparam 

  
  implicit none
  private
  save

!==============================================================================!
!
!   Baltic/Norh Sea         Parameter Block
!
!  different params to be defined for other areas, water types etc.
!
!       NOTE:
!       All timeunits are at the moment in days
!
!==============================================================================!
  integer(4),PARAMETER :: ntbio        = 31            ! No. of biological fields
  integer(4),PARAMETER :: ntben        = 4             ! No. of benthos fields
  integer(4),parameter :: ntall        = ntbio + ntben ! No. of bio and ben fields
  integer(4),parameter :: ntbio_notneg = 12            ! No. of bio fields that should not be negative
  integer(4),parameter :: ntben_notneg = 2             ! No. of ben fields that should not be negative
  integer(4),parameter :: ntall_notneg = ntbio_notneg + ntben_notneg ! No. of bio and ben fields
  integer(4),parameter :: ntbio_nooxy  = 30            ! No. of biological fields except of oxygen
  integer(4),PARAMETER :: ntbio_notag  = 13            ! No. of uncolored biological fields
  integer(4),PARAMETER :: ntben_notag  = 2             ! No. of uncolored benthos fields
  !-
  logical, public      :: ppispmmod       = .false. ! SPM paramterisation in PAR
  logical, public      :: par_calc_don    = .false. ! PAR calculation Neumann 2015
  logical, public      :: npread          = .false. ! Spatially varying NP ratio
  logical, public      :: adread          = .false. ! Spatially varying atm.depos.
  logical, public      :: impl            = .false. ! Implicit scheme
  logical, public      :: impl_cgt        = .false.  ! use cgt implicit scheme 
                                                  ! only considered if impl=.true.
  logical, public      :: warn_bad_solver = .false. ! throw warning on neg. 
                                                    !     results
  logical, public      :: resuspension    = .false. ! Sediment resuspension
  logical, public      :: do_diag_chl     = .false. ! Switch for diagnostic chl
  logical, public      :: do_diag_pp      = .false. ! Switch for diagnostic PP
  logical, public      :: do_diag_secchi  = .false. ! Switch for diag. Secchidepth
  
  real(8), public      :: nb              = 0.01_8      ! Loss rate P to N
  real(8), public      :: deltao          = 0.02_8      ! Loss Rate PP to D
  real(8), public      :: dn              = 0.01_8      ! Loss Rate Detritus to N
  real(8), public      :: ds              = 0.004_8     ! Loss Rate S-Detritus to S
  real(8), public      :: ksdenit         = 0.1_8       ! Half-sat. inhib NO3 denit
  real(8), public      :: ksnit           = 0.01_8      ! Half-sat. O2 nitrification
  real(8), public      :: rnit            = 0.1_8       ! Max nitrification rate at T0
  real(8), public      :: rp0             = 1.0_8       ! Max dia uptake rate at T0
  real(8), public      :: rf0             = 0.7_8       ! Max Flag uptake rate at T0
  real(8), public      :: rb0             = 0.5_8       ! Max cyano uptake rate at T0
  real(8), public      :: cyanotll        = 15.0_8      ! Lower T limit Cyanos
  real(8), public      :: cyanosul        = 12.0_8      ! Upper S limit Cyanos
  real(8), public      :: flagtll         = 0.06_8      ! flag temp scaling
  real(8), public      :: flagtsc         = 0.8_8       ! flag temp dep.
  real(8), public      :: alphap          = 0.3_8       ! Dia half-sat. constant
  real(8), public      :: alphaf          = 0.15_8      ! Flag half-sat. constant
  real(8), public      :: alphab          = 0.5_8       ! Cyano half-sat. constant
  real(8), public      :: p0              = 0.001_8     ! background values dia
  real(8), public      :: f0              = 0.001_8     ! Background values flag
  real(8), public      :: b0              = 0.001_8     ! Background values cyano
  real(8), public      :: z0              = 0.001_8     ! Background values Zoo
  real(8), public      :: min_odial       = 75.0_8      ! Min opt. dia light W/m^2
  real(8), public      :: min_oflagl      = 75.0_8      ! Min opt. flag light W/m^2
  real(8), public      :: min_ocyanol     = 75.0_8      ! Min opt. cyan light W/m^2
  real(8), public      :: att_par_1       = 0.18_8      ! Backr. light attenuation,[/m]
  real(8), public      :: att_par_2       = 0.03_8      ! Light attenuation, POM []
  real(8), public      :: q10_rec         = 0.15_8      ! Recycling temp dep [/degC]
  real(8), public      :: anit            = 0.11_8      ! Nitrification temp dep[/degC]
  real(8), public      :: rfr             = 0.072_8     ! Redfield Ratio P/N
  real(8), public      :: wdetz           = -1.5_8      ! Sinking velocity detritus m/d
  real(8), public      :: wdiaz           = -0.2_8      ! Sinking velocity dia [m/d]
  real(8), public      :: wcyanoz         = 0.1_8       ! Rising vel. cyano [m/d]
  real(8), public      :: srfpoxflx_const = 0.00047097_8  ! Atm. phosphate deposition
  real(8), public      :: srfnoxflx_const = 0.0621_8    ! default surface flux for nitrate
  real(8), public      :: srfnredflx_const = 0.0621_8    ! default surface flux for ammonium
  real(8), public      :: Kspburial       = 0.1_8       ! Half-saturation p burial Oxygen dependance
  real(8), public      :: fnitr           = 0.4_8       ! Fraction of recycled N nitrified in sediment
  real(8), public      :: dn_sed          = 0.01_8      ! Loss Rate Sediment to N
  real(8), public      :: dn_sed_anox     = 0.005_8     ! Loss Rate Sed. to N, anoxia
  real(8), public      :: q10_recs        = 0.15_8      ! Recycling temp dep sed. [/C]
  real(8), public      :: NOR             = 0.012_8     ! Nnorm/Onorm
  real(8), public      :: Nnorm_inv       = 0.2222222222_8  ! 1/Nnorm
  real(8), public      :: Onorm_inv       = 0.0026666667_8  ! 1/Onorm
  real(8), public      :: wdetsz          = -1.5_8      ! Sinking velocity silicate detritus m/d [=wdetz]
  real(8), public      :: mezgraz         = 0.2_8       ! Mesozoopl. grazing constant
  real(8), public      :: mizgraz         = 0.4_8       ! Microzoopl. grazing constant
  real(8), public      :: mezgrazf        = 0.4_8       ! Mesozoopl. food dep. constant
  real(8), public      :: mizgrazf        = 0.2_8       ! Microzoopl. food dep. const
  real(8), public      :: miztk           = 150.0_8     ! Microzoopl. grazing temp dep.
  real(8), public      :: meztk           = 150.0_8     ! Mezozoopl. grazing temp dep.
  real(8), public      :: mez_tscale      = 2.0_8       ! Mezozoopl. temp scaling
  real(8), public      :: mezcl           = 0.02_8      ! Mesozoopl. closure
  real(8), public      :: ldmez           = 0.02_8      ! Mesozoopl. mortality
  real(8), public      :: ldmiz           = 0.05_8      ! Micrezoopl. mortality
  real(8), public      :: lnmez           = 0.45_8      ! Mesozoopl. excretion
  real(8), public      :: lnmiz           = 0.3_8       ! Microzoopl. excretion
  real(8), public      :: mezprefdia      = 1.0_8       ! Preference of mezoo on dia
  real(8), public      :: mezprefflag     = 0.3_8       ! Preference of mezoo on flag
  real(8), public      :: mezprefcyano    = 0.3_8       ! Preference of mezoo on cyano
  real(8), public      :: mezprefmiz      = 1.0_8       ! Preference of mezoo on mizoo
  real(8), public      :: mizprefdia      = 0.3_8       ! Preference of mizoo on dia
  real(8), public      :: mizprefflag     = 1.0_8       ! Preference of mizoo on flag
  real(8), public      :: mizprefcyano    = 0.3_8       ! Preference of mizoo on cyano
  real(8), public      :: ldon_frac       = 0.1_8       ! Fraction of produced lDON
  real(8), public      :: ldon_tor        = 0.03_8      ! lDON turning over rate [d^-1]
  real(8), public      :: secchi_exp      = -0.936_8    ! secchi exponent
  real(8), public      :: secchi_coeff    = 1.614_8     ! secchi coef
  real(8),PARAMETER    :: Nnorm           = 4.5_8       ! Normalizing const. Nutrients
  real(8),PARAMETER    :: Onorm           = 375.0_8     ! Normalizing const. Oxygen
  real(8),PARAMETER    :: ONnitr          = 2.0_8       ! Stoechiometric O/N ratio nitr
  real(8),PARAMETER    :: NNdenit         = 5.3_8       ! Stoechiometric Norg/Nit denit
  real(8),PARAMETER    :: ONniup          = 8.625_8     ! Stoech. O/N ratio nitr uptake
  real(8),PARAMETER    :: ONamup          = 6.625_8     ! Stoech. O/N ratio amm up/rel
  real(8),PARAMETER    :: rfs             = 0.94_8      ! Redfield Ratio S/N
  real(8),PARAMETER    :: zero            = 0.0_8       ! zero
  real(8),PARAMETER    :: one             = 1.0_8       ! one
  real(8),PARAMETER    :: four            = 4.0_8       ! four
  real(8),PARAMETER    :: ten             = 10.0_8      ! ten
  real(8),PARAMETER    :: half            = 0.5_8       ! half
  real(8),PARAMETER    :: tenth           = 0.1_8       ! a tenth (1/10)

  !----------------------------------------------------------------------------!
  !  Light, secchi and lDON calculation pars according to Neumann et al (2015)
  !----------------------------------------------------------------------------!
! real(8), public      :: ldon_frac    = 0.1_8   ! Fraction of produced lDON
! real(8), public      :: ldon_tor     = 0.03_8  ! lDON turning over rate [d^-1]
  real(8), public      :: att_wat      = 0.027_8 ! kw: PAR attenuation coeff
                                                 !     of water [m^-1]
  real(8), public      :: att_chl      = 0.029_8 ! kc: PAR attenuation due to
                                                 !     chlorophyll
                                                 !     [m^2 (mg Chl)^-1]
  real(8), public      :: att_det      = 0.0038_8! kDET: PAR attenuation due to
                                                 !       detritus
                                                 !       [m^2 (mg N)^-1]
  real(8), public      :: att_ldon     = 0.0009_8! kDON: PAR attenuation due to
                                                 !       lDON [m^2 (mg N)^-1]
  real(8), public      :: att_cdom     = 0.221_8 ! Calculation of kCDOM [m^-1]
  real(8), public      :: sal_coeff    = 1.26_8  !   as a function of salinity
  real(8), public      :: sal_exp      = -0.627_8
! real(8), public      :: secchi_exp   = -0.936_8! secchi exponent
! real(8), public      :: secchi_coeff = 1.614_8 ! secchi coef

  !----------------------------------------------------------------------------!
  ! Optional paramters
  !----------------------------------------------------------------------------!
  ! ppispmmod
  real(8), public      :: att_par_3    = 0.02_8  ! Light attenuation SPM 
  real(8), public      :: att_vdiff    = 0.01_8  ! Reference vert. diffusivity
  real(8), public      :: att_depth    = 64.0_8  ! Reference depth

  ! Zooplankton rates
  real(8), public      :: sigma_b      = 0.03_8 ! Loss Rate Z to Detritus

  !----------------------------------------------------------------------------!
  ! Sediment parameters 
  !----------------------------------------------------------------------------!
  real(8), public       :: reslim      = 0.018_8  ! Critical fric.v. for resusp
  real(8), public       :: resusprate  = 10.0_8   ! Resuspension rate
!  real(8), public       :: dn_sed      = 0.01_8   ! Loss Rate Sediment to N
!  real(8), public       :: dn_sed_anox = 0.005_8  ! Loss Rate Sed. to N, anoxia
!  real(8), public       :: q10_recs    = 0.15_8   ! Recycling temp dep sed. [/C]
!  real(8), public       :: fnitr       = 0.4_8    ! Fraction of recycled N
!                                                  ! nitrified in sediment
  real(8), public       :: pfrac       = 0.0_8    ! fraction of phosphate
                                                  ! buried in oxic conditions
!  real(8), public       :: kspburial   = 0.1_8    ! 

  !----------------------------------------------------------------------------!
  ! some control parameters:
  !----------------------------------------------------------------------------!
  integer(4), public :: ibiofields = 0, NDTBIOMOD = 2
  real(8),    public :: biooutdt = 3600.0_8
  logical,    public :: netcdf_rivers = .false.
  logical,    public :: netcdf_bounds = .false.
  logical,    public :: netcdf_atm    = .false.
  
  ! use variable names defined in nc_varnames.nml (neumannd)
  logical, public    :: use_nc_varname_nml_init = .FALSE.
  logical, public    :: use_nc_varname_nml_init_ben = .FALSE.
  logical, public    :: use_nc_varname_nml_bounds = .FALSE.
  logical, public    :: use_nc_varname_nml_atmos_dep = .FALSE.
  logical, public    :: use_nc_varname_nml_river_infl = .FALSE.
  logical, public    :: use_nc_varname_nml_out = .FALSE.
  logical, public    :: use_nc_varname_nml_out_ben = .FALSE.
  
  ! Set tagged atmospheric deposition and river inflow to zero.
  ! Already tagged tracer concentrations are still considered. Thus, if
  ! we want to follow one particular rinverin inflow event, we set
  ! 'tag_river_infl' to .true. for this particular event and after that
  ! on .false..
  logical, public    :: tag_atmos_dep = .FALSE.
  logical, public    :: tag_river_infl = .FALSE.
  
  ! If these variables are .true.:
  ! Read in only un-tagged tracer concentrations. When binary files
  ! are read-in, it is assumed that only tagged tracers are contained
  ! in the binary file. Thus, ntbio_notag+ntben_notag tracers are read
  ! in. Also, if netCDF files are read, only un-tagged variables will
  ! be read from the file.
  ! Particularly, this is important, if binary files are read in,
  ! which originate from a run without tagging. Then the ben variables
  ! would be written into some bio tracers.
  logical, public    :: read_netcdf_init_notag = .FALSE.
  logical, public    :: read_netcdf_bounds_notag = .FALSE.
  logical, public    :: read_binary_restart_notag = .FALSE.
  logical, public    :: read_binary_rwzk_notag = .FALSE. 
  logical, public    :: write_binary_restart_notag = .FALSE.

  !----------------------------------------------------------------------------!
  ! index of biological params
  !----------------------------------------------------------------------------!
  integer(4), public :: idcs_bio_nooxy(1:ntbio_nooxy)  ! indices of non-oxygen tracers
  integer(4), public :: idcs_all_notneg(1:ntall_notneg) ! indices of not-negative tracers
  logical, public    :: notnegbio(1:ntbio)       ! .true. if not-negative bio tracer
  logical, public    :: notnegben(1:ntben)       ! .true. if not-negative ben tracer
  logical, public    :: notnegall(1:ntall)       ! .true. if not-negative tracer
                                                 !   first bio then ben
  integer(4), public :: idx_amm             =  1  ! ammonium
  integer(4), public :: idx_nit             =  2  ! nitrate
  integer(4), public :: idx_phos            =  3  ! phosphate
  integer(4), public :: idx_sil             =  4  ! silicate
  integer(4), public :: idx_dia             =  5  ! diatoms
  integer(4), public :: idx_flag            =  6  ! flagellates
  integer(4), public :: idx_cyano           =  7  ! cyano bacteria
  integer(4), public :: idx_mez             =  8  ! meso zooplankton
  integer(4), public :: idx_miz             =  9  ! micro zooplakton
  integer(4), public :: idx_det             =  10  ! Detritus
  integer(4), public :: idx_dets            =  11  ! S-Detritus
  integer(4), public :: idx_ldon            =  12  ! ldon is the labile DON fraction
  integer(4), public :: idx_oxy             =  13  ! Oxygen
  integer(4), public :: idx_amm_with_totatmos_N =  14  ! ammonium; containing 
  integer(4), public :: idx_nit_with_totatmos_N =  15  ! nitrate; containing 
  integer(4), public :: idx_dia_with_totatmos_N =  16  ! diatoms; containing 
  integer(4), public :: idx_flag_with_totatmos_N =  17  ! flagellates; containing 
  integer(4), public :: idx_cyano_with_totatmos_N =  18  ! cyano bacteria; containing 
  integer(4), public :: idx_mez_with_totatmos_N =  19  ! meso zooplankton; containing 
  integer(4), public :: idx_miz_with_totatmos_N =  20  ! micro zooplakton; containing 
  integer(4), public :: idx_det_with_totatmos_N =  21  ! Detritus; containing 
  integer(4), public :: idx_ldon_with_totatmos_N =  22  ! ldon is the labile DON fraction; containing 
  integer(4), public :: idx_amm_with_agri_N =  23  ! ammonium; containing 
  integer(4), public :: idx_nit_with_agri_N =  24  ! nitrate; containing 
  integer(4), public :: idx_dia_with_agri_N =  25  ! diatoms; containing 
  integer(4), public :: idx_flag_with_agri_N =  26  ! flagellates; containing 
  integer(4), public :: idx_cyano_with_agri_N =  27  ! cyano bacteria; containing 
  integer(4), public :: idx_mez_with_agri_N =  28  ! meso zooplankton; containing 
  integer(4), public :: idx_miz_with_agri_N =  29  ! micro zooplakton; containing 
  integer(4), public :: idx_det_with_agri_N =  30  ! Detritus; containing 
  integer(4), public :: idx_ldon_with_agri_N =  31  ! ldon is the labile DON fraction; containing 
  integer(4), public :: bidx_nitr            = 1  ! nitrogen in the sediment
  integer(4), public :: bidx_sili            = 2  ! silicate in the sediment
  integer(4), public :: bidx_nitr_with_totatmos_N = 3  ! nitrogen in the sediment; containing 
  integer(4), public :: bidx_nitr_with_agri_N = 4  ! nitrogen in the sediment; containing 
  !Index for diagnostic vars
  integer(4), public :: didx_chloro = 0
  integer(4), public :: didx_pp     = 0
  integer(4), public :: didx_secchi = 0
  
  !additional coldstart for bio run only
  logical, public    :: biocoldst = .FALSE.
  !switch for output of PAR
  logical, public    :: write_light = .FALSE.
  
  ! Initialization of the standard_name variables. They are read in from
  ! the nc_varnames.nml namelist file.
  ! NOTE 1: They are only used if the appropriate use_nc_varname_nml_* is set
  ! to true.
  ! NOTE 2: The default values are set in the get_standard_name_* routines
  ! of the bionetcdf_input module.
  character(len=120), public :: nc_varname_nml_init_amm             = ''
  character(len=120), public :: nc_varname_nml_init_nit             = ''
  character(len=120), public :: nc_varname_nml_init_phos            = ''
  character(len=120), public :: nc_varname_nml_init_sil             = ''
  character(len=120), public :: nc_varname_nml_init_dia             = ''
  character(len=120), public :: nc_varname_nml_init_flag            = ''
  character(len=120), public :: nc_varname_nml_init_cyano           = ''
  character(len=120), public :: nc_varname_nml_init_mez             = ''
  character(len=120), public :: nc_varname_nml_init_miz             = ''
  character(len=120), public :: nc_varname_nml_init_det             = ''
  character(len=120), public :: nc_varname_nml_init_dets            = ''
  character(len=120), public :: nc_varname_nml_init_ldon            = ''
  character(len=120), public :: nc_varname_nml_init_oxy             = ''
  character(len=120), public :: nc_varname_nml_init_amm_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_nit_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_dia_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_flag_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_cyano_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_mez_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_miz_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_det_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_ldon_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_amm_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_nit_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_dia_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_flag_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_cyano_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_mez_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_miz_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_det_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_ldon_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_init_ben_nitr            = ''
  character(len=120), public :: nc_varname_nml_init_ben_sili            = ''
  character(len=120), public :: nc_varname_nml_init_ben_nitr_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_init_ben_nitr_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_amm             = ''
  character(len=120), public :: nc_varname_nml_bounds_nit             = ''
  character(len=120), public :: nc_varname_nml_bounds_phos            = ''
  character(len=120), public :: nc_varname_nml_bounds_sil             = ''
  character(len=120), public :: nc_varname_nml_bounds_dia             = ''
  character(len=120), public :: nc_varname_nml_bounds_flag            = ''
  character(len=120), public :: nc_varname_nml_bounds_cyano           = ''
  character(len=120), public :: nc_varname_nml_bounds_mez             = ''
  character(len=120), public :: nc_varname_nml_bounds_miz             = ''
  character(len=120), public :: nc_varname_nml_bounds_det             = ''
  character(len=120), public :: nc_varname_nml_bounds_dets            = ''
  character(len=120), public :: nc_varname_nml_bounds_ldon            = ''
  character(len=120), public :: nc_varname_nml_bounds_oxy             = ''
  character(len=120), public :: nc_varname_nml_bounds_amm_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_nit_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_dia_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_flag_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_cyano_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_mez_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_miz_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_det_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_ldon_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_bounds_amm_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_nit_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_dia_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_flag_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_cyano_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_mez_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_miz_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_det_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_bounds_ldon_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_amm             = ''
  character(len=120), public :: nc_varname_nml_river_infl_nit             = ''
  character(len=120), public :: nc_varname_nml_river_infl_phos            = ''
  character(len=120), public :: nc_varname_nml_river_infl_sil             = ''
  character(len=120), public :: nc_varname_nml_river_infl_dia             = ''
  character(len=120), public :: nc_varname_nml_river_infl_flag            = ''
  character(len=120), public :: nc_varname_nml_river_infl_cyano           = ''
  character(len=120), public :: nc_varname_nml_river_infl_mez             = ''
  character(len=120), public :: nc_varname_nml_river_infl_miz             = ''
  character(len=120), public :: nc_varname_nml_river_infl_det             = ''
  character(len=120), public :: nc_varname_nml_river_infl_dets            = ''
  character(len=120), public :: nc_varname_nml_river_infl_ldon            = ''
  character(len=120), public :: nc_varname_nml_river_infl_oxy             = ''
  character(len=120), public :: nc_varname_nml_river_infl_amm_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_nit_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_dia_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_flag_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_cyano_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_mez_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_miz_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_det_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_ldon_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_amm_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_nit_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_dia_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_flag_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_cyano_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_mez_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_miz_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_det_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_river_infl_ldon_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_amm             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_nit             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_phos            = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_sil             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_dia             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_flag            = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_cyano           = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_mez             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_miz             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_det             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_dets            = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_ldon            = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_oxy             = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_amm_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_nit_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_dia_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_flag_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_cyano_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_mez_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_miz_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_det_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_ldon_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_amm_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_nit_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_dia_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_flag_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_cyano_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_mez_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_miz_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_det_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_atmos_dep_ldon_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_amm             = ''
  character(len=120), public :: nc_varname_nml_out_nit             = ''
  character(len=120), public :: nc_varname_nml_out_phos            = ''
  character(len=120), public :: nc_varname_nml_out_sil             = ''
  character(len=120), public :: nc_varname_nml_out_dia             = ''
  character(len=120), public :: nc_varname_nml_out_flag            = ''
  character(len=120), public :: nc_varname_nml_out_cyano           = ''
  character(len=120), public :: nc_varname_nml_out_mez             = ''
  character(len=120), public :: nc_varname_nml_out_miz             = ''
  character(len=120), public :: nc_varname_nml_out_det             = ''
  character(len=120), public :: nc_varname_nml_out_dets            = ''
  character(len=120), public :: nc_varname_nml_out_ldon            = ''
  character(len=120), public :: nc_varname_nml_out_oxy             = ''
  character(len=120), public :: nc_varname_nml_out_amm_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_nit_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_dia_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_flag_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_cyano_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_mez_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_miz_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_det_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_ldon_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_amm_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_nit_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_dia_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_flag_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_cyano_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_mez_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_miz_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_det_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_ldon_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_out_ben_nitr            = ''
  character(len=120), public :: nc_varname_nml_out_ben_sili            = ''
  character(len=120), public :: nc_varname_nml_out_ben_nitr_with_totatmos_N = ''
  character(len=120), public :: nc_varname_nml_out_ben_nitr_with_agri_N = ''
  character(len=120), public :: nc_varname_nml_dummy = ''
  
  ! public parameters:
  public :: Nnorm
  public :: Onorm
  public :: ONnitr
  public :: NNdenit
  public :: ONniup
  public :: ONamup
  public :: rfs
  public :: zero
  public :: one
  public :: four
  public :: ten
  public :: half
  public :: tenth
  PUBLIC :: ntbio, ntben, ntbio_notag, ntben_notag, ntall, ntbio_nooxy, ntbio_notneg, ntben_notneg, ntall_notneg

  ! public methods:
  public :: ReadBioparam, bioInd

! TODO, NEUMANND:
  namelist /bioparameters /nb, deltao, sigma_b, dn, ksdenit, ksnit, rp0,       &
                       rf0, rb0, cyanotll, cyanosul, flagtll, alphap, alphaf,  &
                       alphab, p0, f0, b0, z0,                                 &
                       min_odial,  min_oflagl, min_ocyanol,                    &
                       q10_rec, anit, &
                       wdetz, wdetsz, wdiaz, wcyanoz,   & 
                       dn_sed, dn_sed_anox, q10_recs, fnitr, & 
                       pfrac, kspburial, &
                       ibiofields, NDTBIOMOD, biooutdt,                        &
                       srfpoxflx_const, warn_bad_solver,                       &
                       att_vdiff, att_depth, rfr, impl, impl_cgt, ppispmmod,   &
                       par_calc_don, att_par_3, npread, adread, meztk,         &
                       reslim, resusprate, resuspension, lnmez, lnmiz,         &
                       ldmez, ldmiz, mezcl, mezprefmiz, mezprefdia, mezprefflag,&
                       mezprefcyano, mizprefdia, mizprefflag, mizprefcyano,    &
                       mez_tscale, mezgraz, mezgrazf, mizgrazf, mizgraz, miztk,&
                       flagtsc, biocoldst,                                     &
                       do_diag_pp, do_diag_chl, do_diag_secchi, write_light,   &
                       netcdf_rivers, netcdf_bounds, netcdf_atm,               &
                       use_nc_varname_nml_init, use_nc_varname_nml_init_ben,   &
                       use_nc_varname_nml_atmos_dep, use_nc_varname_nml_bounds,&
                       use_nc_varname_nml_river_infl,                          &
                       use_nc_varname_nml_out, use_nc_varname_nml_out_ben,     &
                       tag_atmos_dep, tag_river_infl,                          &
                       read_netcdf_init_notag, read_netcdf_bounds_notag,       &
                       read_binary_restart_notag, read_binary_rwzk_notag,      &
                       write_binary_restart_notag
                       
  NAMELIST /nc_varnames/            &
                            nc_varname_nml_init_amm            , &
                            nc_varname_nml_init_nit            , &
                            nc_varname_nml_init_phos           , &
                            nc_varname_nml_init_sil            , &
                            nc_varname_nml_init_dia            , &
                            nc_varname_nml_init_flag           , &
                            nc_varname_nml_init_cyano          , &
                            nc_varname_nml_init_mez            , &
                            nc_varname_nml_init_miz            , &
                            nc_varname_nml_init_det            , &
                            nc_varname_nml_init_dets           , &
                            nc_varname_nml_init_ldon           , &
                            nc_varname_nml_init_oxy            , &
                            nc_varname_nml_init_amm_with_totatmos_N, &
                            nc_varname_nml_init_nit_with_totatmos_N, &
                            nc_varname_nml_init_dia_with_totatmos_N, &
                            nc_varname_nml_init_flag_with_totatmos_N, &
                            nc_varname_nml_init_cyano_with_totatmos_N, &
                            nc_varname_nml_init_mez_with_totatmos_N, &
                            nc_varname_nml_init_miz_with_totatmos_N, &
                            nc_varname_nml_init_det_with_totatmos_N, &
                            nc_varname_nml_init_ldon_with_totatmos_N, &
                            nc_varname_nml_init_amm_with_agri_N, &
                            nc_varname_nml_init_nit_with_agri_N, &
                            nc_varname_nml_init_dia_with_agri_N, &
                            nc_varname_nml_init_flag_with_agri_N, &
                            nc_varname_nml_init_cyano_with_agri_N, &
                            nc_varname_nml_init_mez_with_agri_N, &
                            nc_varname_nml_init_miz_with_agri_N, &
                            nc_varname_nml_init_det_with_agri_N, &
                            nc_varname_nml_init_ldon_with_agri_N, &
                            nc_varname_nml_init_ben_nitr           , &
                            nc_varname_nml_init_ben_sili           , &
                            nc_varname_nml_init_ben_nitr_with_totatmos_N, &
                            nc_varname_nml_init_ben_nitr_with_agri_N, &
                            nc_varname_nml_bounds_amm            , &
                            nc_varname_nml_bounds_nit            , &
                            nc_varname_nml_bounds_phos           , &
                            nc_varname_nml_bounds_sil            , &
                            nc_varname_nml_bounds_dia            , &
                            nc_varname_nml_bounds_flag           , &
                            nc_varname_nml_bounds_cyano          , &
                            nc_varname_nml_bounds_mez            , &
                            nc_varname_nml_bounds_miz            , &
                            nc_varname_nml_bounds_det            , &
                            nc_varname_nml_bounds_dets           , &
                            nc_varname_nml_bounds_ldon           , &
                            nc_varname_nml_bounds_oxy            , &
                            nc_varname_nml_bounds_amm_with_totatmos_N, &
                            nc_varname_nml_bounds_nit_with_totatmos_N, &
                            nc_varname_nml_bounds_dia_with_totatmos_N, &
                            nc_varname_nml_bounds_flag_with_totatmos_N, &
                            nc_varname_nml_bounds_cyano_with_totatmos_N, &
                            nc_varname_nml_bounds_mez_with_totatmos_N, &
                            nc_varname_nml_bounds_miz_with_totatmos_N, &
                            nc_varname_nml_bounds_det_with_totatmos_N, &
                            nc_varname_nml_bounds_ldon_with_totatmos_N, &
                            nc_varname_nml_bounds_amm_with_agri_N, &
                            nc_varname_nml_bounds_nit_with_agri_N, &
                            nc_varname_nml_bounds_dia_with_agri_N, &
                            nc_varname_nml_bounds_flag_with_agri_N, &
                            nc_varname_nml_bounds_cyano_with_agri_N, &
                            nc_varname_nml_bounds_mez_with_agri_N, &
                            nc_varname_nml_bounds_miz_with_agri_N, &
                            nc_varname_nml_bounds_det_with_agri_N, &
                            nc_varname_nml_bounds_ldon_with_agri_N, &
                            nc_varname_nml_river_infl_amm            , &
                            nc_varname_nml_river_infl_nit            , &
                            nc_varname_nml_river_infl_phos           , &
                            nc_varname_nml_river_infl_sil            , &
                            nc_varname_nml_river_infl_dia            , &
                            nc_varname_nml_river_infl_flag           , &
                            nc_varname_nml_river_infl_cyano          , &
                            nc_varname_nml_river_infl_mez            , &
                            nc_varname_nml_river_infl_miz            , &
                            nc_varname_nml_river_infl_det            , &
                            nc_varname_nml_river_infl_dets           , &
                            nc_varname_nml_river_infl_ldon           , &
                            nc_varname_nml_river_infl_oxy            , &
                            nc_varname_nml_river_infl_amm_with_totatmos_N, &
                            nc_varname_nml_river_infl_nit_with_totatmos_N, &
                            nc_varname_nml_river_infl_dia_with_totatmos_N, &
                            nc_varname_nml_river_infl_flag_with_totatmos_N, &
                            nc_varname_nml_river_infl_cyano_with_totatmos_N, &
                            nc_varname_nml_river_infl_mez_with_totatmos_N, &
                            nc_varname_nml_river_infl_miz_with_totatmos_N, &
                            nc_varname_nml_river_infl_det_with_totatmos_N, &
                            nc_varname_nml_river_infl_ldon_with_totatmos_N, &
                            nc_varname_nml_river_infl_amm_with_agri_N, &
                            nc_varname_nml_river_infl_nit_with_agri_N, &
                            nc_varname_nml_river_infl_dia_with_agri_N, &
                            nc_varname_nml_river_infl_flag_with_agri_N, &
                            nc_varname_nml_river_infl_cyano_with_agri_N, &
                            nc_varname_nml_river_infl_mez_with_agri_N, &
                            nc_varname_nml_river_infl_miz_with_agri_N, &
                            nc_varname_nml_river_infl_det_with_agri_N, &
                            nc_varname_nml_river_infl_ldon_with_agri_N, &
                            nc_varname_nml_atmos_dep_amm            , &
                            nc_varname_nml_atmos_dep_nit            , &
                            nc_varname_nml_atmos_dep_phos           , &
                            nc_varname_nml_atmos_dep_sil            , &
                            nc_varname_nml_atmos_dep_dia            , &
                            nc_varname_nml_atmos_dep_flag           , &
                            nc_varname_nml_atmos_dep_cyano          , &
                            nc_varname_nml_atmos_dep_mez            , &
                            nc_varname_nml_atmos_dep_miz            , &
                            nc_varname_nml_atmos_dep_det            , &
                            nc_varname_nml_atmos_dep_dets           , &
                            nc_varname_nml_atmos_dep_ldon           , &
                            nc_varname_nml_atmos_dep_oxy            , &
                            nc_varname_nml_atmos_dep_amm_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_nit_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_dia_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_flag_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_cyano_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_mez_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_miz_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_det_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_ldon_with_totatmos_N, &
                            nc_varname_nml_atmos_dep_amm_with_agri_N, &
                            nc_varname_nml_atmos_dep_nit_with_agri_N, &
                            nc_varname_nml_atmos_dep_dia_with_agri_N, &
                            nc_varname_nml_atmos_dep_flag_with_agri_N, &
                            nc_varname_nml_atmos_dep_cyano_with_agri_N, &
                            nc_varname_nml_atmos_dep_mez_with_agri_N, &
                            nc_varname_nml_atmos_dep_miz_with_agri_N, &
                            nc_varname_nml_atmos_dep_det_with_agri_N, &
                            nc_varname_nml_atmos_dep_ldon_with_agri_N, &
                            nc_varname_nml_out_amm            , &
                            nc_varname_nml_out_nit            , &
                            nc_varname_nml_out_phos           , &
                            nc_varname_nml_out_sil            , &
                            nc_varname_nml_out_dia            , &
                            nc_varname_nml_out_flag           , &
                            nc_varname_nml_out_cyano          , &
                            nc_varname_nml_out_mez            , &
                            nc_varname_nml_out_miz            , &
                            nc_varname_nml_out_det            , &
                            nc_varname_nml_out_dets           , &
                            nc_varname_nml_out_ldon           , &
                            nc_varname_nml_out_oxy            , &
                            nc_varname_nml_out_amm_with_totatmos_N, &
                            nc_varname_nml_out_nit_with_totatmos_N, &
                            nc_varname_nml_out_dia_with_totatmos_N, &
                            nc_varname_nml_out_flag_with_totatmos_N, &
                            nc_varname_nml_out_cyano_with_totatmos_N, &
                            nc_varname_nml_out_mez_with_totatmos_N, &
                            nc_varname_nml_out_miz_with_totatmos_N, &
                            nc_varname_nml_out_det_with_totatmos_N, &
                            nc_varname_nml_out_ldon_with_totatmos_N, &
                            nc_varname_nml_out_amm_with_agri_N, &
                            nc_varname_nml_out_nit_with_agri_N, &
                            nc_varname_nml_out_dia_with_agri_N, &
                            nc_varname_nml_out_flag_with_agri_N, &
                            nc_varname_nml_out_cyano_with_agri_N, &
                            nc_varname_nml_out_mez_with_agri_N, &
                            nc_varname_nml_out_miz_with_agri_N, &
                            nc_varname_nml_out_det_with_agri_N, &
                            nc_varname_nml_out_ldon_with_agri_N, &
                            nc_varname_nml_out_ben_nitr           , &
                            nc_varname_nml_out_ben_sili           , &
                            nc_varname_nml_out_ben_nitr_with_totatmos_N, &
                            nc_varname_nml_out_ben_nitr_with_agri_N, &
                            nc_varname_nml_dummy

contains

!===============================================================================

  subroutine ReadBioparam(iu06,ldebug)

    use exits,          only : exitme
    use io_subs,        only : io_new_unit
    implicit none
  
    integer(4), intent(in) :: iu06
    logical,    intent(in) :: ldebug

    integer(4) :: ios, lun

    character(8),  parameter :: fmti='(a15,i3)',fmtl='(a15,l2)'
    character(10), parameter :: fmtr='(a15,f8.2)'
    character(11), parameter :: fmrr='(a15,f10.7)', fmrr2='(a15,f15.9)'

    ! read bioparams.nml
    ! NEUMANND: I renamed the bioparams.txt to bioparams.nml for reasons of
    !           consistency (It took me some time to find the bioparams 
    !           namelist because it didn't show up on `ls *.nml`). In order to
    !           comply with old model setups, first bioparams.nml is opened
    !           and then bioparams.txt.
    lun = io_new_unit()
    open(lun,file='bioparams.nml',status='old',iostat=ios)
    if (ios /= 0) then
      write(iu06,*) 'Warning: File bioparams.nml not found. Trying bioparams.txt. ...'
      write(iu06,*) '        ... If the file is also not found, continue with ...'
      write(iu06,*) '        ... default biological parameter values'
      
      open(lun,file='bioparams.txt',status='old',iostat=ios)
      if (ios /= 0) then
        write(iu06,*) 'Warning: File bioparams.txt not found. Continue with ...'
        write(iu06,*) '        ... default biological parameter values'
      endif
    endif
    
    if (ios == 0) then
      read (unit=lun, nml=bioparameters, iostat=ios)
      if (ios /= 0) then
        write(iu06,*) 'Error: Cannot read values from bioparameters namelist'
        write(iu06,*) 'Error: The iostat failure was: ', ios
        call exitme(1)
      endif
    endif
    close(lun)

    ! read nc_varnames.nml
    lun = io_new_unit()
    open(lun,file='nc_varnames.nml',status='old',iostat=ios)
    if (ios /= 0) then
      write(iu06,*) 'Warning: File nc_varnames.nml not found. Continue ...'
      write(iu06,*) '        ... with default standard names and set ...'
      write(iu06,*) '        ... use_nc_varname_nml_* to .false.'
      use_nc_varname_nml_init = .false.
      use_nc_varname_nml_init_ben = .false.
      use_nc_varname_nml_atmos_dep = .false.
      use_nc_varname_nml_river_infl = .false.
      use_nc_varname_nml_bounds = .false.
      use_nc_varname_nml_out = .false.
      use_nc_varname_nml_out_ben = .false.
    else
      read (unit=lun, nml=nc_varnames, iostat=ios)
      if (ios /= 0) then
        write(iu06,*) 'Error: Cannot read values from nc_varnames namelist'
        write(iu06,*) 'Error: The iostat failure was: ', ios
        call exitme(1)
      endif
    endif
    close(lun)
    
    !! NEUMANND: update some variables
    ! arrays that indicate whether tracer might become negative (.false.)
    !  or they have to be positive (.true.)
    notnegbio(:) = .true.
    notnegben(:) = .true.
    notnegall(:) = .true.
    idcs_all_notneg(:) = -1
    
    ! iterate positive tracers
    idcs_all_notneg(1) = idx_amm
    idcs_all_notneg(2) = idx_nit
    idcs_all_notneg(3) = idx_phos
    idcs_all_notneg(4) = idx_sil
    idcs_all_notneg(5) = idx_dia
    idcs_all_notneg(6) = idx_flag
    idcs_all_notneg(7) = idx_cyano
    idcs_all_notneg(8) = idx_mez
    idcs_all_notneg(9) = idx_miz
    idcs_all_notneg(10) = idx_det
    idcs_all_notneg(11) = idx_dets
    idcs_all_notneg(12) = idx_ldon
    idcs_all_notneg(1+ntbio_notneg) = bidx_nitr
    idcs_all_notneg(2+ntbio_notneg) = bidx_sili

    ! iterate negative tracers
    notnegbio(idx_oxy) = .false.
    notnegall(idx_oxy) = .false.
    notnegbio(idx_amm_with_totatmos_N) = .false.
    notnegall(idx_amm_with_totatmos_N) = .false.
    notnegbio(idx_nit_with_totatmos_N) = .false.
    notnegall(idx_nit_with_totatmos_N) = .false.
    notnegbio(idx_dia_with_totatmos_N) = .false.
    notnegall(idx_dia_with_totatmos_N) = .false.
    notnegbio(idx_flag_with_totatmos_N) = .false.
    notnegall(idx_flag_with_totatmos_N) = .false.
    notnegbio(idx_cyano_with_totatmos_N) = .false.
    notnegall(idx_cyano_with_totatmos_N) = .false.
    notnegbio(idx_mez_with_totatmos_N) = .false.
    notnegall(idx_mez_with_totatmos_N) = .false.
    notnegbio(idx_miz_with_totatmos_N) = .false.
    notnegall(idx_miz_with_totatmos_N) = .false.
    notnegbio(idx_det_with_totatmos_N) = .false.
    notnegall(idx_det_with_totatmos_N) = .false.
    notnegbio(idx_ldon_with_totatmos_N) = .false.
    notnegall(idx_ldon_with_totatmos_N) = .false.
    notnegbio(idx_amm_with_agri_N) = .false.
    notnegall(idx_amm_with_agri_N) = .false.
    notnegbio(idx_nit_with_agri_N) = .false.
    notnegall(idx_nit_with_agri_N) = .false.
    notnegbio(idx_dia_with_agri_N) = .false.
    notnegall(idx_dia_with_agri_N) = .false.
    notnegbio(idx_flag_with_agri_N) = .false.
    notnegall(idx_flag_with_agri_N) = .false.
    notnegbio(idx_cyano_with_agri_N) = .false.
    notnegall(idx_cyano_with_agri_N) = .false.
    notnegbio(idx_mez_with_agri_N) = .false.
    notnegall(idx_mez_with_agri_N) = .false.
    notnegbio(idx_miz_with_agri_N) = .false.
    notnegall(idx_miz_with_agri_N) = .false.
    notnegbio(idx_det_with_agri_N) = .false.
    notnegall(idx_det_with_agri_N) = .false.
    notnegbio(idx_ldon_with_agri_N) = .false.
    notnegall(idx_ldon_with_agri_N) = .false.
    notnegben(bidx_nitr_with_totatmos_N) = .false.
    notnegall(bidx_nitr_with_totatmos_N+ntbio) = .false.
    notnegben(bidx_nitr_with_agri_N) = .false.
    notnegall(bidx_nitr_with_agri_N+ntbio) = .false.
    
    ! indices of non-oxygen tracers
    !  and number of these tracers
    idcs_bio_nooxy(1) = idx_amm
    idcs_bio_nooxy(2) = idx_nit
    idcs_bio_nooxy(3) = idx_phos
    idcs_bio_nooxy(4) = idx_sil
    idcs_bio_nooxy(5) = idx_dia
    idcs_bio_nooxy(6) = idx_flag
    idcs_bio_nooxy(7) = idx_cyano
    idcs_bio_nooxy(8) = idx_mez
    idcs_bio_nooxy(9) = idx_miz
    idcs_bio_nooxy(10) = idx_det
    idcs_bio_nooxy(11) = idx_dets
    idcs_bio_nooxy(12) = idx_ldon
    idcs_bio_nooxy(13) = idx_amm_with_totatmos_N
    idcs_bio_nooxy(14) = idx_nit_with_totatmos_N
    idcs_bio_nooxy(15) = idx_dia_with_totatmos_N
    idcs_bio_nooxy(16) = idx_flag_with_totatmos_N
    idcs_bio_nooxy(17) = idx_cyano_with_totatmos_N
    idcs_bio_nooxy(18) = idx_mez_with_totatmos_N
    idcs_bio_nooxy(19) = idx_miz_with_totatmos_N
    idcs_bio_nooxy(20) = idx_det_with_totatmos_N
    idcs_bio_nooxy(21) = idx_ldon_with_totatmos_N
    idcs_bio_nooxy(22) = idx_amm_with_agri_N
    idcs_bio_nooxy(23) = idx_nit_with_agri_N
    idcs_bio_nooxy(24) = idx_dia_with_agri_N
    idcs_bio_nooxy(25) = idx_flag_with_agri_N
    idcs_bio_nooxy(26) = idx_cyano_with_agri_N
    idcs_bio_nooxy(27) = idx_mez_with_agri_N
    idcs_bio_nooxy(28) = idx_miz_with_agri_N
    idcs_bio_nooxy(29) = idx_det_with_agri_N
    idcs_bio_nooxy(30) = idx_ldon_with_agri_N
    
    ! some inverted constants in order so save computing time
    NOR             = Nnorm/Onorm
    Nnorm_inv       = one/Nnorm
    Onorm_inv       = one/Onorm
    
    

! TODO_MeRamo: tags
    if (ldebug) then
      write(iu06,'(a50)') 'Dump of namelist bioparameters from bioparams.txt:'
      write(iu06,fmrr) 'nb            =',nb ! da
      write(iu06,fmrr) 'deltao        =',deltao ! da
      write(iu06,fmrr) 'sigma_b       =',sigma_b
      write(iu06,fmrr) 'dn            =',dn ! da
      ! FEHLT: ds
      write(iu06,fmrr) 'dn_sed        =',dn_sed ! da
      write(iu06,fmrr) 'dn_sed_anox   =',dn_sed_anox ! da
      write(iu06,fmrr) 'ksdenit       =',ksdenit ! da
      write(iu06,fmrr) 'ksnit         =',ksnit ! da
      ! FEHLT: rnit
      write(iu06,fmtr) 'rp0           =',rp0 ! da
      write(iu06,fmtr) 'rf0           =',rf0 ! da
      write(iu06,fmtr) 'rb0           =',rb0 ! da
      write(iu06,fmtr) 'cyanotll      =',cyanotll ! da
      write(iu06,fmtr) 'cyanosul      =',cyanosul ! da
      write(iu06,fmtr) 'flagtll       =',flagtll ! da
      ! FEHLT: flagsc
      write(iu06,fmtr) 'alphap        =',alphap ! da
      write(iu06,fmtr) 'alphaf        =',alphaf ! da
      write(iu06,fmtr) 'alphab        =',alphab ! da
      write(iu06,fmrr) 'p0            =',p0 ! da
      write(iu06,fmrr) 'f0            =',f0 ! da
      write(iu06,fmrr) 'b0            =',b0 ! da
      write(iu06,fmrr) 'z0            =',z0 ! da
      write(iu06,fmtr) 'min_odial     =',min_odial ! da
      write(iu06,fmtr) 'min_oflagl    =',min_oflagl ! da
      write(iu06,fmtr) 'min_ocyanol   =',min_ocyanol! da
      ! FEHLT: att_par_1
      ! FEHLT: att_par_2
      ! FEHLT: secchi_exp
      ! FEHLT: secchi_coeff
      write(iu06,fmtr) 'q10_rec       =',q10_rec ! da
      write(iu06,fmtr) 'q10_recs      =',q10_recs ! da
      write(iu06,fmtr) 'anit          =',anit ! da
      ! FEHLT: Nnorm
      ! FEHLT: Onorm
      ! FEHLT: ONnitr
      ! FEHLT: NNdenit
      ! FEHLT: ONniup
      ! FEHLT: ONamup
      ! write(iu06,fmtr) 'wdz           =',wdz ! renamed to wdetz and wdetsz
      ! write(iu06,fmtr) 'wpz           =',wpz ! renamed to wdiaz
      ! write(iu06,fmtr) 'wbz           =',wbz ! renamed to wcyanoz
      write(iu06,fmtr) 'wdetz           =',wdetz ! da, neu
      write(iu06,fmtr) 'wdiaz           =',wdiaz ! da, neu
      write(iu06,fmtr) 'wcyanoz         =',wcyanoz ! da, neu
      write(iu06,fmtr) 'wdetsz          =',wdetz ! da, neu
      write(iu06,fmtr) 'fnitr         =',fnitr ! da
      write(iu06,fmtr) 'pfrac         =',pfrac
      write(iu06,fmtr) 'kspburial     =',kspburial ! da
      write(iu06,fmti) 'ibiofields    =',ibiofields
      write(iu06,fmti) 'NDTBIOMOD     =',NDTBIOMOD
      write(iu06,fmtr) 'biooutdt      =',biooutdt
      ! write(iu06,fmrr2) 'srfphosflx   =',srfphosflx ! da, renamed to srfpoxflx_const
      write(iu06,fmrr2) 'srfpoxflx_const   =',srfpoxflx_const ! da, neu
      write(iu06,fmtr) 'att_vdiff     =',att_vdiff
      write(iu06,fmtr) 'att_depth     =',att_depth
      write(iu06,fmtr) 'att_par_3     =',att_par_3
      ! FEHLT: rfs
      write(iu06,fmrr) 'rfr           =',rfr ! da
      ! FEHLT: srfnflx
      write(iu06,fmtl) 'impl          =',impl
      write(iu06,fmtl) 'impl_cgt      =',impl_cgt
      write(iu06,fmtl) 'ppispmmod     =',ppispmmod
      write(iu06,fmtl) 'par_calc_don  =',par_calc_don
      write(iu06,fmtl) 'npread        =',npread
      write(iu06,fmtl) 'adread        =',adread
      write(iu06,fmtl) 'resuspension  =',resuspension
      write(iu06,fmtr) 'meztk         =',meztk
      write(iu06,fmtr) 'miztk         =',miztk
      write(iu06,fmrr) 'reslim        =',reslim
      write(iu06,fmtr) 'resusprate    =',resusprate
      write(iu06,fmtr) 'lnmez         =',lnmez
      write(iu06,fmtr) 'lnmiz         =',lnmiz
      write(iu06,fmtr) 'ldmez         =',ldmez
      write(iu06,fmtr) 'ldmiz         =',ldmiz
      write(iu06,fmtr) 'mezcl         =',mezcl
      write(iu06,fmtr) 'mezprefmiz    =',mezprefmiz
      write(iu06,fmtr) 'mezprefdia    =',mezprefdia
      write(iu06,fmtr) 'mezprefflag   =',mezprefflag
      write(iu06,fmtr) 'mezprefcyano  =',mezprefcyano
      write(iu06,fmtr) 'mizprefdia    =',mizprefdia
      write(iu06,fmtr) 'mizprefflag   =',mizprefflag
      write(iu06,fmtr) 'mizprefcyano  =',mizprefcyano
      write(iu06,fmtr) 'mez_tscale    =',mez_tscale
      write(iu06,fmtr) 'mezgraz       =',mezgraz
      write(iu06,fmtr) 'mezgrazf      =',mezgrazf
      write(iu06,fmtr) 'mizgraz       =',mizgraz
      write(iu06,fmtr) 'mizgrazf      =',mizgrazf
      write(iu06,fmtr) 'flagtsc       =',flagtsc
      write(iu06,fmtl) 'biocoldst     =',biocoldst
      write(iu06,fmtl) 'do_diag_pp    =',do_diag_pp
      write(iu06,fmtl) 'do_diag_chl   =',do_diag_chl
      write(iu06,fmtl) 'do_diag_secchi=',do_diag_secchi
      write(iu06,fmtl) 'write_light   =',write_light
      write(iu06,fmtl) 'netcdf_rivers =',netcdf_rivers
      write(iu06,fmtl) 'netcdf_bounds =',netcdf_bounds
      write(iu06,fmtl) 'netcdf_atm    =',netcdf_atm
      write(iu06,fmtl) 'use_nc_varname_nml_init      =',use_nc_varname_nml_init
      write(iu06,fmtl) 'use_nc_varname_nml_init_ben  =',use_nc_varname_nml_init_ben
      write(iu06,fmtl) 'use_nc_varname_nml_bounds    =',use_nc_varname_nml_bounds
      write(iu06,fmtl) 'use_nc_varname_nml_atmos_dep =',use_nc_varname_nml_atmos_dep
      write(iu06,fmtl) 'use_nc_varname_nml_river_infl=',use_nc_varname_nml_river_infl
      write(iu06,fmtl) 'use_nc_varname_nml_out      =',use_nc_varname_nml_out
      write(iu06,fmtl) 'use_nc_varname_nml_out_ben  =',use_nc_varname_nml_out_ben
      write(iu06,fmtl) 'tag_atmos_dep               =',tag_atmos_dep
      write(iu06,fmtl) 'tag_river_infl              =',tag_river_infl
      write(iu06,fmtl) 'read_netcdf_init_notag      =',read_netcdf_init_notag
      write(iu06,fmtl) 'read_netcdf_bounds_notag    =',read_netcdf_bounds_notag
      write(iu06,fmtl) 'read_binary_restart_notag   =',read_binary_restart_notag
      write(iu06,fmtl) 'read_binary_rwzk_notag      =',read_binary_rwzk_notag
      write(iu06,fmtl) 'write_binary_restart_notag  =',write_binary_restart_notag
    endif

  end subroutine ReadBioparam

!===============================================================================

  subroutine bioInd()

    use exits,          only : exitme

    implicit none

    integer(4)     :: c_diag

!    if (idx_oxy /= ntbio) then
!      call exitme(1, 'ERROR: Ergom could only run if idx_oxy == ntbio ... stop')
!    endif

    c_diag = 2
    
    ! Index for diagnostic vars, remember that light per definition is always 1
    if (do_diag_chl) then
      didx_chloro = c_diag
      c_diag = c_diag + 1
    endif
    if (do_diag_pp) then
      didx_pp = c_diag
      c_diag = c_diag + 1
    endif
    if (do_diag_secchi) then
       didx_secchi = c_diag
       c_diag = c_diag + 1
    endif
    
!   if (do_diag_chl)then
!     didx_chloro = 2
!     if(do_diag_pp)then
!       didx_pp   = 3
!     endif
!   else
!     if(do_diag_pp)then
!       didx_pp   = 2
!     endif  
!   endif   
  end subroutine

end module bioparam
