boronSysArag <- function( d11B, BCa, temperature, salinity, depth, whichOutput_input, whichKD_input, whichBT_input, whichKs_input, nMc_input, d11B_std_input, BCa_std_input, temperature_std_input, salinity_std_input, d11B_sysErr_input, BCa_sysErr_input, temperature_sysErr_input, salinity_sysErr_input) {
	
 
# boronSysArag calculates calcifying fluid carbonate ion concentration
# ([CO3]) and dissolved inorganic carbon (DIC) in umol/kg based on 
# aragonite d11B (boron isotope ratio, permille relative to NIST SRM 951),
# B/Ca (umol/mol), seawater temperature (°C), salinity and water depth.
#
#
# The basic function is:
# boronSysArag(d11B, BCa, temperature, salinity, depth)
# which returns [CO3] given the following inputs:
# d11B -> aragonite boron 11/10 isotope ratio relative to NIST SRM 951
# BCa -> aragonite B/Ca ratio in umol/mol
# temperature -> temperature (°C) of seawater
# salinity -> salinity of seawater
# depth -> water depth (m) (for pressure effects on pKB)
# * inputs may be scalars or vectors (i.e. single values or lists). Some
# inputs may be length n (where n>1) and others length 1, in which case it
# is assumed that inputs of length 1 are constants and their value will be
# applied to all n elements of the longest input.
#
#
# boronSysArag(d11B, BCa, temperature, salinity, depth...)
# estimates the non-systematic errors (1 standard deviation) of derived [CO3] from
# measurement precisions of d11B, B/Ca, temperature, and salinity
# AND, separately:
# estimates the systematic errors (1 standard deviation) of derived [CO3] from
# systematic sources of error including the boron concentration and
# isotopic composition of seawater, the 11B-10B fractionation factor, and
# the B partition coefficient between aragonite and seawater
#
#
# The full range of input options is:
# boronSysArag(d11B, BCa, temperature, salinity, depth, whichOutput, whichKD, whichBT, whichKs, iterations, d11B_std, BCa_std, temp_std, sal_std, d11B_accur, BCa_accur, temp_accur, sal_accur)
# where:
# whichOutput -> "CO3" (default) or "DIC" (both umol/kg)
# whichKD -> selects the KD formula to use: 
#       'H16' is equation 7 from Holcomb et al. (2016) 
#       'M17' (default) is McCulluch et al. (2017) refit
#       'newEq' is new (in prep) logarithmic equation fit to both Holcomb
#           et al. (2016) and Mavromatis et al. (2015) data
# which_BT -> selects which B/salinity relationship to use:
#       'U74' is Uppstrom (1974) where [B] = 0.12842*salinity (mg/kg), or 415.7*salinity/35 (umol/kg)
#       'L10' (default) is Lee et al. (2010) where [B] = 0.1336*salinity (mg/kg), or 432.5*salinity/35 (umol/kg)
# whichKs -> selects which K1 and K2 constants to use:
#       'L00' (default) is Lueker (2000)
#       'D87' is Mehrbach et al (1973) refit by Dickson and Millero (1987)
# iterations -> number of Monte Carlo iterations used for estimating uncertainties
# d11B_std -> precision of d11B measurements in permille
# BCa_std -> precision of B/Ca measurements in umol/mol
# temp_std -> temperature precision
# sal_std -> salinity precision
# d11B_accur -> accuracy of d11B measurements in permille
# BCa_accur -> accuracy of B/Ca measurements in umol/mol
# temp_accur -> temperature accuracy
# sal_accur -> salinity accuracy
#
#
# EXAMPLE uses:
#
# calculate [CO3] for the JCp-1 coral standard:
# JCp1 = boronSysArag(24.3, 459.6, 25, 34.54, 1)
#
# calculate [CO3] for the JCp-1 coral standard using different KD formula:
# JCp1_from_H16 = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3','H16')
# JCp1_from_M17 = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3','M17')
# JCp1_from_newEq = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3','newEq')
#
# calculate [CO3] for the JCp-1 coral standard using different B_total/salinity relationships:
# JCp1_with_L16_BT = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3','H16','L10')
# JCp1_with_U74_BT = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3','H16','U74')
#
# calculate [CO3] with systematic and non-systematic uncertainties for the JCp-1 coral standard:
# boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3','M17','L10','L00',1e3,0.17,24.3,0,0)
#
# calculate [DIC] with systematic and non-systematic uncertainties for the JCp-1 coral standard:
# boronSysArag(24.3, 459.6, 25, 34.54, 1, 'DIC','M17','L10','L00',1e3,0.17,24.3,0,0)
#
# effect of salinity with H16:
# sal_effect = boronSysArag(24.3, 459.6, 25, seq(30,36,1), 1, 'CO3','H16')
# plot(seq(30,36,1), sal_effect$CO3_Mean, type="l")
#
# effect of temperature with M17 and H16:
# temp_effect_M17 = boronSysArag(24.3, 459.6, seq(20,30,1), 35, 1, 'CO3','M17')
# temp_effect_H16 = boronSysArag(24.3, 459.6, seq(20,30,1), 35, 1, 'CO3','H16')
# plot(seq(20,30,1), temp_effect_M17$CO3_Mean,col="blue",type="l",ylim=c(1050,1200))
# lines(seq(20,30,1), temp_effect_H16$CO3_Mean,col="red")
#
#
# REFERENCES:
# Dickson AG, Riley JP (1979). The estimation of acid dissociation constants in seawater media from potentionmetric titrations with strong base. I. The ionic product of water - Kw, Marine Chemistry 7, 89-99.
# Dickson AG, Millero FJ (1987). A comparison of the equilibrium constants for the dissociation of carbonic acid in seawater media, Deep Sea Research Part A. Oceanographic Research Papers 34, 1733-1743.
# Dickson AG (1990). Thermodynamics of the dissociation of boric acid in synthetic seawater from 273.15 to 318.15 K, Deep-Sea Research 37, 755-766.
# Dickson AG (1990). Standard potential of the reaction: AgCl(s) + 12H2(g) = Ag(s) + HCl(aq) and the standard acidity constant of the ion HSO4- in synthetic sea water from 273.15 to 318.15 K, The Journal of Chemical Thermodynamics 22, 113-127.
# Foster GL, Pogge von Strandmann PAE, Rae JWB (2010). Boron and magnesium isotopic composition of seawater. Geochemistry Geophysics Geosystems 11, Q08015.
# Holcomb M, DeCarlo TM, Gaetani GA, McCulloch M (2016). Factors affecting B/Ca ratios in synthetic aragonite, Chemical Geology 437, 67-76.
# Klochko K, Kaufman AJ, Yao W, Byrne RH, Tossell JA (2006). Experimental measurement of boron isotope fractionation in seawater, Earth and Planetary Science Letters 248, 276-285.
# Lee K, Kim T-W, Byrne RH, Millero FJ, Feely RA, Liu Y-M (2010). The universal ratio of boron to chlorinity for the North Pacific and North Atlantic oceans, Geochemica et Cosmochimica Acta 74, 1801-1811.
# Lueker TJ, Dickson AG, Keeling CD (2000). Ocean pCO2 calculated from dissolved inorganic carbon, alkalinity, and equations for K1 and K2: validation based on laboratory measurements of CO2 in gas and seawater at equilibrium, Marine Chemistry 70, 105-119.
# Mavromatis V, Montouillout V, Noireaux J, Gaillardet J, Schott J (2015). Characterization of boron incorporation and speciation in calcite and aragonite from co-precipitation experiments under controlled pH, temperature and precipitation rate, Geochemica et Cosmochimica Acta 150, 299-313.
# McCulloch MT, D'Olivo JP, Falter J, Holcomb M, Trotter JA (2017). Coral calcification in a changing world and the interactive dynamics of pH and DIC upregulation, Nature Communications 8, 15686.
# Millero FJ (1979). The thermodynamics of the carbonate system in seawater, Geochemica et Cosmochimica Acta 43, 1651-1661.
# Morris AW, Riley JP (1966). The bromide/chlorinity and sulphate/chlorinity ratio in sea water, Deep Sea Research and Oceanographic Abstracts 13, 699-705.
# Riley JP (1965). The occurrence of anomalously high fluoride concentrations in the North Atlanic, Deep Sea Research and Oceanographic Abstracts 12, 219-220.
# Uppstrom LR (1974) The boron/chlorinity ratio of deep-sea water from the Pacific Ocean, Deep Sea Research and Oceanographic Abstracts
#
#
# Created 15 September, 2017 by Thomas M. DeCarlo
# University of Western Australia
# Code is provided "as-is". Author assumes not responsibility for its use.
# Contact by email at thomas.decarlo@uwa.edu.au with any questions
#
#
# End of Documentation, do not edit below this line
#
#====================================================================



varargin = nargs() 
 
# Defaults
d11B_std = 0 # d11B measurement precision (permille)
BCa_std = 0 # B/Ca measurement precision (umol/kg)
temperature_std = 0 # temperature precision (degree C)
salinity_std = 0 # salinity precision
whichKD = 2 # which KD to use (McCulloch et al. (2017) is default)
whichBT = 2 # which BT/salinity to use (Lee et al. (2010) is default)
whichKs = 1 # which K1 K2 to use (Lueker et al. (2000) is default)
nMc = 1 # how many Monte Carlo iterations to do
depth[which(depth==0)] = 0.1
whichOutput = 1

# set defaults for accuracy (i.e. systematic errors)
    d11B_sysErr = 0
    BCa_sysErr = 0
    temperature_sysErr = 0
    salinity_sysErr = 0

# check inputs
if (varargin>5) { # choice of output
	if (whichOutput_input=="CO3") { whichOutput = 1}
	if (whichOutput_input=="DIC") { whichOutput = 2}
}
if (varargin>6) { # choice of KD formula
	if (whichKD_input=="H16") { whichKD = 1}
	if (whichKD_input=="M17") { whichKD = 2}
	if (whichKD_input=="newEq") { whichKD = 3}
}
if (varargin>7) { # choice of BT/salinity relationship
	if (whichBT_input=="U74") { whichBT = 1}
	if (whichBT_input=="L10") { whichBT = 2}
}
if (varargin>8) { # choice of K1 K2 constants
	if (whichKs_input=="L00") { whichKs = 1}
	if (whichKs_input=="D87") { whichKs = 2}
}
if (varargin>9) { # Monte Carlo n input
    nMc = nMc_input
}
if (varargin>10) { # d11B error input
    d11B_std = d11B_std_input
}
if (varargin>11) { # B/Ca error input
    BCa_std = BCa_std_input
}
if (varargin>12) { # temperature error input
    temperature_std = temperature_std_input
}
if (varargin>13) { # salinity error input
    salinity_std = salinity_std_input
}
if (varargin>14) { # d11B error input
    d11B_sysErr = d11B_sysErr_input
}
if (varargin>15) { # B/Ca error input
    BCa_sysErr = BCa_sysErr_input
}
if (varargin>16) { # temperature error input
    temperature_sysErr = temperature_sysErr_input
}
if (varargin>17) { # salinity error input
    salinity_sysErr = salinity_sysErr_input
}
    
    # other systematic errors:
    d11B_sw_sysErr = 0.04 # Foster et al (2010)
    B_sw_sysErr = 0.0005 # Lee et al (2010) (to be multiplied by the coefficient below)
    alpha_11B_sysErr = 0.0006 # Klochko et al (2006)
    lnKB_err = 0.0042 # Dickson (1990)
    
    if (whichKD == 1) {
    	# below is repeating the "polyval" MATLAB function to get errors from regression parameters
    	p = c(0.09474, -0.0000119)
		x = seq(100,3000,1)
		nc = length(p)
		R=matrix(,nrow=2,ncol=2)
		R[1,] = c(-8.392276399642210,  -0.005714060701009)*1000
		R[2,] = c(0,  -0.002519823467068)*1000
		df = 37
		normr = 0.046882947784606
		V = matrix(1,nrow=length(x),ncol=2)
		V[,1] = x*V[,1]
		E = V %*% solve(R)
		e = sqrt(1+rowSums(E*E,2))
		delta = normr/sqrt(df)*e
		s = normr/sqrt(df)
		delta = delta/s
		predvar = s*sqrt(delta^2-1)
		crit = qt(1-0.05/2,df)
		delta = crit * predvar
		KD_sysErr = data.frame(co3 = x, KD_err = delta)

    } else if (whichKD == 2) {
        KD_sysErr = 0.17e-3; # McCulloch et al (2017)
    } else if (whichKD == 3) {
    	# below is repeating the "polyval" MATLAB function to get errors from regression parameters
    	p = c(-0.002843, 0.00076771)
		x = seq(100,3000,1)
		nc = length(p)
		R=matrix(,nrow=2,ncol=2)
		R[1,] = c(-45.711596536864732,  -6.860104655396300)
		R[2,] = c(0,  -1.392466917743484)
		df = 47
		normr = 0.002109243097308
		V = matrix(1,nrow=length(x),ncol=2)
		V[,1] = log(x)*V[,1]
		E = V %*% solve(R)
		e = sqrt(1+rowSums(E*E,2))
		delta = normr/sqrt(df)*e
		s = normr/sqrt(df)
		delta = delta/s
		predvar = s*sqrt(delta^2-1)
		crit = qt(1-0.05/2,df)
		delta = crit * predvar
		KD_sysErr = data.frame(co3 = x, KD_err = delta)
    }

n1 = length(d11B)
n2 = length(BCa)
n3 = length(temperature)
n4 = length(salinity)
n5 = length(d11B_std)
n6 = length(BCa_std)
n7 = length(salinity)
n8 = length(d11B_std)
n9 = length(BCa_std)
n = max(c(n1,n2,n3,n4,n5,n6,n7,n8,n9))
	
	if (length(d11B)==1) {
		d11B_in = matrix(d11B,n,nMc)
	} else {
		d11B_in = matrix(d11B,n,nMc)
	}
	if (length(BCa)==1) {
		BCa_in = matrix(BCa,n,nMc)
	} else {
		BCa_in = matrix(BCa,n,nMc)
	}
	if (length(temperature)==1) {
		temperature_in = matrix(temperature,n,nMc)
	} else {
		temperature_in = matrix(temperature,n,nMc)
	}
	if (length(salinity)==1) {
		salinity_in = matrix(salinity,n,nMc)
	} else {
		salinity_in = matrix(salinity,n,nMc)
	}
	if (length(d11B_std)==1) {
		d11B_std_in = matrix(d11B_std,n,nMc)
	} else {
		d11B_std_in = matrix(d11B_std,n,nMc)
	}
	if (length(BCa_std)==1) {
		BCa_std_in = matrix(BCa_std,n,nMc)
	} else {
		BCa_std_in = matrix(BCa_std,n,nMc)
	}
	if (length(temperature_std)==1) {
		temperature_std_in = matrix(temperature_std,n,nMc)
	} else {
		temperature_std_in = matrix(temperature_std,n,nMc)
	}
	if (length(salinity_std)==1) {
		salinity_std_in = matrix(salinity_std,n,nMc)
	} else {
		salinity_std_in = matrix(salinity_std,n,nMc)
	}
	if (length(depth)==1) {
		depth_in = matrix(depth,n,nMc)
	} else {
		depth_in = matrix(depth,n,nMc)
	}
	
	# initialize output storage
outList = matrix(1000,n,nMc)
	
# add random errors for Monte Carlo
d11B_in = d11B_in+matrix(rnorm(n*nMc),n)*d11B_std_in
BCa_in = BCa_in+matrix(rnorm(n*nMc),n)*BCa_std_in
temperature_in = temperature_in+matrix(rnorm(n*nMc),n)*temperature_std_in
salinity_in = salinity_in+matrix(rnorm(n*nMc),n)*salinity_std_in

# carbonate and boron system calculations:
if (whichBT==2) {
    B = (0.1336/10.811*1000)*salinity_in # total [B] (umol/kg) from Lee et al (2010)
} else if (whichBT==1) {
    B = (0.12842/10.811*1000)*salinity_in # total [B] (umol/kg) from Uppstrom (1974)
}

pKB = -log10(exp((-8966.9-2890.53*sqrt(salinity_in)-77.942* salinity_in +1.728*sqrt(salinity_in)* salinity_in-0.0996* salinity_in ^2)/(temperature_in +273.15) + 148.0248 + 137.1942*sqrt(salinity_in) + 1.62142* salinity_in + (-24.4344 - 25.085*sqrt(salinity_in) - 0.2474* salinity_in)*log((temperature_in +273.15),exp(1)) + 0.053105*sqrt(salinity_in)*(temperature_in +273.15))) # Dickson 1990
P_factor = -(-29.48+0.1622*temperature_in+(-2.608*10^-3)*temperature_in^2)/(83.14472*(temperature_in+273.15))*(depth_in/10)+0.5*((-2.84*10^-3)+(0*10^-3)*temperature_in+0*temperature_in^2)/(83.14472*(depth_in/10))*(depth_in/10)^2 # Millero 1979
pKB_press = -log10(exp(P_factor+log(10^(-pKB))))
pH = pKB_press-log10(-(39.61-d11B_in)/(39.61-1.0272* d11B_in-1000*(1.0272-1))) # using Klochko et al 2006 alpha of 1.0272
H = 10^(-pH) # (mol/kg)
BOH4 = (B*10^(-pKB))/(10^(-pKB)+H) # [borate] (umol/kg)
if (whichKs == 1) { # Lueker (2010) K1 and K2 (given in total pH scale):
    K1 = 10^(-((3633.86)/(temperature_in+273.15)-(61.2172)+9.6777*(log(temperature_in+273.15))-0.011555*salinity_in+0.0001152*salinity_in^2))
    K2 = 10^(-((471.78)/(temperature_in+273.15)+(25.929)-3.16967*(log(temperature_in+273.15))-0.01781*salinity_in+0.0001122*salinity_in^2))
} else if (whichKs == 2) { # Mehrbach refit by Dickson and Millero (given in seawater pH scale,
    # converted to total scale):
    TF = (0.000067/18.998)*(salinity_in/1.80655) # in mol/kg-SW Riley, J. P., Deep-Sea Research 12:219-220, 1965
    TS = (0.14/96.062)*(salinity_in/1.80655) # in mol/kg-SW Morris, A. W., and Riley, J. P., Deep-Sea Research 13:699-705, 1966
    IonS = 19.924 * salinity_in / (1000 - 1.005* salinity_in) # This is from the DOE handbook, Chapter 5, p. 13/22, eq. 7.2.4:
    # Dickson, A. G., J. Chemical Thermodynamics, 22:113-127, 1990:
    KS = exp(-4276.1/(temperature_in+273.15) + 141.328 - 23.093*(log(temperature_in+273.15)) + (-13856/(temperature_in+273.15) + 324.57 - 47.986*(log(temperature_in+273.15)))*sqrt(IonS) + (35474/(temperature_in+273.15) - 771.54 + 114.723*(log(temperature_in+273.15)))*IonS + (-2698/(temperature_in+273.15))*sqrt(IonS)*IonS + (1776/(temperature_in+273.15))*IonS^2) * (1 - 0.001005 * salinity_in)
    # Dickson, A. G. and Riley, J. P., Marine Chemistry 7:89-99, 1979:
    KF = exp(1590.2/(temperature_in+273.15) - 12.641 + 1.525*sqrt(IonS)) *(1 - 0.001005*salinity_in)
    swsTOtot = (1 + TS/KS)/(1 + TS/KS + TF/KF)
    K1 = 10^(-(3670.7/(temperature_in+273.15) - 62.008 + 9.7944*(log(temperature_in+273.15)) - 0.0118*salinity_in + 0.000116*salinity_in)) * swsTOtot
    K2 = 10^(-(1394.7/(temperature_in+273.15) + 4.777 - 0.0184*salinity_in + 0.000118*salinity_in^2)) * swsTOtot
}
# K1 and K2 pressure effects:
dV1 = -25.5 + 0.1271*temperature_in
dK1 = (-3.08 + 0.0877*temperature_in)/1000
K1_press = exp((-dV1 + 0.5*dK1*(depth_in/10))*(depth_in/10)/(83.14472*temperature_in)) * K1
dV2 = -15.82 - 0.0219*temperature_in
dK2 = (1.13 - 0.1475*temperature_in)/1000
K2_press = exp((-dV2 + 0.5*dK2*(depth_in/10))*(depth_in/10)/(83.14472*temperature_in)) * K2
# FCO2:
fCO2 =1/(1+H/K2_press+H*H/K1_press*K2_press)


resFunc_H16  = function(co3guess,B, BOH4, BCa) {
        
        KdBCa = -0.01215*(B/1000) -0.0000119*(co3guess) + 0.09474 # Holcomb et al 2016
        CO3_out = (BOH4/((BCa/1000)/KdBCa))^2 # [CO3] (umol/kg)
        
        resid = max(max(sqrt((co3guess-CO3_out)^2))) # minimize this (i.e. [CO3] used for KD is same as CO3 solved from KD)
        return(resid)
        
    }
    
    resFunc_D18  = function(co3guess, BOH4, BCa) {
        
       	KdBCa = log(co3guess)*0.00076771 - 0.002843 # new equation
        CO3_out = KdBCa/BCa*BOH4*1e6 # [CO3] (umol/kg)

        resid = max(max(sqrt((co3guess-CO3_out)^2))) # minimize this (i.e. [CO3] used for KD is same as CO3 solved from KD)
        return(resid)
        
    }

if (whichKD == 2) {
KdBCa = 2.97*exp(-0.02*(H*10^9))/1000; # McCulloch et al. (2017) refit
outList = KdBCa/BCa_in*BOH4*1e6 # [CO3] (umol/kg)
} else{
for (i in 1:1:nMc) {
    for (j in 1:1:n) {
        # perform minimization such that the [CO3] used to predict the KD
        # matches the [CO3] predicted by the KD
        if (whichKD == 1) {
        	thisOut = optimize(resFunc_H16,c(100,10000),B=B[j,i], BOH4= BOH4[j,i], BCa = BCa_in[j,i])
        	outList[j,i] = thisOut$minimum
            } else if (whichKD == 3) {
            thisOut = optimize(resFunc_D18,c(100,10000), BOH4= BOH4[j,i], BCa = BCa_in[j,i])
        	outList[j,i] = thisOut$minimum
        }
    }
}
}

co3_means = rowMeans(outList)

# reset without precision uncertainties
	d11B_in = matrix(d11B,n,nMc)
	BCa_in = matrix(BCa,n,nMc)
	temperature_in = matrix(temperature,n,nMc)
	salinity_in = matrix(salinity,n,nMc)
	depth_in = matrix(depth,n,nMc)

# add systematic errors for Monte Carlo
    d11B_in = d11B_in+matrix(rnorm(n*nMc),n)*d11B_sysErr
    BCa_in = BCa_in+matrix(rnorm(n*nMc),n)*BCa_sysErr
    temperature_in = temperature_in+matrix(rnorm(n*nMc),n)*temperature_sysErr
    salinity_in = salinity_in+matrix(rnorm(n*nMc),n)*salinity_sysErr
    
    # carbonate and boron system calculations:
if (whichBT==2) {	
    B = ( (0.1336+matrix(rnorm(n*nMc),n)*B_sw_sysErr) / 10.811 *1000)*salinity_in # total [B] (umol/kg) from Lee et al (2010)
    } else if (whichBT==1) {
    B = ( (0.12842+matrix(rnorm(n*nMc),n)*B_sw_sysErr) / 10.811 *1000)*salinity_in # total [B] (umol/kg) from Uppstrom (1974)
    }
pKB = -log10(exp( matrix(rnorm(n*nMc),n)*lnKB_err + (-8966.9-2890.53*sqrt(salinity_in)-77.942* salinity_in +1.728*sqrt(salinity_in)* salinity_in-0.0996* salinity_in ^2)/(temperature_in +273.15) + 148.0248 + 137.1942*sqrt(salinity_in) + 1.62142* salinity_in + (-24.4344 - 25.085*sqrt(salinity_in) - 0.2474* salinity_in)*log((temperature_in +273.15),exp(1)) + 0.053105*sqrt(salinity_in)*(temperature_in +273.15))) # Dickson 1990
P_factor = -(-29.48+0.1622*temperature_in+(-2.608*10^-3)*temperature_in^2)/(83.14472*(temperature_in+273.15))*(depth_in/10)+0.5*((-2.84*10^-3)+(0*10^-3)*temperature_in+0*temperature_in^2)/(83.14472*(depth_in/10))*(depth_in/10)^2
pKB_press = -log10(exp(P_factor+log(10^(-pKB))))
pH = pKB_press-log10(-( (39.61+matrix(rnorm(n*nMc),n)*d11B_sw_sysErr) -d11B_in)/( (39.61+matrix(rnorm(n*nMc),n)*d11B_sw_sysErr) - (1.0272+matrix(rnorm(n*nMc),n)*alpha_11B_sysErr) *d11B_in-1000*( (1.0272+matrix(rnorm(n*nMc),n)*alpha_11B_sysErr) -1))) # total scale
H = 10^(-pH) # proton concentration (mol/kg)
BOH4 = (B*10^(-pKB))/(10^(-pKB)+H) # [borate] (umol/kg)
if (whichKs == 1) { # Lueker (2010) K1 and K2 (given in total pH scale):
    K1_sysErr = 10^(-((3633.86)/(temperature_in+273.15)-(61.2172)+9.6777*(log(temperature_in+273.15))-0.011555*salinity_in+0.0001152*salinity_in^2))
    K2_sysErr = 10^(-((471.78)/(temperature_in+273.15)+(25.929)-3.16967*(log(temperature_in+273.15))-0.01781*salinity_in+0.0001122*salinity_in^2))
} else if (whichKs == 2) { # Mehrbach refit by Dickson and Millero (given in seawater pH scale,
    # converted to total scale):
    TF_sysErr = (0.000067/18.998)*(salinity_in/1.80655) # in mol/kg-SW Riley, J. P., Deep-Sea Research 12:219-220, 1965
    TS_sysErr = (0.14/96.062)*(salinity_in/1.80655) # in mol/kg-SW Morris, A. W., and Riley, J. P., Deep-Sea Research 13:699-705, 1966
    IonS_sysErr = 19.924 * salinity_in / (1000 - 1.005* salinity_in) # This is from the DOE handbook, Chapter 5, p. 13/22, eq. 7.2.4:
    # Dickson, A. G., J. Chemical Thermodynamics, 22:113-127, 1990:
    KS_sysErr = exp(-4276.1/(temperature_in+273.15) + 141.328 - 23.093*(log(temperature_in+273.15)) + (-13856/(temperature_in+273.15) + 324.57 - 47.986*(log(temperature_in+273.15)))*sqrt(IonS_sysErr) + (35474/(temperature_in+273.15) - 771.54 + 114.723*(log(temperature_in+273.15)))*IonS_sysErr + (-2698/(temperature_in+273.15))*sqrt(IonS_sysErr)*IonS_sysErr + (1776/(temperature_in+273.15))*IonS_sysErr^2) * (1 - 0.001005 * salinity_in)
    # Dickson, A. G. and Riley, J. P., Marine Chemistry 7:89-99, 1979:
    KF_sysErr = exp(1590.2/(temperature_in+273.15) - 12.641 + 1.525*sqrt(IonS_sysErr)) *(1 - 0.001005*salinity_in)
    swsTOtot_sysErr = (1 + TS_sysErr/KS_sysErr)/(1 + TS_sysErr/KS_sysErr + TF_sysErr/KF_sysErr)
    K1_sysErr = 10^(-(3670.7/(temperature_in+273.15) - 62.008 + 9.7944*(log(temperature_in+273.15)) - 0.0118*salinity_in + 0.000116*salinity_in)) * swsTOtot_sysErr
    K2_sysErr = 10^(-(1394.7/(temperature_in+273.15) + 4.777 - 0.0184*salinity_in + 0.000118*salinity_in^2)) * swsTOtot_sysErr
}
# K1 and K2 pressure effects:
dV1_sysErr = -25.5 + 0.1271*temperature_in
dK1_sysErr = (-3.08 + 0.0877*temperature_in)/1000
K1_press_sysErr = exp((-dV1 + 0.5*dK1_sysErr*(depth_in/10))*(depth_in/10)/(83.14472*temperature_in)) * K1_sysErr
dV2_sysErr = -15.82 - 0.0219*temperature_in
dK2_sysErr = (1.13 - 0.1475*temperature_in)/1000
K2_press_sysErr = exp((-dV2_sysErr + 0.5*dK2_sysErr*(depth_in/10))*(depth_in/10)/(83.14472*temperature_in)) * K2_sysErr
# FCO2:
fCO2_sysErr =1/(1+H/K2_press_sysErr+H*H/K1_press_sysErr*K2_press_sysErr)
    
    if (whichKD == 1) {
    	KdBCa_e = vector(,length(co3_means))
    	for (j in 1:1:length(co3_means)) {
    	KdBCa_e[j] = KD_sysErr$KD_err[which(KD_sysErr$co3==round(co3_means[j],0))]
    	}
        KdBCa = -0.01215*(B/1000) -0.0000119*(co3_means) + 0.09474 # Holcomb et al 2016
        CO3_sysErr_all = (BOH4/((BCa_in/1000)/ (KdBCa+matrix(rnorm(n*nMc),n)*(KdBCa_e/1.96)) ))^2 # [CO3] (umol/kg)
    } else if (whichKD == 2) {
        KdBCa = 2.97*exp(-0.02*(H*10^9))/1000 + matrix(rnorm(n*nMc),n)*KD_sysErr/1.96 # McCulloch et al. (2017) refit
        CO3_sysErr_all = KdBCa/BCa*BOH4*1e6 # [CO3] (umol/kg)
    } else if (whichKD == 3) {
        KdBCa_e = vector(,length(co3_means))
    	for (j in 1:1:length(co3_means)) {
    	KdBCa_e[j] = KD_sysErr$KD_err[which(KD_sysErr$co3==round(co3_means[j],0))]
    	}
        KdBCa = log(co3_means)*0.00076771 - 0.002843 # new equation
        CO3_sysErr_all = (KdBCa+matrix(rnorm(n*nMc),n)*(KdBCa_e/1.96)) /BCa_in*BOH4*1e6 # [CO3] (umol/kg)
    }
    

DIC_list = outList/fCO2
DIC_list_sysErr = CO3_sysErr_all/fCO2_sysErr
DIC_means = rowMeans(DIC_list)
DIC_stds = vector(,length(DIC_means))
DIC_sys_err = vector(,length(DIC_means))
co3_means = rowMeans(outList)
co3_stds = vector(,length(co3_means))
co3_sys_err = vector(,length(co3_means))
for (j in 1:1:length(co3_stds)) {
	co3_stds[j] = sd(outList[j,])
	co3_sys_err[j] = sd(CO3_sysErr_all[j,])
	DIC_stds[j] = sd(DIC_list[j,])
	DIC_sys_err[j] = sd(DIC_list_sysErr[j,])
}
co3 = data.frame(CO3_Mean=co3_means, CO3_SD = co3_stds, CO3_SysErr = co3_sys_err)
dic = data.frame(DIC_Mean=DIC_means, DIC_SD = DIC_stds, DIC_SysErr = DIC_sys_err)

if (whichOutput==1) {out = co3}
if (whichOutput==2) {out = dic}

return(out)

}


