function [varargout] = boronSysArag( d11B, BCa, temperature, salinity, depth, varargin )

% boronSysArag calculates calcifying fluid carbonate ion concentration
% ([CO3]) and dissolved inorganic carbon (DIC) in umol/kg based on 
% aragonite d11B (boron isotope ratio, permille relative to NIST SRM 951),
% B/Ca (umol/mol), seawater temperature (C), salinity and water depth.
%
% The basic function is:
% boronSysArag(d11B, BCa, temperature, salinity, depth)
% which returns [CO3] given the following inputs:
% d11B -> aragonite boron 11/10 isotope ratio relative to NIST SRM 951
% BCa -> aragonite B/Ca ratio in umol/mol
% temperature -> temperature (C) of seawater
% salinity -> salinity of seawater
% depth -> water depth (m) (for pressure effects on pKB)
% * inputs may be scalars or vectors (i.e. single values or lists). Some
% inputs may be length n (where n>1) and others length 1, in which case it
% is assumed that inputs of length 1 are constants and their value will be
% applied to all n elements of the longest input.
%
%
% [CO3_mean, CO3_std] = boronSysArag(d11B, BCa, temperature, salinity, depth,...)
% estimates the non-systematic errors (1 standard deviation) of derived [CO3] from
% measurement precisions of d11B, B/Ca, temperature, and salinity
%
%
% [CO3_mean, CO3_std, CO3_sysErr] = boronSysArag(d11B, BCa, temperature, salinity, depth,...)
% estimates the systematic errors (1 standard deviation) of derived [CO3] from
% systematic sources of error including the boron concentration and
% isotopic composition of seawater, the 11B-10B fractionation factor, and 
% the B partition coefficient between aragonite and seawater
%
%
% The full range of input options is:
% [means, std, sysErrs] = boronSysArag(d11B, BCa, temperature, salinity, depth, whichOutput, whichKD, whichBT, whichKs, iterations, d11B_std, BCa_std, temp_std, sal_std, d11B_accur, BCa_accur, temp_accur, sal_accur)
% where:
% whichOutput -> "CO3" (default) or "DIC" (both umol/kg)
% whichKD -> selects the KD formula to use: 
%       'H16' is equation 7 from Holcomb et al. (2016) 
%       'M17' (default) is McCulluch et al. (2017) refit
%       'newEq' is new (in prep) logarithmic equation fit to both Holcomb
%           et al. (2016) and Mavromatis et al. (2015) data
% whichBT -> selects which B/salinity relationship to use:
%       'U74' is Uppstrom (1974) where [B] = 0.12842*salinity (mg/kg), or 415.7*salinity/35 (umol/kg)
%       'L10' (default) is Lee et al. (2010) where [B] = 0.1336*salinity (mg/kg), or 432.5*salinity/35 (umol/kg)
% whichKs -> selects which K1 and K2 constants to use:
%       'L00' (default) is Lueker (2000)
%       'D87' is Mehrbach et al (1973) refit by Dickson and Millero (1987)
% iterations -> number of Monte Carlo iterations used for estimating uncertainties
% d11B_std -> precision of d11B measurements in permille
% BCa_std -> precision of B/Ca measurements in umol/mol
% temp_std -> temperature precision
% sal_std -> salinity precision
% d11B_accur -> accuracy of d11B measurements in permille
% BCa_accur -> accuracy of B/Ca measurements in umol/mol
% temp_accur -> temperature accuracy
% sal_accur -> salinity accuracy
%
%
% EXAMPLE uses:
%
% calculate [CO3] for the JCp-1 coral standard:
% JCp1 = boronSysArag(24.3, 459.6, 25, 34.54, 1)
%
% calculate [CO3] for the JCp-1 coral standard using different KD formula:
% JCp1_from_H16 = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3', 'H16')
% JCp1_from_M17 = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3', 'M17')
% JCp1_from_newEq = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3', 'newEq')
%
% calculate [CO3] for the JCp-1 coral standard using different B_total/salinity relationships:
% JCp1_with_L16_BT = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3', 'H16','L10')
% JCp1_with_U74_BT = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3', 'H16','U74')
%
% calculate [CO3] with non-systematic uncertainties for the JCp-1 coral standard:
% [JCp1_mean, JCp1_std] = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3', 'M17','L10','L00',1e3,0.17,24.3,0,0)
%
% calculate [CO3] with systematic uncertainties for the JCp-1 coral standard:
% [JCp1_mean, JCp1_std, JCp1_sysErr] = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'CO3', 'M17','L10','L00',1e3,0.17,24.3)
%
% calculate DIC with systematic uncertainties for the JCp-1 coral standard:
% [JCp1_DIC_mean, JCp1_DIC_std, JCp1_DIC_sysErr] = boronSysArag(24.3, 459.6, 25, 34.54, 1, 'DIC', 'M17','L10','L00',1e3,0.17,24.3)
%
% effect of salinity with H16:
% plot([30:36],boronSysArag(24.3, 459.6, 25, [30:36], 1, 'CO3', 'H16'))
%
% effect of temperature with M17 and H16:
% plot([20:30],boronSysArag(24.3, 459.6, [20:30], 35, 1, 'CO3', 'M17'),'b')
% hold on
% plot([20:30],boronSysArag(24.3, 459.6, [20:30], 35, 1, 'CO3', 'H16'),'r')
%
% Load some published datasets:
% load('boron_datasets.mat')
% whichStudy = 7; % this is McCulloch et al 2017 (see boron.study.name for all others)
% example = boronSysArag(boron.study(whichStudy).d11B, boron.study(whichStudy).BCa, boron.study(whichStudy).temp, boron.study(whichStudy).salt, boron.study(whichStudy).depth)
%
% Visualize parameter space and KD differences:
% ----------------------------------------------------------------------->
% d11B_vec = 21:0.1:26; % vector of d11B
% BCa_vec = 250:10:750; % vector of B/Ca
% [d11B_mat,BCa_mat] = meshgrid(d11B_vec,BCa_vec); % make grid
% CO3 = NaN(length(BCa_vec),length(d11B_vec),3); % initialize
% for i = 1:length(d11B_mat(:,1)) % calculate across the grid
%     CO3(i,:,1) = boronSysArag( d11B_mat(i,:), BCa_mat(i,:), 25, 35, 1, 'CO3', 'H16'); 
%     CO3(i,:,2) = boronSysArag( d11B_mat(i,:), BCa_mat(i,:), 25, 35, 1, 'CO3', 'M17');
%     CO3(i,:,3) = boronSysArag( d11B_mat(i,:), BCa_mat(i,:), 25, 35, 1, 'CO3', 'newEq');
% end
% figure
% subplot(221)
% pcolor(d11B_mat,BCa_mat,CO3(:,:,1))
% shading interp
% xlabel('\delta^{11}B')
% ylabel('B/Ca (\mumol/mol)')
% colorbar
% title('[CO_3^{2-}] (\mumol/kg)  ^{}(H16 K_D)')
% for i=1:3
% subplot(2,2,i+1)
% if i == 1  pcolor(d11B_mat,BCa_mat,CO3(:,:,2)-CO3(:,:,1)); title('\Delta[CO_3^{2-}]  ^{}M17 - H16');  end 
% if i == 2  pcolor(d11B_mat,BCa_mat,CO3(:,:,2)-CO3(:,:,3)); title('\Delta[CO_3^{2-}]  ^{}M17 - new Eq.');  end 
% if i == 3  pcolor(d11B_mat,BCa_mat,CO3(:,:,3)-CO3(:,:,1)); title('\Delta[CO_3^{2-}]  ^{}new Eq. - H16');  end 
% shading interp
% xlabel('\delta^{11}B')
% ylabel('B/Ca (\mumol/mol)')
% set(gca,'CLim',[-250,250])
% colorbar
% end
% <-----------------------------------------------------------------------
%
%
% % REFERENCES:
% Dickson AG, Riley JP (1979). The estimation of acid dissociation constants in seawater media from potentionmetric titrations with strong base. I. The ionic product of water - Kw, Marine Chemistry 7, 89-99.
% Dickson AG, Millero FJ (1987). A comparison of the equilibrium constants for the dissociation of carbonic acid in seawater media, Deep Sea Research Part A. Oceanographic Research Papers 34, 1733-1743.
% Dickson AG (1990). Thermodynamics of the dissociation of boric acid in synthetic seawater from 273.15 to 318.15 K, Deep-Sea Research 37, 755-766.
% Dickson AG (1990). Standard potential of the reaction: AgCl(s) + 12H2(g) = Ag(s) + HCl(aq) and the standard acidity constant of the ion HSO4- in synthetic sea water from 273.15 to 318.15 K, The Journal of Chemical Thermodynamics 22, 113-127.
% Foster GL, Pogge von Strandmann PAE, Rae JWB (2010). Boron and magnesium isotopic composition of seawater. Geochemistry Geophysics Geosystems 11, Q08015.
% Holcomb M, DeCarlo TM, Gaetani GA, McCulloch M (2016). Factors affecting B/Ca ratios in synthetic aragonite, Chemical Geology 437, 67-76.
% Klochko K, Kaufman AJ, Yao W, Byrne RH, Tossell JA (2006). Experimental measurement of boron isotope fractionation in seawater, Earth and Planetary Science Letters 248, 276-285.
% Lee K, Kim T-W, Byrne RH, Millero FJ, Feely RA, Liu Y-M (2010). The universal ratio of boron to chlorinity for the North Pacific and North Atlantic oceans, Geochemica et Cosmochimica Acta 74, 1801-1811.
% Lueker TJ, Dickson AG, Keeling CD (2000). OceanpCO2calculated from dissolved inorganic carbon, alkalinity, and equations forK1andK2: validation based on laboratory measurements of CO2in gas and seawater at equilibrium, Marine Chemistry 70, 105-119.
% Mavromatis V, Montouillout V, Noireaux J, Gaillardet J, Schott J (2015). Characterization of boron incorporation and speciation in calcite and aragonite from co-precipitation experiments under controlled pH, temperature and precipitation rate, Geochemica et Cosmochimica Acta 150, 299-313.
% McCulloch MT, D'Olivo JP, Falter J, Holcomb M, Trotter JA (2017). Coral calcification in a changing world and the interactive dynamics of pH and DIC upregulation, Nature Communications 8, 15686.
% Millero FJ (1979). The thermodynamics of the carbonate system in seawater, Geochemica et Cosmochimica Acta 43, 1651-1661.
% Morris AW, Riley JP (1966). The bromide/chlorinity and sulphate/chlorinity ratio in sea water, Deep Sea Research and Oceanographic Abstracts 13, 699-705.
% Riley JP (1965). The occurrence of anomalously high fluoride concentrations in the North Atlanic, Deep Sea Research and Oceanographic Abstracts 12, 219-220.
% Uppstrom LR (1974) The boron/chlorinity ratio of deep-sea water from the Pacific Ocean, Deep Sea Research and Oceanographic Abstracts
%
%
% Created 15 September, 2017 by Thomas M. DeCarlo
% University of Western Australia
% Code is provided "as-is". Author assumes not responsibility for its use.
% Contact by email at thomas.decarlo@uwa.edu.au with any questions
%
%
% End of Documentation, do not edit below this line
%
%=========================================================================


% Defaults
d11B_std = 0; % d11B measurement precision (permille)
BCa_std = 0; % B/Ca measurement precision (umol/kg)
temperature_std = 0; % temperature precision (degree C)
salinity_std = 0; % salinity precision
whichKD = 2; % which KD to use (McCulloch et al. (2017) is default)
whichBT = 2; % which BT/salinity to use (Lee et al. (2010) is default)
whichKs = 1; % which K1 K2 to use (Lueker et al (2000) is default)
nMc = 1; % how many Monte Carlo iterations to do
depth(depth==0) = 0.1;
whichOutput = 1; % return CO3 by default

% check inputs
if (~isempty(varargin)) % choice of output data
    switch varargin{1}
        case {'CO3'}
            whichOutput = 1;
        case {'DIC'}
            whichOutput = 2;    
        otherwise
            error(['Invalid output choice, ', ...
                varargin{1}]);
    end
end
if (length(varargin)>1) % choice of KD formula
    switch varargin{2}
        case {'H16'}
            whichKD = 1;
        case {'M17'}
            whichKD = 2;    
        case {'newEq'}
            whichKD = 3;
        otherwise
            error(['Invalid KD choice, ', ...
                varargin{2}]);
    end
end
if (length(varargin)>2) % which BT
    switch varargin{3}
        case {'U74'}
            whichBT = 1;
        case {'L10'}
            whichBT = 2;
        otherwise
            error(['Invalid BT/salinity choice, ', ...
                varargin{3}]);
    end
end
if (length(varargin)>3) % choice of K1 K2 constants
    switch varargin{4}
        case {'L00'}
            whichKs = 1;
        case {'D87'}
            whichKs = 2;    
        otherwise
            error(['Invalid K1 K2 choice, ', ...
                varargin{4}]);
    end
end
if (length(varargin)>4) % Monte Carlo n input
    nMc = varargin{5};
end
if (length(varargin)>5) % d11B error input
    d11B_std = varargin{6};
end
if (length(varargin)>6) % B/Ca error input
    BCa_std = varargin{7};
end
if (length(varargin)>7) % temperature error input
    temperature_std = varargin{8};
end
if (length(varargin)>8) % salinity error input
    salinity_std = varargin{9};
end

if nargout == 3 % do systematic error estimates
    
    % set defaults for accuracy (i.e. systematic errors)
    d11B_sysErr = 0;
    BCa_sysErr = 0;
    temperature_sysErr = 0;
    salinity_sysErr = 0;
    
    if (length(varargin)>9) % d11B error input
        d11B_sysErr = varargin{10};
    end
    if (length(varargin)>10) % B/Ca error input
        BCa_sysErr = varargin{11};
    end
    if (length(varargin)>11) % temperature error input
        temperature_sysErr = varargin{12};
    end
    if (length(varargin)>12) % salinity error input
        salinity_sysErr = varargin{13};
    end
    
    % other systematic errors:
    d11B_sw_sysErr = 0.04; % Foster et al (2010)
    B_sw_sysErr = 0.0005; % Lee et al (2010) (to be multiplied by the coefficient below)
    alpha_11B_sysErr = 0.0006; % Klochko et al (2006)
    lnKB_err = 0.0042; % Dickson (1990)
    
    % KD systematic errors:
    if whichKD == 1
        KD_sysErr = struct('R',1.0e+03*[-8.392276399642210,  -0.005714060701009;0,  -0.002519823467068]...
            ,'df',37,'normr',0.046882947784606); % from Holcomb et al. (2016)
    elseif whichKD == 2
        KD_sysErr = 0.17e-3; % McCulloch et al (2017)
    elseif whichKD == 3
        KD_sysErr = struct('R',[-45.711596536864732,  -6.860104655396300;0,  -1.392466917743484]...
            ,'df',47,'normr',0.002109243097308); % new equation with Holcomb et al. (2016) and Mavromatis et al. (2015)
    end
    
end

% make sure everything is in column vectors
d11B = d11B(:);
BCa = BCa(:);
temperature = temperature(:);
salinity = salinity(:);
d11B_std = d11B_std(:);
BCa_std = BCa_std(:);
temperature_std = temperature_std(:);
salinity_std = salinity_std(:);
depth = depth(:);

% get length of longest vector
n1 = length(d11B);
n2 = length(BCa);
n3 = length(temperature);
n4 = length(salinity);
n5 = length(d11B_std);
n6 = length(BCa_std);
n7 = length(temperature_std);
n8 = length(salinity_std);
n9 = length(depth);
n = max([n1,n2,n3,n4,n5,n6,n7,n8,n9]);

% adjust vector lengths: assuming all inputs are either length 1 or length
% n -> replicate those of length 1 to be length n
if (length(d11B)==1)
    d11B_in = repmat(d11B,n,nMc);
else
    d11B_in = repmat(d11B,1,nMc);
end
if (length(BCa)==1)
    BCa_in = repmat(BCa,n,nMc);
else
    BCa_in = repmat(BCa,1,nMc);
end
if (length(temperature)==1)
    temperature_in = repmat(temperature,n,nMc);
else
    temperature_in = repmat(temperature,1,nMc);
end
if (length(salinity)==1)
    salinity_in = repmat(salinity,n,nMc);
else
    salinity_in = repmat(salinity,1,nMc);
end
if (length(d11B_std)==1)
    d11B_std_in = repmat(d11B_std,n,nMc);
else
    d11B_std_in = repmat(d11B_std,1,nMc);
end
if (length(BCa_std)==1)
    BCa_std_in = repmat(BCa_std,n,nMc);
else
    BCa_std_in = repmat(BCa_std,1,nMc);
end
if (length(temperature_std)==1)
    temperature_std_in = repmat(temperature_std,n,nMc);
else
    temperature_std_in = repmat(temperature_std,1,nMc);
end
if (length(salinity_std)==1)
    salinity_std_in = repmat(salinity_std,n,nMc);
else
    salinity_std_in = repmat(salinity_std,1,nMc);
end
if (length(depth)==1)
    depth_in = repmat(depth,n,nMc);
else
    depth_in = repmat(depth,1,nMc);
end

% initialize output storage
outList = 1e3*ones(n,nMc);

% add random errors for Monte Carlo
d11B_in = d11B_in+randn(n,nMc).*d11B_std_in;
BCa_in = BCa_in+randn(n,nMc).*BCa_std_in;
temperature_in = temperature_in+randn(n,nMc).*temperature_std_in;
salinity_in = salinity_in+randn(n,nMc).*salinity_std_in;

% carbonate and boron system calculations:
if whichBT==2
    B = (0.1336./10.811.*1000).*salinity_in; % total [B] (umol/kg) from Lee et al (2010)
elseif whichBT==1
    B = (0.12842./10.811.*1000).*salinity_in; % total [B] (umol/kg) from Uppstrom (1974)
end
pKB = -log10(exp((-8966.9-2890.53.*sqrt(salinity_in)-77.942.*salinity_in+1.728.*sqrt(salinity_in).*salinity_in-0.0996.*salinity_in.^2)...
    ./(temperature_in+273.15) + 148.0248 + 137.1942.*sqrt(salinity_in) + 1.62142.*salinity_in + (-24.4344 - 25.085.*sqrt(salinity_in)...
    - 0.2474.*salinity_in).*log((temperature_in +273.15)) + 0.053105.*sqrt(salinity_in).*(temperature_in+273.15))); % Dickson 1990
P_factor = -(-29.48+0.1622.*temperature_in+(-2.608.*10.^-3).*temperature_in.^2)./(83.14472.*(temperature_in+273.15)).*(depth_in./10)+0.5.*((-2.84.*10.^-3)+(0.*10.^-3).*temperature_in+0.*temperature_in.^2)./(83.14472.*(depth_in./10)).*(depth_in./10).^2;
pKB_press = -log10(exp(P_factor+log(10.^(-pKB))));
pH = pKB_press-log10(-(39.61-d11B_in)./(39.61-1.0272.*d11B_in-1000.*(1.0272-1))); % total scale
H = 10.^(-pH); % proton concentration (mol/kg)
BOH4 = (B.*10.^(-pKB))./(10.^(-pKB)+H); % [borate] (umol/kg)
if whichKs == 1 % Lueker (2010) K1 and K2 (given in total pH scale):
    K1 = 10.^(-((3633.86)./(temperature_in+273.15)-(61.2172)+9.6777.*(log(temperature_in+273.15))-0.011555.*salinity_in+0.0001152.*salinity_in.^2));
    K2 = 10.^(-((471.78)./(temperature_in+273.15)+(25.929)-3.16967.*(log(temperature_in+273.15))-0.01781.*salinity_in+0.0001122.*salinity_in.^2));
elseif whichKs == 2 % Mehrbach refit by Dickson and Millero (given in seawater pH scale,
    % converted to total scale):
    TF = (0.000067./18.998).*(salinity_in./1.80655); % in mol/kg-SW Riley, J. P., Deep-Sea Research 12:219-220, 1965
    TS = (0.14./96.062).*(salinity_in./1.80655); % in mol/kg-SW Morris, A. W., and Riley, J. P., Deep-Sea Research 13:699-705, 1966
    IonS = 19.924 .* salinity_in ./ (1000 - 1.005.* salinity_in); % This is from the DOE handbook, Chapter 5, p. 13/22, eq. 7.2.4:
    % Dickson, A. G., J. Chemical Thermodynamics, 22:113-127, 1990:
    KS = exp(-4276.1./(temperature_in+273.15) + 141.328 - 23.093.*(log(temperature_in+273.15)) +...
        (-13856./(temperature_in+273.15) + 324.57 - 47.986.*(log(temperature_in+273.15))).*sqrt(IonS) +...
        (35474./(temperature_in+273.15) - 771.54 + 114.723.*(log(temperature_in+273.15))).*IonS +...
        (-2698./(temperature_in+273.15)).*sqrt(IonS).*IonS + (1776./(temperature_in+273.15)).*IonS.^2) .* (1 - 0.001005 .* salinity_in);
    % Dickson, A. G. and Riley, J. P., Marine Chemistry 7:89-99, 1979:
    KF = exp(1590.2./(temperature_in+273.15) - 12.641 + 1.525.*sqrt(IonS)) .*(1 - 0.001005.*salinity_in) ;
    swsTOtot = (1 + TS./KS)./(1 + TS./KS + TF./KF);
    K1 = 10.^(-(3670.7./(temperature_in+273.15) - 62.008 + 9.7944.*(log(temperature_in+273.15))...
        - 0.0118.*salinity_in + 0.000116.*salinity_in)) .* swsTOtot;
    K2 = 10.^(-(1394.7./(temperature_in+273.15) + 4.777 - 0.0184.*salinity_in + 0.000118.*salinity_in.^2)) .* swsTOtot;
end
% K1 and K2 pressure effects:
dV1 = -25.5 + 0.1271.*temperature_in;
dK1 = (-3.08 + 0.0877.*temperature_in)./1000;
K1_press = exp((-dV1 + 0.5.*dK1.*(depth_in./10)).*(depth_in./10)./(83.14472.*temperature_in)) .* K1;
dV2 = -15.82 - 0.0219.*temperature_in;
dK2 = (1.13 - 0.1475.*temperature_in)./1000;
K2_press = exp((-dV2 + 0.5.*dK2.*(depth_in./10)).*(depth_in./10)./(83.14472.*temperature_in)) .* K2;
% FCO2:
fCO2 =1./(1+H./K2_press+H.*H./K1_press.*K2_press);

options = optimset('MaxFunEvals', 10e10,...
    'MaxIter', 1e3,'TolFun',1e-8,'TolX',0.01,'Display','off');

for i = 1:nMc
    
    res = 100; % initial residual (difference in [CO3] predicted): just to make it > 0.1 for loop below
    while max(res) > 1
        % perform minimization such that the [CO3] used to predict the KD
        % matches the [CO3] predicted by the KD
        if whichKD == 1
            for j = 1:length(outList(:,i))
                [outList(j,i), res] = fminsearch(@resFunc_H16,outList(j,i),options);
            end
        elseif whichKD == 2
            outList = resFunc_M17;
            res = 0;
        elseif whichKD == 3
            for j = 1:length(outList(:,i))
                [outList(j,i), res] = fminsearch(@resFunc_newEq,outList(j,i),options);
            end
        end
    end
    
end

% assign outputs
DIC_list = outList./fCO2;
DIC = mean(DIC_list,2);
DIC_std = std(DIC_list,[],2);
CO3 = mean(outList,2);
CO3_std = std(outList,[],2);

if whichOutput == 1
    varargout{1} = CO3;
    varargout{2} = CO3_std;
elseif whichOutput == 2
    varargout{1} = DIC;
    varargout{2} = DIC_std;
end



    function residual = resFunc_H16(in)
        
        % co3guess is variable that will be adjusted and solved for here
        co3guess = in;
        
        KdBCa = -0.01215.*(B(j,i)/1000) -0.0000119.*(co3guess) + 0.09474; % Holcomb et al 2016
        CO3_out = (BOH4(j,i)./((BCa_in(j,i)./1000)./KdBCa)).^2; % [CO3] (umol/kg)
        
        residual = max(max(sqrt((co3guess-CO3_out).^2))); % minimize this (i.e. [CO3] used for KD is same as CO3 solved from KD)
        
    end

    function output = resFunc_M17
        
        KdBCa = 2.97.*exp(-0.02.*(H.*10.^9))./1000; % McCulloch et al. (2017) refit
        CO3_out = KdBCa./BCa_in.*BOH4.*1e6; % [CO3] (umol/kg)
        
        output = CO3_out;
        
    end

    function residual = resFunc_newEq(in)
        
        % co3guess is variable that will be adjusted and solved for here
        co3guess = in;
        
        KdBCa = log(co3guess).*0.00076771 - 0.002843; % new equation
        CO3_out = KdBCa./BCa_in(j,i).*BOH4(j,i).*1e6; % [CO3] (umol/kg)
        
        residual = max(sqrt((co3guess-CO3_out).^2)); % minimize this (i.e. [CO3] used for KD is same as CO3 solved from KD)
        
    end



if nargout == 3 % do systematic error estimates

    % reset without precision uncertainties
    if (length(d11B)==1)
        d11B_in = repmat(d11B,n,nMc);
    else
        d11B_in = repmat(d11B,1,nMc);
    end
    if (length(BCa)==1)
        BCa_in = repmat(BCa,n,nMc);
    else
        BCa_in = repmat(BCa,1,nMc);
    end
    if (length(temperature)==1)
        temperature_in = repmat(temperature,n,nMc);
    else
        temperature_in = repmat(temperature,1,nMc);
    end
    if (length(salinity)==1)
        salinity_in = repmat(salinity,n,nMc);
    else
        salinity_in = repmat(salinity,1,nMc);
    end
    if (length(depth)==1)
        depth_in = repmat(depth,n,nMc);
    else
        depth_in = repmat(depth,1,nMc);
    end
    
    % add systematic errors for Monte Carlo
    d11B_in = d11B_in+randn(n,nMc).*d11B_sysErr;
    BCa_in = BCa_in+randn(n,nMc).*BCa_sysErr;
    temperature_in = temperature_in+randn(n,nMc).*temperature_sysErr;
    salinity_in = salinity_in+randn(n,nMc).*salinity_sysErr;
    
    % carbonate and boron system calculations:
    B = ( (0.1336+randn(n,nMc).*B_sw_sysErr) ./ 10.811 .*1000).*salinity_in; % total [B] (umol/kg) from Lee et al (2010)
    if whichBT==2
        B = ( (0.1336+randn(n,nMc).*B_sw_sysErr) ./ 10.811 .*1000).*salinity_in; % total [B] (umol/kg) from Lee et al (2010)
    elseif whichBT==1
        B = ( (0.12842+randn(n,nMc).*B_sw_sysErr) ./ 10.811 .*1000).*salinity_in; % total [B] (umol/kg) from Uppstrom (1974)
    end
    pKB = -log10(exp( randn(n,nMc).*lnKB_err + (-8966.9-2890.53.*sqrt(salinity_in)-77.942.*salinity_in+1.728.*sqrt(salinity_in).*salinity_in-0.0996.*salinity_in.^2)...
        ./(temperature_in+273.15) + 148.0248 + 137.1942.*sqrt(salinity_in) + 1.62142.*salinity_in + (-24.4344 - 25.085.*sqrt(salinity_in)...
        - 0.2474.*salinity_in).*log((temperature_in +273.15)) + 0.053105.*sqrt(salinity_in).*(temperature_in+273.15))); % Dickson 1990
    P_factor = -(-29.48+0.1622.*temperature_in+(-2.608.*10.^-3).*temperature_in.^2)./(83.14472.*(temperature_in+273.15)).*(depth_in./10)+0.5.*((-2.84.*10.^-3)+(0.*10.^-3).*temperature_in+0.*temperature_in.^2)./(83.14472.*(depth_in./10)).*(depth_in./10).^2;
    pKB_press = -log10(exp(P_factor+log(10.^(-pKB))));
    pH = pKB_press-log10(-( (39.61+randn(n,nMc).*d11B_sw_sysErr) -d11B_in)./( (39.61+randn(n,nMc).*d11B_sw_sysErr)...
        - (1.0272+randn(n,nMc).*alpha_11B_sysErr) .*d11B_in-1000.*( (1.0272+randn(n,nMc).*alpha_11B_sysErr) -1))); % total scale
    H = 10.^(-pH); % proton concentration (mol/kg)
    BOH4 = (B.*10.^(-pKB))./(10.^(-pKB)+H); % [borate] (umol/kg)
    if whichKs == 1 % Lueker (2010) K1 and K2 (given in total pH scale):
        K1 = 10.^(-((3633.86)./(temperature_in+273.15)-(61.2172)+9.6777.*(log(temperature_in+273.15))-0.011555.*salinity_in+0.0001152.*salinity_in.^2));
        K2 = 10.^(-((471.78)./(temperature_in+273.15)+(25.929)-3.16967.*(log(temperature_in+273.15))-0.01781.*salinity_in+0.0001122.*salinity_in.^2));
    elseif whichKs == 2 % Mehrbach refit by Dickson and Millero (given in seawater pH scale,
        % converted to total scale):
        TF = (0.000067./18.998).*(salinity_in./1.80655); % in mol/kg-SW Riley, J. P., Deep-Sea Research 12:219-220, 1965
        TS = (0.14./96.062).*(salinity_in./1.80655); % in mol/kg-SW Morris, A. W., and Riley, J. P., Deep-Sea Research 13:699-705, 1966
        IonS = 19.924 .* salinity_in ./ (1000 - 1.005.* salinity_in); % This is from the DOE handbook, Chapter 5, p. 13/22, eq. 7.2.4:
        % Dickson, A. G., J. Chemical Thermodynamics, 22:113-127, 1990:
        KS = exp(-4276.1./(temperature_in+273.15) + 141.328 - 23.093.*(log(temperature_in+273.15)) +...
            (-13856./(temperature_in+273.15) + 324.57 - 47.986.*(log(temperature_in+273.15))).*sqrt(IonS) +...
            (35474./(temperature_in+273.15) - 771.54 + 114.723.*(log(temperature_in+273.15))).*IonS +...
            (-2698./(temperature_in+273.15)).*sqrt(IonS).*IonS + (1776./(temperature_in+273.15)).*IonS.^2) .* (1 - 0.001005 .* salinity_in);
        % Dickson, A. G. and Riley, J. P., Marine Chemistry 7:89-99, 1979:
        KF = exp(1590.2./(temperature_in+273.15) - 12.641 + 1.525.*sqrt(IonS)) .*(1 - 0.001005.*salinity_in) ;
        swsTOtot = (1 + TS./KS)./(1 + TS./KS + TF./KF);
        K1 = 10.^(-(3670.7./(temperature_in+273.15) - 62.008 + 9.7944.*(log(temperature_in+273.15))...
            - 0.0118.*salinity_in + 0.000116.*salinity_in)) .* swsTOtot;
        K2 = 10.^(-(1394.7./(temperature_in+273.15) + 4.777 - 0.0184.*salinity_in + 0.000118.*salinity_in.^2)) .* swsTOtot;
    end
    % K1 and K2 pressure effects:
    dV1 = -25.5 + 0.1271.*temperature_in;
    dK1 = (-3.08 + 0.0877.*temperature_in)./1000;
    K1_press = exp((-dV1 + 0.5.*dK1.*(depth_in./10)).*(depth_in./10)./(83.14472.*temperature_in)) .* K1;
    dV2 = -15.82 - 0.0219.*temperature_in;
    dK2 = (1.13 - 0.1475.*temperature_in)./1000;
    K2_press = exp((-dV2 + 0.5.*dK2.*(depth_in./10)).*(depth_in./10)./(83.14472.*temperature_in)) .* K2;
    % FCO2:
    fCO2 =1./(1+H./K2_press+H.*H./K1_press.*K2_press);
    
    if whichKD == 1
        KdBCa = -0.01215.*(B./1000) -0.0000119.*(CO3) + 0.09474; % Holcomb et al 2016
        [val, KdBCa_e] = polyconf([0.09474, -0.0000119],CO3,KD_sysErr,'predopt','curve');
        CO3_sysErr_all = (BOH4./((BCa_in./1000)./ (KdBCa+randn(n,nMc).*(KdBCa_e./1.96)) )).^2; % [CO3] (umol/kg)
    elseif whichKD == 2
        KdBCa = 2.97.*exp(-0.02.*(H.*10.^9))./1000 + randn(n,nMc).*KD_sysErr./1.96; % McCulloch et al. (2017) refit
        CO3_sysErr_all = KdBCa./BCa.*BOH4.*1e6; % [CO3] (umol/kg)
    elseif whichKD == 3
        KdBCa = log(CO3).*0.00076771 - 0.002843; % new equation
        [val, KdBCa_e] = polyconf([-0.002843, 0.00076771],log(CO3),KD_sysErr,'predopt','curve');
        CO3_sysErr_all = (KdBCa+randn(n,nMc).*(KdBCa_e./1.96)) ./BCa_in.*BOH4.*1e6; % [CO3] (umol/kg)
    end
    
    DIC_sysErr = std((CO3_sysErr_all./fCO2),[],2);
    CO3_sysErr = std(CO3_sysErr_all,[],2);
    if whichOutput == 1
        varargout{3} = CO3_sysErr;
    elseif whichOutput == 2
        varargout{3} = DIC_sysErr;
    end
    
end


end

